#!/usr/bin/env python3
import getpass
import json
import logging
import os
import sys
import time
import urllib.error
import urllib.parse
import urllib.request
import random

import requests
from requests.adapters import HTTPAdapter, Retry

if "run_import_hook" in sys.modules:
    from drive.backend.api.objects import Object, Offer, Tag, TelematicsCommand, TelematicsCommandResult, \
        CarDetails, Scanner, CarInfo, \
        PaymentInfo, Roles, User, ModelInfo, SessionsHistory, Area
    from drive.backend.api.session import Session
else:
    from objects import Object, Offer, Tag, TelematicsCommand, TelematicsCommandResult, CarDetails, Scanner, CarInfo, \
        PaymentInfo, Roles, User, ModelInfo, SessionsHistory, Area
    from session import Session


def get_uname_safe():
    try:
        return str(os.uname())
    except Exception as e:
        logging.warning("cannot get OS uname: {}".format(e))
        return "unknown_uname"


def get_login_safe():
    try:
        return str(getpass.getuser())
    except Exception as e:
        logging.warning("cannot get OS login: {}".format(e))
        return "unknown_login"


def generate_reqid():
    return f'{int(time.time())}-{random.randint(100000, 999999)}-{get_login_safe()}'


def get_separator(url):
    return '&' if ('?' in url) else '?'


class BackendClient:
    def __init__(self, endpoint="https://prestable.carsharing.yandex.net", public_token=None, private_token=None, authorization_header=None,
                 timeout=60, lat="55.73394308913616", lon="37.58874029575177", app_build_name="AC",
                 app_build_value=10000, require_public_token=True, login=None):
        assert endpoint
        endpoint_split = endpoint.split('?')
        assert len(endpoint_split) > 0
        self._endpoint = endpoint_split[0]
        if len(endpoint_split) > 1:
            self._extra_cgi = endpoint_split[1]
        else:
            self._extra_cgi = None
        self._device_id = None
        self._timeout = timeout
        self._public_token = public_token
        self._private_token = private_token
        self._authorization_header = authorization_header
        self._lat = lat
        self._lon = lon
        self._app_build_name = app_build_name
        self._app_build_value = app_build_value
        self._login = login
        self._force_fetch_permissions = "false"
        if not self._public_token and require_public_token and not self._authorization_header:
            token_path = os.path.expanduser("~/.drive/yandex")
            logging.debug("extracting public token from {}".format(token_path))
            if not os.path.exists(token_path):
                raise EnvironmentError("public token not found in {}".format(token_path))
            with open(token_path) as token_file:
                self._public_token = token_file.readline().strip(' ').strip('\n')
        if not self._private_token and not self._authorization_header:
            token_path = os.path.expanduser("~/.drive/staff")
            logging.debug("extracting private token from {}".format(token_path))
            if os.path.exists(token_path):
                with open(token_path) as token_file:
                    self._private_token = token_file.readline().strip(' ').strip('\n')

    def _headers(self, path, delegate=None, content_type=None, device_id=None):
        token = None
        if ("api/staff" in path) or ("admin" in path) or ("api/support_staff" in path):
            token = self._private_token
        else:
            token = self._public_token
        assert token or self._authorization_header

        if not content_type:
            content_type = "application/json"
        if not device_id:
            device_id = self._device_id
        if not device_id:
            device_id = "autotest"

        auth_header = self._authorization_header
        if not auth_header:
            auth_header = "OAuth {}".format(token)

        headers = {
            "Authorization": auth_header,
            "OS_Uname": get_uname_safe(),
            "OS_User": get_login_safe(),
            "Content-Type": content_type,
            "DeviceId": device_id,
            "ForceFetchPermissions": self._force_fetch_permissions
        }
        if ("user_app" in path) or ("api/yandex" in path) or ("service_app" in path):
            headers.update({"Lat": self._lat, "Lon": self._lon})
            headers.update({"{}_AppBuild".format(self._app_build_name): str(self._app_build_value)})
        if delegate:
            headers.update({"UserIdDelegation": delegate})
        return headers

    def _request(self, path, json_data=None, method=None, delegate=None, post_data=None, content_type=None,
                 binary=False, response_handler=None):
        assert path
        endpoint = self._endpoint
        url = os.path.join(endpoint, path)
        url += f'{get_separator(url)}reqid={generate_reqid()}'
        if self._extra_cgi:
            url += f'{get_separator(url)}{self._extra_cgi}'
        logging.debug(url)

        data = None
        if json_data:
            data = bytes(json.dumps(json_data), 'utf-8')
        elif post_data:
            data = post_data
        if not method:
            method = "POST" if data is not None else "GET"

        headers = self._headers(path, delegate, content_type)
        retry_strategy = Retry(
            total=3,
            status_forcelist=[401, 402, 419, 500, 502, 503, 504],
            raise_on_status=False
        )
        self._session = requests.Session()
        adapter = HTTPAdapter(max_retries=retry_strategy)
        self._session.mount("https://", adapter)
        self._session.mount("http://", adapter)
        request = requests.Request(
            url=url,
            data=data,
            headers=headers,
            method=method
        )
        prepped = self._session.prepare_request(request)
        try:
            response = self._session.send(request=prepped, timeout=self._timeout)
            response.encoding = 'utf-8'
            if response_handler:
                response_handler(response)
            response.raise_for_status()
            if binary:
                return response.content

            logging.debug("{} {} response: {} {}".format(self._login, path, response.status_code, response.text))
            if response.text:
                return response.json()
            else:
                return None

        except requests.exceptions.RequestException as e:
            if isinstance(e, requests.exceptions.HTTPError):
                logging.error("{} error: {} {}".format(e.response.url, e.response.status_code, e.response.text))
                msg = f'{e.args[0]} \n {e.response.text}'
                raise requests.exceptions.HTTPError(msg, response=e.response) from e
            elif isinstance(e, requests.exceptions.RequestException):
                # logging.error("{} error: {} {}".format(e.response.url, e.response.status_code, e.response.text))
                logging.error(e.response)
                raise e
            else:
                msg = f'{path} {repr(e)}'
                if e.response:
                    msg += f' response{e.response}'
                logging.error(msg)
                raise e

    def set_timeout(self, timeout):
        self._timeout = timeout

    def set_force_fetch_permissions(self, force_fetch_permissions):
        self._force_fetch_permissions = force_fetch_permissions

    def set_location(self, lat, lon):
        self._lat = lat
        self._lon = lon

    def set_app_build(self, app_build_name, app_build_value):
        self._app_build_name = app_build_name
        self._app_build_value = app_build_value

    def set_device_id(self, device_id):
        self._device_id = device_id

    def get_endpoint(self):
        return self._endpoint.rstrip('/')

    def get_extra_cgi(self):
        return self._extra_cgi

    def get_host(self):
        response = self._request("api/staff/user/permissions?dump=eventlog")
        assert response
        return response["__host"]

    def analyzer(self, session_id=None, ride_id=None, tracks_service=None, object_id=None):
        assert session_id or ride_id or object_id
        query = "api/staff/track/analyzer?"
        if ride_id:
            query += "&ride={}".format(ride_id)
        if session_id:
            query += "&session={}".format(session_id)
        if tracks_service:
            query += "&tracks-service={}".format(tracks_service)
        if object_id:
            query += "&car={}".format(object_id)

        response = self._request(query)
        assert response
        return response["tracks"]

    def accept_offer(self, offer):
        assert offer
        self._request("api/yandex/offers/book", json_data=offer.serialize(), method="POST")
        logging.info("accepted offer {}".format(offer.id))

    def raw_create_offers(self, car, user_position=None, destination_name=None, offer_name=None, user_destination=None):
        params = {}
        if car:
            params.update({'car_number': car.number})
        if user_position:
            params.update({'user_position': user_position})
        if destination_name:
            params.update({'destination_name': destination_name})
        if offer_name:
            params.update({'offer_name': offer_name})
        if user_destination:
            params.update({'user_destination': user_destination})
        response = self._request(f'api/yandex/offers/create?{urllib.parse.urlencode(params)}')
        assert response
        return response

    def get_devices_list(self, user_id):
        user_id = user_id or self.current_session().user.id
        response = self._request('api/staff/devices/list?user_id={}'.format(user_id))
        return response

    def remove_devices(self, user_id, devices):
        user_id = user_id or self.current_session().user.id
        assert devices
        self._request('api/staff/devices/remove?user_id={}&devices={}'.format(user_id, devices))

    def create_offers(self, car, user_position=None, destination_name=None, offer_name=None, user_destination=None):
        assert car
        response = self.raw_create_offers(car, user_position, destination_name, offer_name, user_destination)
        assert response
        offers = [Offer.from_json(car) for car in response["offers"]]
        logging.info("created {} offers".format(len(offers)))
        return offers

    def raw_current_session(self, need_bill=True):
        response = self._request("user_app/sessions/current?need_bill={}".format(need_bill))
        assert response
        return response

    def current_session(self, need_bill=True):
        response = self.raw_current_session(need_bill)
        assert response
        return Session.from_json(response)

    def get_actual_sessions(self):
        response = self._request("api/staff/sessions/actual")
        assert response
        return response["sessions"]

    def get_sessions_raw(self, session_id=None):
        params = {}
        if session_id:
            params.update(session_id=session_id)
        params.update(report="user_app")
        response = self._request(f"user_app/sessions/history?{urllib.parse.urlencode(params)}")
        assert response
        return response

    def get_sessions(self):
        response = self.get_sessions_raw()
        objects = SessionsHistory.from_json(response)
        return objects

    def get_session(self, session_id):
        response = self.get_sessions_raw(session_id)
        objects = SessionsHistory.from_json(response)
        return objects

    def drop_future_car(self, tag_id):
        return self._request("api/yandex/futures/drop?tag_id={}".format(tag_id))

    def get_car_info(self, car):
        response = self._request("api/staff/car/info?car_id={}".format(car.id))
        assert response
        return CarInfo.from_json(response)

    def start_session(self):
        return self.evolve_to("old_state_acceptance")

    def evolve_to_riding(self):
        return self.evolve_to("old_state_riding")

    def evolve_to_parking(self):
        return self.evolve_to("old_state_parking")

    def end_session(self, user_choice=None):
        return self.evolve_to("old_state_reservation", user_choice)

    def switch_offer(self, offer_id):
        return self._request("api/yandex/offers/switch?offer_id={}".format(offer_id))

    def evolve_to(self, name, user_choice=None):
        assert name
        tag = Tag(name=name)
        return self.evolve(tag, user_choice)

    def evolve(self, tag=None, user_choice=None):
        assert tag
        params = {}
        if user_choice:
            params.update({'user_choice': user_choice})
        logging.info("tag evolving: {}".format(tag.name))
        response = self._request(f'api/yandex/tag/evolve?{urllib.parse.urlencode(params)}', json_data=tag.serialize(),
                                 method="POST")
        logging.info("tag evolved: {}".format(json.dumps(response) if response else "null"))
        return response

    def search(self, query):
        response = self._request("api/staff/search?has_all_of={}".format(query))
        assert response
        return response

    def search_cars(self, query):
        response = self.search(query)
        objects = [Object.from_json(car) for car in response["objects"]["cars"]]
        return objects

    def search_users(self, query):
        response = self.search(query)
        objects = [User.from_json(user) for user in response["objects"]["users"]]
        return objects

    def search_users_by_tags(self, any_of=None, all_of=None, none_of=None, limit=None):
        cgi = str()
        if all_of:
            cgi += "&has_all_of={}".format(",".join(all_of))
        if any_of:
            cgi += "&has_one_of={}".format(",".join(any_of))
        if none_of:
            cgi += "&has_none_of={}".format(",".join(none_of))
        if limit:
            cgi += "&limit={}".format(limit)
        return self._request("api/staff/tags/search?" + cgi)["results"][0]["report"]

    def edit_car(self, obj, force=False):
        self._request("api/staff/car/edit?force={}".format(force), json_data=obj.serialize(), method="POST")

    def raw_list_cars(self, traits=["ReportLocationDetails", "ReportIMEI", "ReportVIN"], tags_filter=[]):
        query = "api/staff/car/list?"
        if traits:
            for trait in traits:
                query += "&traits={}".format(trait)
        if tags_filter:
            for tag in tags_filter:
                query += "&tags_filter={}".format(tag)
        return self._request(query)

    def list_cars(self, **kwargs):
        response = self.raw_list_cars(**kwargs)
        assert response
        objects = [Object.from_json(car) for car in response.get("cars", [])]
        return objects

    def raw_list_cars_user(self, traits=[]):
        response = self._request("user_app/car/list?traits={}".format(traits))
        return response

    def list_cars_user(self, traits=[]):
        response = self.raw_list_cars_user(traits)
        assert response
        objects = [Object.from_json(car) for car in response.get("cars", [])]
        return objects

    def list_tags(self, obj):
        assert obj.id
        response = self._request("api/staff/car/tag/list?car_id={}".format(obj.id))
        assert response

        tags = [Tag.from_json(t) for t in response["records"]]
        return tags

    def add_tag(self, tag):
        assert tag.object_id
        self._request("api/staff/car/tag/add", json_data=tag.serialize(), method="POST")

    def add_tags(self, obj, tags):
        data = {
            "add_tags": [tag.serialize() for tag in tags],
            "object_id": obj.id,
        }
        self._request("api/staff/car/tag/add", json_data=data, method="POST")

    def get_tag(self, tag_id):
        response = self._request("api/staff/car/tag/details?tag_id={}".format(tag_id))
        assert response
        return Tag.from_json(response["tag"])

    def remove_tag(self, tag_id):
        self._request("api/staff/car/tag/remove?tag_id={}".format(tag_id), method="POST")

    def edit_user(self, user_id=None, username="", first_name="", last_name="", pn="", phone="",
                  is_phone_verified=False, is_mechanic_transmission_allowed=True, email="", status="onboarding", uid=0):

        user_id = user_id or self.current_session().user.id
        assert user_id

        payload = {"id": user_id, "first_name": first_name, "last_name": last_name,
                   "pn": pn, "username": username, "uid": uid, "phone": phone,
                   "is_phone_verified": is_phone_verified,
                   "is_mechanic_transmission_allowed": is_mechanic_transmission_allowed,
                   "email": email, "status": status}
        response = self.upsert_user(payload)
        return User.from_json(response)

    def upsert_user(self, user):
        response = self._request("api/staff/user/edit", json_data=user, method="POST")
        return response["user"] if response else None

    def get_user_info(self, user_id=None):
        user_id = user_id or self.current_session().user.id
        assert user_id
        response = self._request("api/staff/user/info?user_id={}".format(user_id))
        return User.from_json(response)

    def list_users(self, type_):
        response = self._request("api/staff/user/list?type={}".format(type_))
        assert response
        return response["users"]

    def command(self, car, command):
        assert car.id
        data = command.post_data()
        data["car_id"] = car.id
        logging.debug("car:{} command:{} id:{} value:{}".format(data.get("car_id"), data.get("command"), data.get("id"),
                                                                data.get("value")))
        response = self._request("api/staff/car/control", json_data=data, method="POST")
        logging.debug("{}".format(json.dumps(response) if response else "null"))
        return TelematicsCommandResult.from_json(response)

    def command_user(self, action):
        data = {
            "action": action
        }
        return self._request("api/yandex/car/control", json_data=data, method="POST")

    def action_flash(self):
        self.command_user("blinker-flush")

    def action_heating(self):
        self.command_user("start-heating")

    def get_telematics_password(self, car):
        response = self._request("api/staff/car/telematics/password?car_id={}".format(car.id))
        assert response
        return response["password"]

    def get_telematics_state(self, car):
        response = self._request("api/staff/car/telematics/state?car_id={}".format(car.id))
        state = {}
        sensors = response.get("sensors")
        if sensors:
            for i in sensors:
                name = i["name"]
                value = i["value"]
                state[name] = value
        return state

    def get_parameter(self, car, id, subid=None):
        command = TelematicsCommand("GET_PARAM", id, subid)
        result = self.command(car, command)
        result.raise_on_error()
        return result.value

    def set_parameter(self, car, id, value, subid=None):
        command = TelematicsCommand("SET_PARAM", id, subid, value)
        result = self.command(car, command)
        result.raise_on_error()

    def car_details(self, obj, by="id"):
        if by == "id":
            response = self._request("api/staff/car/details?car_id={}".format(obj.id))
        if by == "number":
            response = self._request("api/staff/car/details?car_number={}".format(obj.number))
        assert response
        car_details = CarDetails.from_json(response["cars"][0])
        return car_details

    def drop_session(self, user_id=None, mode="force", session_id=None):
        delegate = user_id or self.current_session().user.id
        params = {}
        params.update({'evolution_mode': mode})
        if session_id:
            params.update({'session_id': session_id})
        self._request(f'api/staff/sessions/drop?{urllib.parse.urlencode(params)}', delegate=delegate)

    def radar_start(self, lat, lon, filter="", action="order", walking_time=900, livetime=1800, delay=0):
        scanner = Scanner(lat=lat, lon=lon, filter=filter, action=action, walking_time=walking_time,
                          livetime=livetime, delay=delay)
        self._request("user_app/scanner/start", json_data=scanner.serialize())

    def radar_stop(self):
        self._request("user_app/scanner/stop")

    def tag_evolve(self, tag_id, tag_json, delegate):
        self._request("api/staff/tag/evolve?tag_id={}".format(tag_id), json_data=tag_json, delegate=delegate,
                      method="POST")

    def get_drop_areas_info_raw(self, use_style=1, offer_id=None, session_id=None):
        if offer_id:
            response = self._request("user_app/areas/drop/info?use_styles={}&offer_id={}".format(use_style, offer_id))
        elif session_id:
            response = self._request(
                "user_app/areas/drop/info?use_styles={}&session_id={}".format(use_style, session_id))
        else:
            response = self._request("user_app/areas/drop/info?use_styles={}".format(use_style))
        return response

    def get_drop_areas_by_area(self, offer_id=None, session_id=None):
        response = self.get_drop_areas_info_raw(offer_id=offer_id, session_id=session_id)
        return [Area.from_json(areas) for areas in response.get('areas')]

    def radar_areas(self, lat, lon, walking_time=900, num_areas="25"):
        data = {
            "walking_time": walking_time,
            "action": "order",
            "num_areas": num_areas,
            "lat": lat,
            "lon": lon
        }
        return self._request("user_app/scanner/areas", json_data=data)

    def _send_command(self, action, user_id=None):
        user_id = user_id or self.current_session().user.id
        data = {
            "reason": "blocked_driving_ban",
            "comment": "test"
        }
        self._request("admin/v1/users/{}/commands/{}/".format(user_id, action),
                      json_data=data)

    def operation_areas_raw(self):
        response = self._request("user_app/areas/operation/info?use_style=1")
        return response

    def block_user(self):
        self._send_command(action="block")

    def unblock_user(self):
        self._send_command(action="unblock")

    def set_payment_ticket(self, sum, user_id=None, type="ticket"):
        data = {
            "user_id": user_id or self.current_session().user.id,
            "type": type,
            "comment": {
                "type": type,
                "comment": "test"
            },
            "sum": sum
        }
        self._request("api/staff/billing/payment/ticket", json_data=data, method='POST')

    def get_payment_info(self, user_id=None, details=0, since=None, until=None):
        from datetime import datetime
        from datetime import timedelta
        since = int(datetime.now().timestamp() - timedelta(days=1).days) if not since else since
        until = int(datetime.now().timestamp()) if not until else until
        user_id = user_id or self.current_session().user.id
        response = self._request(
            "api/staff/billing/payment/info?user_id={}&details={}&since={}&until={}".format(user_id,
                                                                                            details,
                                                                                            since,
                                                                                            until))
        return PaymentInfo.from_json(response)

    def cancel_payment(self, session_id=None):
        data = {
            "session_id": session_id,
            "comment": "Return incorrect ticket"
        }
        self._request("api/staff/billing/payment/cancel", json_data=data)

    def add_user_tag_native(self, tag):
        assert tag.object_id
        self._request("api/staff/user_tags/add", json_data=tag.serialize(), method="POST")

    def add_user_tag(self, tag_name, user_id=None, **kwargs):
        user_id = user_id or self.current_session().user.id
        data = {"tag": tag_name,
                "object_id": user_id
                }
        for key, value in kwargs.items():
            data[key] = value

        self._request("api/staff/user_tags/add", json_data=data)

    def remove_user_tag(self, tag_id):
        self._request(f'api/staff/user_tags/remove?tag_id={tag_id}')

    def get_user_tags(self, user_id=None):
        user_id = user_id or self.current_session().user.id
        response = self._request(f'api/staff/user_tags/list?object_id={user_id}')
        assert response
        return [Tag.from_json(t) for t in response["records"]]

    def get_service_car_info(self, car):
        response = self._request("service_app/car/details?car_id={}".format(car.id))
        assert response
        return CarInfo.from_json(response)

    def add_firmware(self, name, data, alias=None, model=None):
        info = {
            "full_name": name,
            "model": model,
        }
        if alias:
            info["alias"] = alias
        serialized_info = json.dumps(info)
        self._request(
            "api/staff/firmware/add?info={}".format(urllib.parse.quote(serialized_info)),
            post_data=data,
            content_type="audio/fake"  # TODO: use proper Content-Type
        )

    def list_firmware(self):
        response = self._request("api/staff/firmware/list")
        assert response
        return response["firmware"]

    def remove_firmware(self, name, alias=None):
        info = {
            "full_name": name,
        }
        if alias:
            info["alias"] = alias
        self._request("api/staff/firmware/remove", json_data=info)

    def get_permissions(self, user_id=None):
        query = "api/staff/user/permissions"
        if user_id:
            query += "?user_id={}".format(user_id)
        response = self._request(query)
        assert response
        return response

    def get_root(self):
        self._request("api/staff/user/rootify")

    def get_user_roles(self, user_id=None):
        user_id = user_id or self.current_session().user.id
        response = self._request(
            "api/staff/user/roles/list?user_id={}".format(user_id),
        )
        if not response:
            return []
        roles = [Roles.from_json(role) for role in response["report"]]
        return roles

    def operate_user_role(self, role_id, action, user_id=None, activate=True):
        user_id = user_id or self.current_session().user.id
        json_data = {
            "role_id": role_id,
            "user_id": user_id,
        }
        if activate is not None:
            json_data.update(
                {
                    "activate": activate
                }
            )
        return self._request("api/staff/user/roles/{}?user_id={}&role_id={}"
                             .format(action, user_id, role_id), json_data=json_data)

    def activate_role(self, role_id, user_id=None):
        self.operate_user_role(user_id=user_id, role_id=role_id, action="activate")
        logging.info(f'role activated successfully: {role_id}')

    def deactivate_role(self, role_id, user_id=None):
        self.operate_user_role(user_id=user_id, role_id=role_id, action="deactivate")
        logging.info(f'role deactivated successfully: {role_id}')

    def add_user_role(self, role_id, user_id=None, activate=True):
        self.operate_user_role(role_id=role_id, user_id=user_id, action="add", activate=activate)

    def remove_user_role(self, role_id, user_id=None):
        self.operate_user_role(role_id=role_id, user_id=user_id, action="remove")

    def add_action(self, action):
        self._request("api/staff/actions/add", json_data=action)

    def add_role(self, role):
        self._request("api/staff/roles/add", json_data=role)

    def link_action(self, action_id, role_id, role_action_meta={}):
        actions = [
            {
                "action_id": action_id,
                "role_action_meta": role_action_meta,
                "role_id": role_id
            }
        ]
        self.link_to_role(actions)

    def link_to_role(self, actions=[], roles=[]):
        data = {
            "actions": actions,
            "roles": roles,
        }
        self._request("api/staff/roles/actions/add", json_data=data)

    def unlink_from_role(self, role_id, action_ids=[], subrole_ids=[]):
        self._request("api/staff/roles/actions/remove?role={}&actions={}&roles={}".format(
            role_id,
            ','.join(action_ids),
            ','.join(subrole_ids)
        ))

    def list_actions(self, deprecated=True):
        response = self._request("api/staff/actions/list")
        assert response
        actions = response["report"]
        if deprecated:
            actions = actions + response.get("report_deprecated", [])
        result = []
        for action in actions:
            is_proposition_only = action.get("is_proposition_only", False)
            if not is_proposition_only:
                result.append(action)
        return result

    def list_roles(self, report=None):
        return self._request("api/staff/roles/list?report={}".format(report))["report"]

    def list_role_users(self, role_id, limit=0):
        return self._request("api/staff/roles/users?roles={}&limit={}".format(role_id, limit))["report"]

    def propose_role(self, role_snapshot, comment):
        snapshots = [
            role_snapshot
        ]
        return self._request("api/staff/roles/snapshot/propose?comment={}".format(urllib.parse.quote(comment)),
                             json_data=snapshots)

    def remove_action(self, action_id, erase=False):
        return self._request("api/staff/actions/remove?actions={}&full_remove={}".format(
            action_id,
            "true" if erase else "false"
        ))

    def remove_role(self, role_id):
        return self._request("api/staff/roles/remove?roles={}".format(role_id))

    def remove_ranking_model(self, name):
        data = {
            "name": name
        }
        return self._request("api/staff/ranking/model/remove", json_data=data)

    def list_tag_descriptions(self):
        response = self._request("api/staff/tag/description/list")
        assert response
        return response["records"] + response.get("deprecated", [])

    def list_tag_types(self):
        response = self._request("api/staff/tag/description/list")
        assert response
        return response["types"]

    def add_tag_description(self, description, force=False):
        self._request("api/staff/tag/description/add?force={}".format(force), json_data=description)

    def get_model_info(self):
        result = self._request("api/staff/model/info")
        return [ModelInfo.from_json(model) for model in result["car_models"]]

    def add_area(self, area):
        self.add_areas([area])

    def add_areas(self, areas):
        data = {
            "areas": areas
        }
        self._request("api/staff/areas/upsert", json_data=data)

    def add_area_tag(self, tag):
        self._request("api/staff/area_tags/add", json_data=tag)

    def list_areas(self):
        return self._request("api/staff/areas/info")["areas"]

    def remove_area(self, area_id):
        area = {
            "area_id": area_id
        }
        self.remove_areas([area])

    def remove_areas(self, areas):
        data = {
            "areas": areas
        }
        self._request("api/staff/areas/remove", json_data=data)

    def add_ranking_model(self, meta=None, bin=None, proto=None, name=None):
        assert meta or proto
        if proto:
            self._request("api/staff/ranking/model/add?format=proto&name={}".format(name or ""), post_data=proto,
                          content_type="audio/fake")
            return

        serialized_meta = json.dumps(meta)
        if bin:
            self._request("api/staff/ranking/model/add?meta={}".format(urllib.parse.urlencode(serialized_meta)),
                          post_data=bin, content_type="audio/fake")
        else:
            self._request("api/staff/ranking/model/add?meta={}".format(urllib.parse.urlencode(serialized_meta)))

    def get_ranking_model(self, name):
        return self._request("api/staff/ranking/model/get?format=proto&name={}".format(name), binary=True)

    def list_ranking_models(self):
        return self._request("api/staff/ranking/model/list")["models"]

    def list_parkings(self, latitude, longitude):
        response = self._request("api/staff/parking/list?coordinate={}+{}".format(longitude, latitude))
        assert response
        return response["parkings"]

    def list_car_attachments(self, car_id):
        response = self._request("api/staff/car/attachments/list?car_id={}".format(car_id))
        assert response
        return response["attachments"]

    def remove_car_attachment(self, attachment_id):
        self._request("api/staff/car/attachments/unassign?attachment_id={}".format(attachment_id))

    def add_car_model(self, model):
        self._request("api/staff/model/info", json_data=model)

    def list_car_models(self):
        response = self._request("api/staff/model/info")
        assert response
        return response["car_models"]

    def add_landing(self, landing):
        self._request("api/staff/landing/upsert", json_data=landing)

    def list_landings(self):
        response = self._request("api/staff/landing/get")
        assert response
        return response["landings"]

    def add_setting(self, key, value):
        setting = {
            "setting_key": key,
            "setting_value": value,
        }
        self.add_settings([setting])

    def get_photos_markup(self):
        response = self.photos_murkup("get")
        return response

    def update_photos_markup(self):

        response = self.photos_murkup("update")
        return response

    def photos_murkup(self, method, data):
        response = self._request(f"api/staff/photo/markup/{method}", json_data=data)
        return response

    def add_settings(self, settings):
        data = {
            "settings": settings
        }
        self._request("api/staff/settings/upsert", json_data=data)

    def list_settings(self, prefix=None):
        url = "api/staff/settings/info"
        if prefix is not None:
            url += '?prefix={}'.format(prefix)
        response = self._request(url)
        assert response
        return response["settings"]

    def remove_setting(self, key):
        settings = [
            {
                "setting_key": key
            }
        ]
        self.remove_settings(settings)

    def remove_settings(self, settings):
        data = {
            "settings": settings
        }
        self._request("api/staff/settings/remove", json_data=data)

    def add_localization(self, localization):
        self._request("api/staff/localization/upsert", json_data=localization)

    def list_localizations(self):
        response = self._request("api/staff/localization/info")
        assert response
        return response["resources"]

    def add_state_filter(self, state_filter):
        return self.add_state_filters([state_filter])

    def add_state_filters(self, state_filters):
        data = {
            "objects": state_filters
        }
        self._request("api/staff/state_filters/upsert", json_data=data)

    def list_state_filters(self):
        response = self._request("api/staff/state_filters/info")
        assert response
        return response["objects"]

    def add_notifier(self, notifier):
        return self.add_notifiers([notifier])

    def add_notifiers(self, notifiers):
        data = {
            "objects": notifiers
        }
        self._request("api/staff/notifiers/upsert", json_data=data)

    def list_notifiers(self):
        response = self._request("api/staff/notifiers/info")
        assert response
        return response["objects"]

    def add_rtbg(self, process):
        data = {
            "backgrounds": [
                process
            ]
        }
        self._request("api/staff/bg/upsert", json_data=data)

    def list_rtbg(self, ids=[]):
        response = self._request("api/staff/bg/info?ids={}".format(','.join(ids)))
        return response["rt_backgrounds"]

    def add_wallet(self, wallet):
        self._request("api/staff/billing/accounts/update", json_data=wallet)

    def list_wallets(self):
        return self._request("api/staff/billing/accounts/descriptions")["accounts"]

    def get_bills_history(self, cursor=0, limit=100, delegate=None):
        response = self._request("api/staff/bills/history?cursor={}&limit={}".format(cursor, limit), delegate=delegate)
        assert response
        return response["cursor"], response["sessions"]

    def get_chat_history_unread_raw(self):
        response = self._request("support_api/chat/history/unread")
        return response

    def get_chats_list_raw(self):
        response = self._request("support_api/chats/list")
        return response

    def get_offers_fixpoint(self, data):
        """
        data = {
            "work": {
                "name": "Работа",
                "coordinate": [
                    37.64084895021607,
                    55.73678588807751
                ]
            },
            "home": {
                "name": "Дом",
                "coordinate": [
                    37.398815000000006,
                    55.876096999999994
                ]
            }
        }
        """
        response = self._request("api/yandex/offers/fixpoint", json_data=data, method="POST")
        return response

    def start_servicing(self, car_id, tag_id):
        json_data = {
            "car_id": car_id,
            "tag_id": tag_id
        }
        self._request("service_app/servicing/start", json_data=json_data)
        logging.info(f"tag {tag_id} started for car {car_id}")

    def finish_servicing(self, car_id, tag_id, drop=False):
        json_data = {
            "car_id": car_id
        }
        if drop:
            json_data.update({"drop_tag_ids": tag_id})
        else:
            json_data.update({"tag_id": tag_id})
        self._request("service_app/servicing/finish", json_data=json_data)
        logging.info(f"tag {tag_id} finished for car {car_id}")

    def get_tags_by_performer(self, performer_id=None):
        performer_id = performer_id or self.current_session().user.id
        response = self._request(f'api/staff/tags/by_performer?performer={performer_id}')
        return [Tag.from_json(t) for t in response["records"]]

    def register_user_force(self, user_id=None):
        user_id = user_id or self.current_session().user.id
        self._request(f'api/support_staff/user/register-force?user_id={user_id}')


def client_main():
    client = BackendClient()
    for car in client.list_cars():
        print(car.id)


if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO)
    client_main()
