import logging


class Location:
    def __init__(self, lat, lon, tags=None):
        self.lat = lat
        self.lon = lon
        self.tags = tags

    @classmethod
    def from_json(cls, j):
        lat = j["lat"]
        lon = j["lon"]
        tags = j.get("tags")
        return cls(lat, lon, tags)

    def serialize(self):
        r = {
            "lat": self.lat,
            "lon": self.lon,
        }
        if self.tags:
            r["tags"] = self.tags
        return r


class Object:
    def __init__(self, id=None, model=None, number=None, status=None, location=None, imei=None, vin=None, patches=None,
                 futures_location=None, fuel_level=None):
        self.id = id
        self.imei = imei
        self.model = model
        self.number = number
        self.status = status
        self.location = location
        self.vin = vin
        self.patches = patches
        self.futures_location = futures_location
        self.fuel_level = fuel_level
        self.osago_mds_key = None
        self.registration_mds_key = None

    @classmethod
    def from_json(cls, j):
        logging.debug("deserializing {} into Object".format(j))
        id = j.get("id")
        imei = j.get("imei")
        model = j.get("model_id")
        number = j.get("number")
        status = j.get("status")
        location_ = j.get("location")
        vin = j.get("vin")
        patches = j.get("patches")
        futures_location = j.get("futures_location")
        fuel_level = j.get("telematics", {}).get("fuel_level")
        location = Location.from_json(location_) if location_ else None
        result = cls(id, model, number, status, location, imei, vin, patches, futures_location, fuel_level)
        result.sf = j.get("sf")
        result.filters = j.get("filters")
        print(f'number: {number}, location:{location_}, future location: {futures_location}')
        return result

    def serialize(self):
        assert self.id or self.number
        logging.debug("serializing Object {}".format(self.id))
        r = {}
        r["model_code"] = self.model
        r["model_id"] = self.model
        if self.id:
            r["id"] = self.id
        if self.imei:
            r["imei"] = self.imei
        if self.location:
            r["location"] = self.location.serialize()
        if self.number:
            r["number"] = self.number
        if self.status:
            r["status"] = self.status
        if self.vin:
            r["vin"] = self.vin
        if self.fuel_level:
            r["telematics"] = {
                "fuel_level": self.fuel_level,
            }
        if self.osago_mds_key:
            r["osago_mds_key"] = self.osago_mds_key
        if self.registration_mds_key:
            r["registration_mds_key"] = self.registration_mds_key
        return r


class Error:
    def __init__(self, http_code=None, ui_message=None, special_info=None, debug_message=None):
        self.http_code = http_code
        self.ui_message = ui_message
        self.special_info = special_info
        self.debug_message = debug_message

    @classmethod
    def from_json(cls, j):
        http_code = j.get("http_code")
        ui_message = j.get("ui_message")
        special_info = j.get("special_info")
        debug_message = j.get("debug_message")
        result = cls(http_code, ui_message, special_info, debug_message)
        result.landings = [Landing.from_json(landing) for landing in special_info["landings"]]
        return result


class Landing:
    def __init__(self, landing_id=None):
        self.button_style = None
        self.points = None
        self.other_buttons = None
        self.title = None
        self.landing_type = None
        self.button_color = None
        self.color1 = None
        self.landing_id = landing_id
        self.transparence1 = None
        self.button = None
        self.color2 = None
        self.button_type = None
        self.transparence2 = None

    @classmethod
    def from_json(cls, j):
        landing_id = j.get("id")
        result = cls(landing_id)
        result.landing_type = j.get("type")
        return result


class PropertyPatch:
    def __init__(self, name=None, index=None, comment=None, patches=None, display_name=None):
        self.name = name
        self.index = index
        self.comment = comment
        self.patches = patches
        self.display_name = display_name

    @classmethod
    def from_json(cls, j):
        name = j.get("name")
        index = j.get("index")
        comment = j.get("comment")
        patches = j.get("patches")
        display_name = j.get("display_name")
        result = cls(name, index, comment, patches, display_name)
        return result


class Roles:
    def __init__(self, role_id):
        self.role_id = role_id
        self.deadline = None
        self.active = None
        self.user_id = None

    @classmethod
    def from_json(cls, j):
        role_id = j.get("role_id")

        result = cls(role_id)
        result.deadline = j.get("deadline")
        result.active = j.get("active")
        result.user_id = j.get("user_id")
        return result


class Offer:
    def __init__(self, id=None):
        self.id = id
        self.constructor_id = None
        self.car_info = None
        self.name = None
        self.group_name = None
        self.from_scanner = None
        self.short_description = None
        self.type = None
        self.detailed_description = None
        self.short_name = None
        self.pack_price = None
        self.riding_price = None
        self.parking_price = None
        self.discounts = None
        self.riding_discounted = None
        self.constructor_id = None
        self.finish_area_border = None
        self.finish = None
        self.finish_area = None
        self.futures_location = None
        self.switchable = None

    @classmethod
    def from_json(cls, j):
        id = j.get("offer_id")

        result = cls(id)
        result.constructor_id = j.get("constructor_id")
        result.name = j.get("name")
        result.group_name = j.get("group_name")
        result.from_scanner = j.get("from_scanner")
        result.short_description = j.get("short_description")
        car_info = j.get("car_info")
        if car_info:
            result.car_info = Object.from_json(car_info)
        result.type = j.get("type")
        result.short_name = j.get("short_name")
        result.pack_price = j.get("pack_price")
        result.constructor_id = j.get("constructor_id")
        result.detailed_description = j.get("detailed_description")
        result.riding_price = j["prices"]["riding"]
        result.parking_price = j["prices"]["parking"]
        result.riding_discounted = j["prices"]["riding_discounted"]
        result.switchable = j.get("switchable")
        result.finish_area_border = j.get("finish_area_border")
        result.finish = j.get("finish")
        result.finish_area = j.get("finish_area")
        discounts = j.get("discounts")
        result.futures_location = j.get("futures_location")
        result.car_waiting_duration_hr = j.get("car_waiting_duration_hr")
        result.car_waiting_duration = j.get("car_waiting_duration")
        result.discounts = []
        if discounts:
            result.discounts = [Discount.from_json(discount) for discount in discounts]
        return result

    def serialize(self):
        assert self.id
        r = {}
        r["offer_id"] = self.id
        return r


class OfferState:
    def __init__(self):
        self.pack_price_round = None
        self.remaining_time = None
        self.remaining_distance = None
        self.exceeded = None
        self.overrun_distance = None
        self.recalculated_on_finish = None
        self.pack_price = None
        self.duration_threshold_push_sent = None
        self.distance_threshold_push_sent = None
        self.overrun_time = None
        self.waiting_price = None

    @classmethod
    def from_json(cls, j):
        result = cls()

        result.pack_price_round = j.get("pack_price_round")
        result.remaining_time = j.get("remaining_time")
        result.remaining_distance = j.get("remaining_distance")
        result.exceeded = j.get("exceeded")
        result.overrun_distance = j.get("overrun_distance")
        result.pack_price = j.get("pack_price")
        result.duration_threshold_push_sent = j.get("duration_threshold_push_sent")
        result.distance_threshold_push_sent = j.get("distance_threshold_push_sent")
        result.overrun_time = j.get("overrun_time")
        result.waiting_price = j.get("waiting_price")

        return result


class Discount:
    def __init__(self, id=None):
        self.id = id
        self.discount = None
        self.title = None

    @classmethod
    def from_json(cls, j):
        id = j.get("id")

        result = cls(id)
        result.discount = j.get("discount")
        result.title = j.get("title")
        return result


class ModelInfo:
    def __init__(self, code):
        self.code = code

    @classmethod
    def from_json(cls, j):
        code = j.get("code")
        result = cls(code)
        return result


class Tag:
    def __init__(self, name=None, object_id=None, tag_id=None, performer=None, priority=None, data=None):
        self.name = name
        self.object_id = object_id
        self.tag_id = tag_id
        self.comment = None
        self.performer = performer
        self.priority = priority
        self.data = data

    @classmethod
    def from_json(cls, j):
        name = j.get("tag_name") or j.get("tag")
        object_id = j.get("object_id")
        tag_id = j.get("tag_id")
        performer = j.get("performer")
        priority = j.get("priority")
        data = j
        return cls(name, object_id, tag_id, performer, priority, data)

    def serialize(self):
        r = self.data if self.data else {}
        assert self.name
        r["tag_name"] = self.name
        r["object_id"] = self.object_id
        if self.tag_id:
            r["tag_id"] = self.tag_id
        if self.comment:
            r["comment"] = self.comment
        return r


class TelematicsCommand:
    def __init__(self, name=None, id=None, subid=None, value=None, raw=None):
        self.name = name
        self.id = id
        self.subid = subid
        self.value = value
        self.raw = raw or {}

    @classmethod
    def from_json(cls, j):
        if isinstance(j, str):
            return cls(j)
        else:
            return cls(raw=j)

    def cgi_string(self):
        assert self.name
        result = "command={}".format(self.name)
        if self.id:
            result += "&id={}".format(self.id)
        if self.subid:
            result += "&subid={}".format(self.id)
        if self.value:
            result += "&value={}".format(self.value)
        return result

    def post_data(self):
        result = self.raw
        if self.name:
            result["command"] = self.name
        if self.id:
            result["id"] = self.id
        if self.subid:
            result["subid"] = self.subid
        if self.value is not None:
            result["value"] = self.value
        return result


class CarInfo:
    def __init__(self, id, location=None, location_tags=None):
        self.id = id
        self.current_offer = None
        self.location_tags = location_tags
        self.location = location

    @classmethod
    def from_json(cls, j):
        id = j["id"]
        location = j.get("location")
        result = cls(id, location)
        if location:
            result.location_tags = location.get("tags")
        session_info = j.get("session_info")
        if session_info:
            current_offer = session_info["session"]["specials"].get("current_offer")
            if current_offer:
                result.current_offer = Offer.from_json(current_offer)
        return result


class TelematicsCommandResult:
    def __init__(self, status, message=None, value=None):
        self.status = status
        self.message = message
        self.value = value

    def get_error(self):
        assert not self.succeeded()
        if self.message and len(self.message) > 0:
            return self.message
        else:
            return self.status

    def raise_on_error(self):
        if not self.succeeded():
            raise RuntimeError("{}: {}".format(self.status, self.message))

    def succeeded(self):
        return self.status.lower() == "success"

    @classmethod
    def from_json(cls, j):
        assert j
        status = j["telematic_status"]
        message = j.get("telematic_message")
        value = j.get("value")
        return cls(status, message, value)


class User:
    def __init__(self, id):
        self.id = id
        self.scanner = None
        self.debt_amount = 0
        self.username = None
        self.phone_number = None
        self.is_phone_verified = None
        self.email = None
        self.billing = None
        self.debt = None
        self.details = None
        self.status = None

    @classmethod
    def from_json(cls, j):
        id = j.get("user_id") or j.get("id")
        result = cls(id)

        result.first_name = j.get("first_name")
        result.last_name = j.get("last_name")
        result.pn = j.get("pn")
        result.username = j.get("username")

        result.is_mechanic_transmission_allowed = j.get("is_mechanic_transmission_allowed")

        result.status = j.get("status")
        result.uid = j.get("uid")

        setup = j.get("setup")
        if setup:
            result.phone_number = setup.get("phone")["number"]
            result.is_phone_verified = setup.get("phone")["verified"]
            result.email = setup.get("email")["address"]

        scanner = j.get("scanner")
        if scanner:
            result.scanner = Scanner.from_json(scanner)
        billing = j.get("billing")
        if billing:
            result.debt = billing.get("debt")
            bonuses = billing.get("bonuses")
            result.bonuses_amount = bonuses.get("amount") if bonuses else 0
        details = j.get("details")
        if details:
            result.status = details["status"]
        if result.debt:
            result.debt_amount = billing["debt"]["amount"]
        return result


class Area:
    def __init__(self, area_id=None):
        self.area_id = area_id

    @classmethod
    def from_json(cls, j):
        result = cls()
        result.title = j.get('title')
        result.fee = j.get('fee')
        result.area_coords = j.get("area_coords")
        result.center = j.get('center')

        return result


class UserCard:
    def __init__(self, id):
        self.id = id

    @classmethod
    def from_json(cls, j):
        assert j
        id = j.get("id")
        result = cls(id)

        result.first_name = j.get("first_name")
        result.last_name = j.get("last_name")
        result.pn = j.get("pn")
        result.username = j.get("username")
        result.phone = j.get("phone")
        result.is_phone_verified = j.get("is_phone_verified")
        result.is_mechanic_transmission_allowed = j.get("is_mechanic_transmission_allowed")
        result.email = j.get("email")
        result.status = j.get("status")
        result.uid = j.get("uid")


class Scanner:

    def __init__(self, lat, lon, filter=None, action=None, walking_time=None, livetime=None, delay=None):
        self.lat = lat
        self.lon = lon
        self.filter = filter
        self.action = action
        self.walking_time = walking_time
        self.livetime = livetime
        self.delay = delay

    @classmethod
    def from_json(cls, j):
        position = j.get("position")
        lat = float(position.split(" ")[1])
        lon = float(position.split(" ")[0])
        result = cls(lat, lon)
        result.start = j.get("start")
        result.deadline = j.get("deadline")
        result.filter = j.get("filter")
        result.is_finished = j.get("is_finished")
        result.action = j.get("action")
        result.walk_duration = j.get("walk_duration")

        return result

    def serialize(self):
        assert self.lat and self.lon
        logging.debug("serializing Scanner lat:{} lon:{} filter:{} action:{} walking time:{} livetime:{} delay:{}"
                      .format(self.lat, self.lon, self.filter, self.action, self.walking_time, self.livetime,
                              self.delay)
                      )
        r = {"lat": self.lat, "lon": self.lon, "filter": self.filter, "action": self.action,
             "walking_time": self.walking_time, "livetime": self.livetime, "delay": self.delay}
        return r


class CarDetails:
    def __init__(self, tags=None, car_id=None):
        self.car_id = car_id
        self.tags = tags

    @classmethod
    def from_json(cls, j):
        car_id = j.get("id")
        tags = j["location"]["tags"]
        result = cls(tags, car_id)
        result.is_allow_drop = "allow_drop_car" in tags
        result.is_allow_riding = "allow_riding" in tags
        result.cargo_allow_drop_car = "cargo_allow_drop_car" in tags
        result.cargo_deny_drop_car = "cargo_deny_drop_car" in tags
        return result


class SessionDetail:
    def __init__(self, session_id=None):
        self.session_id = session_id
        self.start = None
        self.finish = None
        self.total_price = None
        self.bill = None
        self.bill_lines = None

    @classmethod
    def from_json(cls, j):
        if "segment" in j:
            session_id = j["segment"]["session_id"]
        result = cls(session_id)
        result.start = j["segment"]["start"]
        result.finish = j["segment"]["finish"]
        result.total_price = j["segment"]["total_price"]

        result.car = Object.from_json(j["car"])
        bill = j["segment"]["bill"]
        assert bill
        result.bill = bill
        result.bill_lines = [Bill.from_json(bill_line) for bill_line in bill]
        return result


class SessionsHistory:
    def __init__(self):
        self.sessions = None
        self.has_more = None
        self.views = None
        self.sf = None
        self.property_patches = None
        self.models = None
        self.bill = None
        self.feedback = None

    @classmethod
    def from_json(cls, j):
        result = cls()
        result.sessions = [SessionDetail.from_json(session) for session in j["sessions"]]
        result.has_more = j.get("has_more")
        multi_bill = j.get("multi_bill")
        if multi_bill:
            bills = multi_bill["bills"]
        bill = []
        for b in bills:
            bill_lines = [Bill.from_json(bill_line) for bill_line in b]
            bill.append(bill_lines)
        result.bill = bill
        return result


class Bill:
    def __init__(self, line_type=None):
        self.line_type = line_type
        self.cost = None
        self.duration = None
        self.id = None

    @classmethod
    def from_json(cls, j):
        line_type = j["type"]
        result = cls(line_type)
        result.id = j.get("id")
        result.cost = j.get("cost")
        result.duration = j.get("duration")

        return result


class PaymentInfo:
    def __init__(self, history=None, current=None):
        self.history = history
        self.current = current

    @classmethod
    def from_json(cls, j):
        history = j.get("history")

        current = j.get("current")
        result = cls(history, current)
        result.compiled_history = j.get('compiled_history')
        if current:
            result.current = [CurrentPayment.from_json(i) for i in current]
        return result


class CurrentPayment:
    def __init__(self, session_id=None):
        self.session_id = session_id
        self.state = None

    @classmethod
    def from_json(cls, j):
        session_id = j.get("session_id")

        result = cls(session_id)
        result.state = j.get("state")
        result.bill = j.get("bill")
        result.billing_type = j.get("billing_type")
        result.task_status = j.get('task_status')
        return result
