#include "areas.h"

#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/model.h>
#include <drive/backend/data/operation_area.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/tags/tags_manager_impl.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/iterator.h>

#include <util/string/join.h>
#include <util/string/split.h>

NJson::TJsonValue TTooltip::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("url", Url);
    result.InsertValue("message", Message);
    result.InsertValue("beacon_message", BeaconMessage);
    result.InsertValue("title", Title);
    if (Icon) {
        result.InsertValue("icon", Icon);
    }
    if (Image) {
        result.InsertValue("image", Image);
    }
    if (SmallImage) {
        result.InsertValue("small_image", SmallImage);
    }
    if (TopColor) {
        result.InsertValue("top_color", TopColor);
    }
    if (BottomColor) {
        result.InsertValue("bottom_color", BottomColor);
    }
    if (TextColor) {
        result.InsertValue("text_color", TextColor);
    }
    return result;
}

NJson::TJsonValue TTooltip::BuildReport(const TMaybe<TString>& beaconParkingPlaceNumber, ELocalization locale, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    NJson::TJsonValue result;
    result.InsertValue("url", Url);
    result.InsertValue("message", localization ? localization->GetLocalString(locale, Message) : Message);
    if (beaconParkingPlaceNumber && BeaconMessage) {
        auto beaconMessage = localization ? localization->GetLocalString(locale, BeaconMessage) : BeaconMessage;
        SubstGlobal(beaconMessage, "_BEACON_PARKING_PLACE_", *beaconParkingPlaceNumber);
        result.InsertValue("title", beaconMessage);
    } else {
        result.InsertValue("title", Title);
    }
    if (Icon) {
        result.InsertValue("icon", Icon);
    }
    if (Image) {
        result.InsertValue("image", Image);
    }
    if (SmallImage) {
        result.InsertValue("small_image", SmallImage);
    }
    if (TopColor) {
        result.InsertValue("top_color", TopColor);
    }
    if (BottomColor) {
        result.InsertValue("bottom_color", BottomColor);
    }
    if (TextColor) {
        result.InsertValue("text_color", TextColor);
    }
    return result;
}

bool TTooltip::DeserializeFromJson(const TString& infoStr) {
    NJson::TJsonValue jsonInfo;
    if (NJson::ReadJsonFastTree(infoStr, &jsonInfo)) {
        return DeserializeFromJson(jsonInfo);
    } else {
        return false;
    }
}

bool TTooltip::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_STRING(info, "message", Message);
    JREAD_STRING(info, "title", Title);
    JREAD_STRING_OPT(info, "icon", Icon);
    JREAD_STRING_OPT(info, "image", Image);
    JREAD_STRING_OPT(info, "url", Url);
    JREAD_STRING_OPT(info, "small_image", SmallImage);
    JREAD_STRING_OPT(info, "top_color", TopColor);
    JREAD_STRING_OPT(info, "bottom_color", BottomColor);
    return NJson::ParseField(info["beacon_message"], BeaconMessage)
        && NJson::ParseField(info["text_color"], TextColor);
}

TArea::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Identifier = GetFieldDecodeIndex("area_id", decoderBase);
    Title = GetFieldDecodeIndex("area_title", decoderBase);
    Index = GetFieldDecodeIndex("area_index", decoderBase);
    Type = GetFieldDecodeIndex("area_type", decoderBase);
    Tooltip = GetFieldDecodeIndex("area_tooltip", decoderBase);
    Tags = GetFieldDecodeIndex("area_tags", decoderBase);
    Coords = GetFieldDecodeIndex("area_coords", decoderBase);
    Revision = GetFieldDecodeIndex("revision", decoderBase);
    Details = GetFieldDecodeIndex("area_details", decoderBase);
}

bool TArea::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Identifier);
    READ_DECODER_VALUE(decoder, values, Title);
    READ_DECODER_VALUE_DEF(decoder, values, Revision, 0);
    READ_DECODER_VALUE_DEF(decoder, values, Index, 0);
    READ_DECODER_VALUE_DEF(decoder, values, Type, Type);
    if (!Identifier && Type != "zone") {
        return false;
    }

    NJson::TJsonValue areaDetails;
    READ_DECODER_VALUE_JSON(decoder, values, areaDetails, Details);
    if (!DeserializeDetailsFromJson(areaDetails)) {
        return false;
    }

    {
        TString coordsStr;
        READ_DECODER_VALUE_TEMP(decoder, values, coordsStr, Coords);

        NJson::TJsonValue discarded;
        if (
            !TGeoCoord::DeserializeVector(coordsStr, Coords) &&
            (!NJson::ReadJsonFastTree(coordsStr, &discarded) || !NJson::TryFromJson(discarded, Coords)))
        {
            return false;
        }
    }

    {
        TString tooltipStr;
        READ_DECODER_VALUE_TEMP(decoder, values, tooltipStr, Tooltip);
        if (!!tooltipStr) {
            Tooltip = new TTooltip();
            if (!Tooltip->DeserializeFromJson(tooltipStr)) {
                return false;
            }
        }
    }
    TVector<TGeoCoord> coordsLocal = Coords;
    if (!coordsLocal.empty() && coordsLocal.front().GetLengthTo(coordsLocal.back()) > 1e-5) {
        coordsLocal.emplace_back(coordsLocal.front());
    }
    Polyline = TPolyLine<TGeoCoord>(std::move(coordsLocal));

    TString tagsStr;
    READ_DECODER_VALUE_TEMP(decoder, values, tagsStr, Tags);
    Tags = MakeSet(StringSplitter(tagsStr).SplitBySet(", ").SkipEmpty().ToList<TString>());
    return true;
}

bool TArea::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBaseDecoder::DeserializeFromJson(*this, jsonInfo);
}

NJson::TJsonValue TArea::SerializeToReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    JWRITE(result, "area_coords", TGeoCoord::SerializeVectorToJsonIFace(Coords));
    if (!!Tooltip) {
        JWRITE(result, "area_tooltip", Tooltip->SerializeToJson());
    }
    JWRITE(result, "area_title", Title);
    JWRITE(result, "area_type", Type);
    JWRITE(result, "area_index", Index);
    result.InsertValue("area_details", SerializeDetailsToJson());
    result.InsertValue("area_id", Identifier);
    result.InsertValue("area_tags", JoinSeq(", ", Tags));
    result.InsertValue("revision", Revision);
    return result;
}

NJson::TJsonValue TArea::SerializeToPublicReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    JWRITE(result, "coords", TGeoCoord::SerializeVectorToJsonIFace(Coords));
    JWRITE(result, "center", TGeoCoord::CalcCenter(Coords).SerializeToJson());
    if (!!Tooltip) {
        JWRITE(result, "tooltip", Tooltip->SerializeToJson());
    }
    JWRITE(result, "title", Title);
    JWRITE(result, "type", Type);
    JWRITE(result, "index", Index);
    if (!!HintPosition) {
        JWRITE(result, "hint_position", HintPosition->ToString());
    }

    return result;
}

bool TArea::Parse(const NStorage::TTableRecord& record) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, record);
}

NDrive::TScheme TArea::GetScheme(const IServerBase& server) {
    NDrive::TScheme result;
    result.Add<TFSString>("area_id", "Идентификатор полигона");
    result.Add<TFSString>("area_coords", "Координаты полигона (lon1 lat1 lon2 lat2 ...)");
    result.Add<TFSString>("area_tags", "Теги");
    result.Add<TFSString>("area_title", "Название");
    result.Add<TFSString>("area_index", "Index").SetReadOnly(true);
    result.Add<TFSString>("area_type", "Тип");

    NDrive::TScheme& detailsScheme = result.Add<TFSStructure>("area_details", "Детали").SetStructure<NDrive::TScheme>();
    TAttributedEntity::InsertScheme(server, "areas", detailsScheme);
    detailsScheme.Add<TFSString>("hint_position", "положение hint (lon lat)");

    result.Add<TFSString>("revision").SetReadOnly(true);
    result.Add<TFSString>("area_tooltip");
    return result;
};

NStorage::TTableRecord TArea::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("area_id", Identifier)
        .Set("area_coords", TGeoCoord::SerializeVector(Coords))
        .Set("area_tags", JoinSeq(", ", Tags))
        .Set("area_title", Title)
        .Set("area_index", Index)
        .Set("area_type", Type)
        .Set("area_details", SerializeDetailsToJson().GetStringRobust())
        .Set("revision", Revision);
    if (!!Tooltip) {
        result.Set("area_tooltip", Tooltip->SerializeToJson());
    }
    return result;
}

NStorage::TTableRecord TArea::SerializeUniqueToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("area_id", Identifier);
    return result;
}

TConstAreasSnapshot::TPtr TAreasSnapshot::BuildConstSnapshot() const {
    TReadGuard g(Mutex);
    return new TConstAreasSnapshot(Areas);
}

bool TAreasSnapshot::ForAllAreas(const TFunction& action) const {
    TReadGuard rg(Mutex);
    for (auto&& i : Areas) {
        if (!action(i)) {
            return false;
        }
    }
    return true;
}

bool TAreasSnapshot::UpsertArea(const TArea& area) {
    TWriteGuard g(Mutex);
    bool found = false;
    for (auto& i : Areas) {
        if (i.GetId() == area.GetIdentifier()) {
            TFullAreaInfo newAreaInfo(area);
            newAreaInfo.MutableTags() = i.MutableTags();
            i = newAreaInfo;
            found = true;
            break;
        }
    }
    if (!found) {
        Areas.emplace_back(area);
    }
    std::sort(Areas.begin(), Areas.end());
    return true;
}

bool TAreasSnapshot::RemoveArea(const TArea& area) {
    TWriteGuard g(Mutex);
    const auto pred = [&area](const TFullAreaInfo& info) -> bool {
        return info.GetId() == area.GetIdentifier();
    };
    Areas.erase(std::remove_if(Areas.begin(), Areas.end(), pred), Areas.end());
    return true;
}

bool TAreasSnapshot::UpsertTag(const TConstDBTag& tag, const ITagsHistoryContext& context) {
    for (auto& i : Areas) {
        if (i.GetId() == tag.GetObjectId()) {
            i.RefreshTag(tag, context);
            return true;
        }
    }
    return false;
}

bool TAreasSnapshot::RemoveTag(const TConstDBTag& tag) {
    for (auto& i : Areas) {
        if (i.GetId() == tag.GetObjectId()) {
            i.RemoveTag(tag);
            return true;
        }
    }
    return false;
}

TVector<TFullAreaInfo> TAreasSnapshot::ExtractAreas() && {
    return std::move(Areas);
}

void TAreasSnapshot::SetAreas(TAreasSnapshot&& snapshot) {
    TWriteGuard g(Mutex);
    Areas = std::move(snapshot).ExtractAreas();
}


void TAreaTagsManager::AcceptHistoryEventUnsafe(const TObjectEvent<TConstDBTag>& ev) const {
    TBase::AcceptHistoryEventUnsafe(ev);
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        if (!Snapshot.RemoveTag(ev)) {
            Y_ASSERT(false);
        }
    } else {
        if (!Snapshot.UpsertTag(ev, Context)) {
            Y_ASSERT(false);
        }
    }
}

bool TAreaTagsManager::DoRebuildCacheUnsafe() const {
    if (!TBase::DoRebuildCacheUnsafe()) {
        return false;
    }
    for (auto&& i : TBase::Objects) {
        for (auto&& tag : i.second.GetTags()) {
            if (!Snapshot.UpsertTag(tag, Context)) {
                Y_ASSERT(false);
            }
        }
    }
    return true;
}

bool TAreasDB::DoRebuildCacheUnsafe() const {
    if (!TBase::DoRebuildCacheUnsafe()) {
        return false;
    }
    {
        TAreasSnapshot snapshot;
        for (auto&& i : TBase::Objects) {
            snapshot.UpsertArea(i.second);
        }
        Snapshot.SetAreas(std::move(snapshot));
    }
    RebuildCurrentSnapshot();
    return true;
}

bool TAreasDB::DoStart() {
    if (!TBase::DoStart()) {
        return false;
    }
    if (!TagsManager.Start()) {
        return false;
    }
    RebuildCurrentSnapshot();
    return true;
}

bool TAreasDB::DoStop() {
    return
        TagsManager.Stop() &&
        TBase::DoStop();
}

bool TAreasDB::DoRefresh(const TInstant reqActuality) const {
    return
        TBase::RefreshCache(reqActuality) &&
        TagsManager.RefreshCache(reqActuality);
}

void TAreasDB::RebuildCurrentSnapshot() const {
    TWriteGuard wg(ConstAreasSnapshotMutex);
    ConstSnapshot = Snapshot.BuildConstSnapshot();
}

TAreasDB::EAreaStyle TAreasDB::GetStyle(EDropAbility dropAbility, bool allowRiding, bool allowDelivery, bool withFees) {
    switch (dropAbility) {
        case EDropAbility::Allow:
            if (withFees) {
                return EAreaStyle::AllowDropFees;
            } else {
                return EAreaStyle::AllowDropNoFees;
            }
        case EDropAbility::Deny:
        case EDropAbility::DenyIncorrectData:
            return EAreaStyle::DenyDrop;
        case EDropAbility::NotAllow:
            if (allowDelivery) {
                return EAreaStyle::AllowDelivery;
            }
            if (allowRiding) {
                return EAreaStyle::AllowRiding;
            } else {
                return EAreaStyle::Default;
            }
    }
}

NDrive::TLocationTags TAreasDB::CheckGeoTags(const TGeoCoord& c, const NDrive::TLocationTags& expectedTagNames, const TInstant reqActuality) const {
    NDrive::TLocationTags result;
    if (expectedTagNames.empty()) {
        return result;
    }
    const auto actor = [&result, &expectedTagNames](const TFullAreaInfo& area) {
        for (auto&& i : area.GetArea().GetTags()) {
            if (expectedTagNames.contains(i)) {
                result.emplace(i);
            }
        }
        for (auto&& i : area.GetTags()) {
            if (expectedTagNames.contains(i->GetName())) {
                result.emplace(i->GetName());
            }
        }
        return true;
    };
    ProcessTagsInPoint(c, actor, expectedTagNames, reqActuality);
    return result;
}

NDrive::TLocationTags TAreasDB::GetAreaTags(const TInstant reqActuality, TStringBuf locationType) const {
    NDrive::TLocationTags result;
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
    if (!snapshot) {
        return result;
    }
    for (auto&& i : *snapshot) {
        if (locationType && i.GetArea().GetType() != locationType) {
            continue;
        }
        const NDrive::TLocationTags& tags = i.GetArea().GetTags();
        result.insert(tags.begin(), tags.end());
    }
    return result;
}

NDrive::TLocationTags TAreasDB::GetAreaTags(const TDuration maxAge) const {
    return GetAreaTags(Now() - maxAge);
}

NDrive::TAreaIds TAreasDB::GetAreaIds(const TInstant reqActuality) const {
    NDrive::TAreaIds result;
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
    if (!snapshot) {
        return result;
    }
    for (auto&& i : *snapshot) {
        result.emplace(i.GetId());
    }
    return result;
}

NDrive::TAreaIds TAreasDB::GetAreaIds(TStringBuf locationType, const TInstant reqActuality) const {
    NDrive::TAreaIds result;
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
    if (!snapshot) {
        return result;
    }
    for (auto&& i : *snapshot) {
        if (locationType && i.GetArea().GetType() == locationType) {
            result.emplace(i.GetId());
        }
    }
    return result;
}

NDrive::TAreaIds TAreasDB::GetAreaIdsInPoint(const TGeoCoord& c, TInstant actuality, const TInternalPointContext internalPointContext) const {
    NDrive::TAreaIds result;
    const auto actor = [&result](const TFullAreaInfo& info) -> bool {
        result.emplace(info.GetArea().GetIdentifier());
        return true;
    };
    ProcessTagsInPoint(c, actor, {}, actuality, internalPointContext);
    return result;
}

NDrive::TLocationTags TAreasDB::GetTagsInPoint(const TGeoCoord& c, TInstant actuality, const TInternalPointContext& internalContext) const {
    NDrive::TLocationTags result;
    const auto actor = [&result](const TFullAreaInfo& info) -> bool {
        for (auto&& i : info.GetArea().GetTags()) {
            result.emplace(i);
        }
        return true;
    };
    ProcessTagsInPoint(c, actor, {}, actuality, internalContext);
    return result;
}

TConstAreasSnapshot::TPtr TAreasDB::GetSnapshot(TInstant actuality) const {
    Y_ENSURE_BT(DoRefresh(actuality));
    TReadGuard wg(ConstAreasSnapshotMutex);
    return ConstSnapshot;
}

TConstAreasSnapshot::TPtr TAreasDB::GetSnapshot(const TGeoCoord& coord, TInstant actuality) const {
    TFullAreaInfos areas;
    ProcessTagsInPoint(coord, [&areas](const TFullAreaInfo& area) {
        areas.push_back(area);
        return true;
    }, actuality);
    return MakeAtomicShared<TConstAreasSnapshot>(std::move(areas));
}

bool TAreasDB::ProcessAreaIdsImpl(const TSet<TString>& ids, const TFunction& actor, TConstAreasSnapshot::TPtr areasInfoHint) {
    if (!areasInfoHint) {
        return false;
    }
    for (auto&& i : *areasInfoHint) {
        if (!ids.contains(i.GetId())) {
            continue;
        }
        if (!actor(i)) {
            return false;
        }
    }
    return true;
}

bool TAreasDB::ProcessAreaIds(const TSet<TString>& ids, const TFunction& actor, TInstant actuality, TConstAreasSnapshot::TPtr areasInfoHint) const {
    if (areasInfoHint) {
        return ProcessAreaIdsImpl(ids, actor, areasInfoHint);
    } else {
        return ProcessAreaIdsImpl(ids, actor, GetSnapshot(actuality));
    }
}

bool TAreasDB::ProcessAreasInPoint(const TGeoCoord& c, const TFunction& actor, TInstant actuality, const TInternalPointContext& internalContext) const {
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(actuality);
    if (!snapshot) {
        return false;
    }
    for (auto&& i : *snapshot) {
        if (!i.IsPointInternal(c, internalContext)) {
            continue;
        }
        if (!actor(i)) {
            return false;
        }
    }
    return true;
}

bool TAreasDB::ProcessAreasIntersectingPolyline(const TPolyLine<TGeoCoord>& rhs, const TFunction& actor, TInstant actuality) const {
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(actuality);
    if (!snapshot) {
        return false;
    }
    for (auto&& i : *snapshot) {
        if (i.GetArea().GetPolyline().AreasIntersection(rhs).empty()) {
            continue;
        }
        if (!actor(i)) {
            return false;
        }
    }
    return true;
}

bool TAreasDB::ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, TInstant actuality, const TInternalPointContext& internalContext) const {
    return ProcessTagsInPoint(c, actor, {}, actuality, internalContext);
}

bool TAreasDB::ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, TDuration maxAge, const TInternalPointContext& internalContext) const {
    return ProcessTagsInPoint(c, actor, Now() - maxAge, internalContext);
}

bool TAreasDB::ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, const TSet<TString>& tagNames, TDuration maxAge, const TInternalPointContext& internalContext) const {
    return ProcessTagsInPoint(c, actor, tagNames, Now() - maxAge, internalContext);
}

bool TAreasDB::ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, const TSet<TString>& tagNames, TInstant actuality, const TInternalPointContext& internalContext) const {
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(actuality);
    if (!snapshot) {
        return false;
    }
    for (auto&& i : *snapshot) {
        if (!tagNames.empty() && !i.HasTags(tagNames)) {
            continue;
        }
        if (i.IsPointInternal(c, internalContext)) {
            if (!actor(i)) {
                break;
            }
        }
    }
    return true;
}

bool TAreasDB::ProcessTagsInPointWithInternalTypes(const TGeoCoord& c, const TFunctionWithInternalType& actor, const TSet<TString>& tagNames, TInstant actuality, const double precision) const {
    TConstAreasSnapshot::TPtr snapshot = GetSnapshot(actuality);
    if (!snapshot) {
        return false;
    }
    for (auto&& i : *snapshot) {
        if (!tagNames.empty() && !i.HasTags(tagNames)) {
            continue;
        }
        const TInternalPointContext::EInternalPointType pType = i.CheckPointInternal(c, precision);
        if (pType != TInternalPointContext::EInternalPointType::ExternalGuarantee) {
            if (!actor(i, pType)) {
                break;
            }
        }
    }
    return true;
}

bool TAreasDB::Process(IMessage* message) {
    const TAfterCacheInvalidateNotification* invalidateMessage = dynamic_cast<const TAfterCacheInvalidateNotification*>(message);
    if (invalidateMessage) {
        if (invalidateMessage->GetTableName() == "drive_areas_history") {
            RebuildCurrentSnapshot();
        }
        return true;
    }
    return TBase::Process(message);
}

bool TAreasDB::Refresh() {
    return DoRefresh(Now());
}

TFullAreaInfo::TFullAreaInfo(const TArea& area)
    : TTaggedArea(area.GetIdentifier(), TDBTags{}, TInstant::Zero())
    , Area(area)
    , AreaSize(Area.GetArea())
{
}

TFullAreaInfo::TFullAreaInfo(const TArea& area, const TTaggedArea& tagged)
    : TTaggedArea(tagged)
    , Area(area)
    , AreaSize(Area.GetArea())
{
    for (auto&& i : tagged.GetTags()) {
        HardTags.emplace(i->GetName());
    }
}

NJson::TJsonValue TFullAreaInfo::SerializeToReport() const {
    NJson::TJsonValue result = Area.SerializeToReport();
    result.InsertValue("area_size", AreaSize);
    NJson::TJsonValue& hardTags = result.InsertValue("hard_tags", NJson::JSON_ARRAY);
    for (auto&& i : GetTags()) {
        hardTags.AppendValue(i.SerializeToJson());
    }
    return result;
}

bool TFullAreaInfo::IsPointInternal(const TGeoCoord& c, const TInternalPointContext& internalContext) const {
    return internalContext.IsPointInternal(Area.GetPolyline(), c);
}

TInternalPointContext::EInternalPointType TFullAreaInfo::CheckPointInternal(const TGeoCoord& c, const double precision) const {
    TInternalPointContext context;
    context.SetPrecision(precision);
    return context.CheckPointInternal(Area.GetPolyline(), c);
}

bool TFullAreaInfo::HasTags(const TSet<TString>& tagNames) const {
    for (auto&& i : tagNames) {
        if (Area.GetTags().contains(i)) {
            return true;
        }
        if (HardTags.contains(i)) {
            return true;
        }
    }
    return false;
}

const TString& TCarLocationContextObjectId::GetObjectId(const IOffer* offer) const {
    if (offer) {
        return offer->GetObjectId();
    } else {
        return ObjectIdOriginal;
    }
}

TCarLocationContext TCarLocationContext::BuildByArea(const TString& objectId, const TArea& areaInfo, const NDrive::IServer& server) {
    TCarLocationContext result(objectId);
    if (areaInfo.GetPolyline().GetCoords().empty()) {
        return result;
    }
    const auto actor = [&result, &areaInfo](const TFullAreaInfo& info, const TInternalPointContext::EInternalPointType pointType) {
        if (!info.GetArea().GetPolyline().Size()) {
            return true;
        }
        if (!info.GetArea().GetPolyline().GetRectSafe().Contain(areaInfo.GetPolyline().GetRectSafe())) {
            return true;
        }
        if (info.GetId() == areaInfo.GetIdentifier()) {
            return true;
        }
        switch (pointType) {
            case TInternalPointContext::EInternalPointType::InternalGuarantee:
                result.AreasGuarantee.emplace(info.GetId());
            case TInternalPointContext::EInternalPointType::InternalPotentially:
            case TInternalPointContext::EInternalPointType::ExternalPotentially:
                result.AreasPotentially.emplace(info.GetId());
            case TInternalPointContext::EInternalPointType::ExternalGuarantee:
                break;
        };
        return true;
    };
    server.GetDriveAPI()->GetAreasDB()->ProcessTagsInPointWithInternalTypes(areaInfo.GetPolyline().GetCoords().front(), actor, {}, TInstant::Zero(), 0);
    result.AreasGuarantee.emplace(areaInfo.GetIdentifier());
    result.AreasPotentially.emplace(areaInfo.GetIdentifier());
    result.TagsGuarantee.insert(areaInfo.GetTags().begin(), areaInfo.GetTags().end());
    result.TagsPotentially.insert(areaInfo.GetTags().begin(), areaInfo.GetTags().end());
    return result;
}

TCarLocationContext TCarLocationContext::BuildByCoord(const TString& objectId, const TGeoCoord& c, const NDrive::IServer& server) {
    TCarLocationContext result(objectId);
    const auto actor = [&result](const TFullAreaInfo& info, const TInternalPointContext::EInternalPointType pointType) {
        switch (pointType) {
            case TInternalPointContext::EInternalPointType::InternalGuarantee:
                result.AreasGuarantee.emplace(info.GetId());
                result.TagsGuarantee.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
                result.TagsGuarantee.insert(info.GetHardTags().begin(), info.GetHardTags().end());
            case TInternalPointContext::EInternalPointType::InternalPotentially:
            case TInternalPointContext::EInternalPointType::ExternalPotentially:
                result.AreasPotentially.emplace(info.GetId());
                result.TagsPotentially.insert(info.GetArea().GetTags().begin(), info.GetArea().GetTags().end());
                result.TagsPotentially.insert(info.GetHardTags().begin(), info.GetHardTags().end());
            case TInternalPointContext::EInternalPointType::ExternalGuarantee:
                break;
        };
        return true;
    };
    server.GetDriveAPI()->GetAreasDB()->ProcessTagsInPointWithInternalTypes(c, actor, {}, TInstant::Zero(), server.GetSettings().GetValueDef<double>("snapshots.area_tags_precision", 20));
    return result;
}

TCarLocationContext::TCarLocationContext(const TString& objectId, const TRTDeviceSnapshot& snapshot, TConstAreasSnapshot::TPtr areasInfo)
    : TCarLocationContextObjectId(objectId)
    , Location(snapshot.GetLocation())
    , AreasGuarantee(snapshot.GetAreaIdsGuarantee())
    , AreasPotentially(snapshot.GetAreaIds())
    , TagsGuarantee(snapshot.GetLocationTagsGuarantee())
    , TagsPotentially(snapshot.GetLocationTagsPotentially())
    , IsRTLocation(Location && Location->IsRealtime())
    , AreasInfo(areasInfo)
{
}

TCarLocationFeatures TCarLocationContext::GetCarAreaFeatures(const bool force, const NDrive::IServer* server, const TInstant reqActuality /*= TInstant::Zero()*/) const {
    auto sBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", reqActuality);
    if (!sBuilder) {
        return TCarLocationFeatures::BuildDefault();
    }
    auto userSession = sBuilder->GetLastObjectSession(GetObjectId(nullptr));
    if (!userSession) {
        return GetCarAreaFeatures(force, nullptr, server, reqActuality);
    }
    const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(userSession.Get());
    if (!bSession) {
        return GetCarAreaFeatures(force, nullptr, server, reqActuality);
    }
    return GetCarAreaFeatures(force, bSession->GetCurrentOffer().Get(), server, reqActuality);
}

TTagsFilter TCarLocationContext::GetTagsFilterAllow(const IOffer* offer, const NDrive::IServer* server) const {
    const TStandartOffer* stOffer = dynamic_cast<const TStandartOffer*>(offer);
    if (stOffer && !stOffer->GetFinishAreaTagsFilter().IsEmpty()) {
        return stOffer->GetFinishAreaTagsFilter();
    } else if (SpecialAreasInfo) {
        return TTagsFilter(SpecialAreasInfo->GetAllowDropLocationTags(), false, true);
    } else {
        auto object = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(GetObjectId(offer));
        auto modelTag = object ? object->GetFirstTagByClass<TModelTag>() : TDBTag();
        if (modelTag) {
            return TModelTag::GetAllowDropTags(modelTag, *server);
        }
        return NDrive::DefaultAllowDropLocationTagsFilter;
    }
}

TTagsFilter TCarLocationContext::GetTagsFilterDeny(const IOffer* offer, const NDrive::IServer* server) const {
    if (SpecialAreasInfo) {
        return TTagsFilter(SpecialAreasInfo->GetDenyDropLocationTags(), false, true);
    }
    auto object = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(GetObjectId(offer));
    auto modelTag = object ? object->GetFirstTagByClass<TModelTag>() : TDBTag();
    if (modelTag) {
        auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(modelTag->GetName());
        auto description = dynamic_cast<const TModelTag::TDescription*>(tagDescription.Get());
        if (description && description->GetDenyDropTags().size()) {
            return TTagsFilter(description->GetDenyDropTags(), false, true);
        }
    }
    return NDrive::DefaultDenyDropLocationTagsFilter;
}

TTagsFilter TCarLocationContext::GetTagsFilterForceAllow(const IOffer* offer, const NDrive::IServer* server) const {
    if (SpecialAreasInfo) {
        return TTagsFilter(SpecialAreasInfo->GetForceAllowDropLocationTags(), false, true);
    }
    auto object = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(GetObjectId(offer));
    auto modelTag = object ? object->GetFirstTagByClass<TModelTag>() : TDBTag();
    if (modelTag) {
        auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(modelTag->GetName());
        auto description = dynamic_cast<const TModelTag::TDescription*>(tagDescription.Get());
        if (description && description->GetForceAllowDropTags().size()) {
            return TTagsFilter(description->GetForceAllowDropTags(), false, true);
        }
    }
    return NDrive::DefaultAllowForcedDropLocationTagsFilter;
}

TTagsFilter TCarLocationContext::GetTagsFilterAllowRiding(const IOffer* offer, const NDrive::IServer* server) const {
    const TStandartOffer* stOffer = dynamic_cast<const TStandartOffer*>(offer);
    if (stOffer && !stOffer->GetRidingAreaTagsFilter().IsEmpty()) {
        return stOffer->GetRidingAreaTagsFilter();
    } else if (SpecialAreasInfo) {
        const auto& allowRidingLocationTags = SpecialAreasInfo->GetAllowRidingLocationTags();
        return !allowRidingLocationTags.empty() ? TTagsFilter(allowRidingLocationTags, false, true) : NDrive::DefaultAllowRidingLocationTagsFilter;
    } else {
        auto object = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(GetObjectId(offer));
        auto modelTag = object ? object->GetFirstTagByClass<TModelTag>() : TDBTag();
        if (modelTag) {
            return TModelTag::GetAllowRidingTags(modelTag, *server);
        }
        return NDrive::DefaultAllowRidingLocationTagsFilter;
    }
}

TCarLocationFeatures TCarLocationContext::GetCarAreaFeaturesWithoutFees(bool force, const IOffer* offer, const NDrive::IServer* server) const {
    TCarLocationFeatures result;
    result.SetLocation(Location);
    if (!force) {
        const TTagsFilter& tagsFilterAllow = GetTagsFilterAllow(offer, server);
        const TTagsFilter& tagsFilterDeny = GetTagsFilterDeny(offer, server);
        const TTagsFilter& tagsFilterForceAllow = GetTagsFilterForceAllow(offer, server);
        const TTagsFilter& tagsFilterAllowRiding = GetTagsFilterAllowRiding(offer, server);
        if (!IsRTLocation) {
            if (tagsFilterForceAllow.IsMatching(TagsPotentially) || server->GetSettings().GetValueDef("parking_withno_gps", false)) {
                result.SetAllowDrop(EDropAbility::Allow);
            } else {
                result.SetAllowDrop(EDropAbility::DenyIncorrectData);
            }
            result.SetAllowRiding(true);
        } else {
            if (tagsFilterAllow.IsMatching(TagsPotentially)) {
                result.SetAllowDrop(EDropAbility::Allow);
            } else {
                result.SetAllowDrop(EDropAbility::NotAllow);
            }
            if (tagsFilterDeny.IsMatching(TagsGuarantee)) {
                result.SetAllowDrop(EDropAbility::Deny);
            }
        }
        if (tagsFilterAllowRiding.IsMatching(TagsPotentially)) {
            result.SetAllowRiding(true);
        } else {
            result.SetAllowRiding(false);
        }
    } else {
        result.SetAllowDrop(EDropAbility::Allow);
        result.SetAllowRiding(true);
    }
    return result;
}

TCarLocationFeatures TCarLocationContext::GetCarAreaFeatures(const bool force, const IOffer* offer, const NDrive::IServer* server, const TInstant reqActuality /*= TInstant::Zero()*/) const {
    TCarLocationFeatures result = GetCarAreaFeaturesWithoutFees(force, offer, server);
    if ((!!offer || !GetTakeFeesFromOffer()) && result.GetAllowDropDef(EDropAbility::Deny) == EDropAbility::Allow) {
        TMaybe<TOfferDropPolicy> feeInfo;
        const auto action = [this, offer, &feeInfo, &server](const TFullAreaInfo& areaInfo) {
            for (auto&& dbTag : areaInfo.GetTags()) {
                const TDropAreaFeatures* tag = dbTag.GetTagAs<TDropAreaFeatures>();
                if (!tag) {
                    continue;
                }
                if (GetTakeFeesFromOffer()) {
                    TMaybe<IOffer::TAreaInfo> offerAreaInfo = offer->GetOfferAreaInfo(areaInfo.GetId());
                    if (!offerAreaInfo) {
                        return true;
                    }
                    feeInfo = tag->GetFeeBySession(offer, server, offerAreaInfo->GetFee(), GetNeedInternalFees());
                } else {
                    feeInfo = tag->BuildFeeBySession(offer, server, areaInfo.GetId(), GetNeedInternalFees());
                }
                if (feeInfo) {
                    feeInfo->SetFeeConstructorId(dbTag.GetTagId());
                    return false;
                }
            }
            return true;
        };
        if (!server->GetDriveAPI()->GetAreasDB()->ProcessAreaIds(GuaranteeFees ? AreasGuarantee : AreasPotentially, action, reqActuality, AreasInfo)) {
            result.SetFeeInfo(std::move(feeInfo));
        }
    }

    return result;
}

TCarLocationFeatures TCarLocationContext::GetCarAreaFeatures(const bool force, const TSet<TString>& offerTags, const NDrive::IServer* server, const TInstant reqActuality /*= TInstant::Zero()*/) const {
    TCarLocationFeatures result = GetCarAreaFeaturesWithoutFees(force, nullptr, server);
    if (result.GetAllowDropDef(EDropAbility::Deny) == EDropAbility::Allow) {
        TMaybe<TOfferDropPolicy> feeInfo;
        const auto action = [this, offerTags, &feeInfo, server](const TFullAreaInfo& areaInfo) {
            for (auto&& dbTag : areaInfo.GetTags()) {
                const TDropAreaFeatures* tag = dbTag.GetTagAs<TDropAreaFeatures>();
                if (!tag) {
                    continue;
                }
                feeInfo = tag->BuildFeeBySession(&offerTags, server, areaInfo.GetId(), GetNeedInternalFees());
                if (feeInfo) {
                    feeInfo->SetFeeConstructorId(dbTag.GetTagId());
                    return false;
                }
            }
            return true;
        };
        if (!server->GetDriveAPI()->GetAreasDB()->ProcessAreaIds(GuaranteeFees ? AreasGuarantee : AreasPotentially, action, reqActuality, AreasInfo)) {
            result.SetFeeInfo(std::move(feeInfo));
        }
    }

    return result;
}

TMaybe<ui32> TCarLocationFeatures::GetDeltaFees(const TCarLocationFeatures& startFeatures) const {
    if (!FeeInfo) {
        return {};
    }
    if (startFeatures.GetAllowDropDef(EDropAbility::NotAllow) != EDropAbility::Allow) {
        return {};
    }
    if (!startFeatures.HasFeeInfo()) {
        return FeeInfo->GetFee();
    }
    if (FeeInfo->GetFeeConstructorId() == startFeatures.GetFeeInfoUnsafe().GetFeeConstructorId()) {
        return {};
    }
    if (FeeInfo->GetPolicy() != startFeatures.GetFeeInfoUnsafe().GetPolicy()) {
        return FeeInfo->GetFee();
    }

    if (FeeInfo->GetFee() > startFeatures.GetFeeInfoUnsafe().GetFee()) {
        switch (FeeInfo->GetPolicy()) {
            case EDropOfferPolicy::FeesMax:
                return FeeInfo->GetFee();
            default:
                return FeeInfo->GetFee() - startFeatures.GetFeeInfoUnsafe().GetFee();
        }
    } else {
        return 0;
    }
}

template <>
NJson::TJsonValue NJson::ToJson(const EDropAbility& object) {
    return ToString(object);
}

template class TEntityTagsManager<TTaggedArea, TAreaTagsHistoryManager>;
