#pragma once

#include "drop_object_features.h"
#include "location.h"

#include <drive/backend/data/operation_area.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/server/location/location.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/polyline.h>
#include <rtline/util/types/accessor.h>

class IOffer;

class TTooltip {
    R_FIELD(TString, Url);
    R_FIELD(TString, Icon);
    R_FIELD(TString, Message);
    R_FIELD(TString, BeaconMessage);
    R_FIELD(TString, Title);
    R_FIELD(TString, Image);
    R_FIELD(TString, SmallImage);
    R_FIELD(TString, TopColor);
    R_FIELD(TString, BottomColor);
    R_FIELD(TString, TextColor);

public:
    TTooltip() = default;

    NJson::TJsonValue SerializeToJson() const;
    NJson::TJsonValue BuildReport(const TMaybe<TString>& beaconParkingPlaceNumber, ELocalization locale, const NDrive::IServer& server) const;
    bool DeserializeFromJson(const TString& infoStr);
    bool DeserializeFromJson(const NJson::TJsonValue& info);
};

enum class EDropAbility {
    Allow,
    Deny,
    DenyIncorrectData,
    NotAllow
};

class TArea: public TAttributedEntity<TAttributedEntityDefaultFieldNames> {
    R_FIELD(NDrive::TAreaId, Identifier);
    R_FIELD(TString, Title);
    R_FIELD(ui32, Index, 0);
    R_FIELD(TString, Type, "ordinary");
    R_FIELD(TVector<TGeoCoord>, Coords);
    R_FIELD(TSet<TString>, Tags);
    R_FIELD(ui32, Revision, 0);
    R_READONLY(TPolyLine<TGeoCoord>, Polyline);
    R_OPTIONAL(TGeoCoord, HintPosition);
    R_FIELD(TString, ZoneName);

private:
    TAtomicSharedPtr<TTooltip> Tooltip;

    NJson::TJsonValue SerializeDetailsToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        if (HintPosition) {
            result.InsertValue("hint_position", HintPosition->ToString());
        }
        if (GetZoneName()) {
            result.InsertValue("group", GetZoneName());
        }
        TAttributedEntity::InsertObjectToJson(result);
        return result;
    }

    bool DeserializeDetailsFromJson(const NJson::TJsonValue& jsonInfo) {
        if (!TAttributedEntity::DeserializeObjectFromJson(jsonInfo)) {
            return false;
        }
        if (jsonInfo.Has("hint_position")) {
            if (!jsonInfo["hint_position"].IsString()) {
                return false;
            }
            TGeoCoord c;
            if (!c.DeserializeFromString(jsonInfo["hint_position"].GetString())) {
                return false;
            }
            HintPosition = c;
        }
        if (Type == "zone") {
            if (!jsonInfo.Has("group")) {
                return false;
            }
            ZoneName = jsonInfo["group"].GetString();
        }
        return true;
    }

public:
    using TId = TString;
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Identifier, -1);
        R_FIELD(i32, Title, -1);
        R_FIELD(i32, Index, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Tooltip, -1);
        R_FIELD(i32, Coords, -1);
        R_FIELD(i32, Tags, -1);
        R_FIELD(i32, Revision, -1);
        R_FIELD(i32, Details, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    static TString GetPropositionsTableName() {
        return "drive_area_propositions";
    }

    static TString GetAdministrativeEntity() {
        return "area";
    }

    static const IDBEntitiesWithPropositionsManager<TArea>* GetManager(const NDrive::IServer* server) {
        return server->GetAreasManager();
    }

public:
    TArea() = default;
    TArea(const NDrive::TAreaId& identifier)
        : Identifier(identifier)
    {
    }

    bool IsCluster() const {
        return Type == "cluster";
    }

    bool operator!() const {
        return (!Identifier) || (Coords.size() < 3);
    }

    const TString& GetInternalId() const {
        return GetIdentifier();
    }

    static TString GetTableName() {
        return "drive_areas";
    }

    static TString GetHistoryTableName() {
        return "drive_areas_history";
    }

    double GetArea() const {
        return Polyline.GetArea();
    }

    TAtomicSharedPtr<TTooltip> GetTooltip() const {
        return Tooltip;
    }

    TMaybe<ui32> OptionalRevision() const {
        return Revision ? Revision : TMaybe<ui32>();
    }

    bool operator<(const TArea& rhs) const {
        return GetIdentifier() < rhs.GetIdentifier();
    }

    bool operator==(const TArea& rhs) const {
        return GetIdentifier() == rhs.GetIdentifier();
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);

    NJson::TJsonValue SerializeToReport() const;
    NJson::TJsonValue SerializeToPublicReport() const;
    NJson::TJsonValue BuildJsonReport() const {
        return SerializeToReport();
    }

    static NDrive::TScheme GetScheme(const IServerBase& server);

    bool Parse(const NStorage::TTableRecord& record);

    NStorage::TTableRecord SerializeToTableRecord() const;
    NStorage::TTableRecord SerializeUniqueToTableRecord() const;
};

class TAreasHistoryManager: public TIndexedAbstractHistoryManager<TArea> {
private:
    using TBase = TIndexedAbstractHistoryManager<TArea>;

public:
    TAreasHistoryManager(const ITagsHistoryContext& context)
        : TBase(context, "drive_areas_history", THistoryConfig().SetDeep(TDuration::Max()))
    {
    }
};

class TAreaTagsHistoryManager
    : public TAbstractTagsHistoryManager
    , public TTagEventsManager
{
private:
    using TBase = TAbstractTagsHistoryManager;

protected:
    virtual NEntityTagsManager::EEntityType GetManagerEntityType() const override {
        return NEntityTagsManager::EEntityType::Area;
    }

public:
    static TString TagsHistoryTableName() {
        return "area_tags_history";
    }

public:
    TAreaTagsHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, config, TagsHistoryTableName())
        , TTagEventsManager(context, TagsHistoryTableName())
    {
    }
};

class TTaggedArea: public TTaggedObject {
private:
    using TBase = TTaggedObject;

public:
    using TBase::TBase;
};

class TFullAreaInfo: public TTaggedArea {
public:
    using TId = TString;

public:
    R_FIELD(TArea, Area);
    R_READONLY(TSet<TString>, HardTags);
    R_READONLY(double, AreaSize, Max<ui32>());

public:
    TFullAreaInfo(const TArea& area);
    TFullAreaInfo(const TArea& area, const TTaggedArea& tagged);

    bool operator<(const TFullAreaInfo& item) const {
        return AreaSize < item.AreaSize;
    }

    NJson::TJsonValue SerializeToReport() const;

    bool HasTags(const TSet<TString>& tagNames) const;
    bool IsPointInternal(const TGeoCoord& c, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    TInternalPointContext::EInternalPointType CheckPointInternal(const TGeoCoord& c, const double precision) const;
};
using TFullAreaInfos = TVector<TFullAreaInfo>;

class TConstAreasSnapshot : TNonCopyable {
private:
    TVector<TFullAreaInfo> Areas;

public:
    using TPtr = TAtomicSharedPtr<TConstAreasSnapshot>;

public:
    TConstAreasSnapshot(const TVector<TFullAreaInfo>& areas)
        : Areas(areas)
    {
    }
    TConstAreasSnapshot(TVector<TFullAreaInfo>&& areas)
        : Areas(std::move(areas))
    {
    }

    TVector<TFullAreaInfo>::const_iterator begin() const {
        return Areas.begin();
    }
    TVector<TFullAreaInfo>::const_iterator end() const {
        return Areas.end();
    }
    const TVector<TFullAreaInfo>& GetAreas() const {
        return Areas;
    }
};

class TAreasSnapshot {
private:
    TRWMutex Mutex;
    TVector<TFullAreaInfo> Areas;

public:
    using TFunction = std::function<bool(const TFullAreaInfo&)>;
    using TPtr = TAtomicSharedPtr<TAreasSnapshot>;

public:
    TConstAreasSnapshot::TPtr BuildConstSnapshot() const;

    bool ForAllAreas(const TFunction& action) const;
    bool UpsertArea(const TArea& area);
    bool RemoveArea(const TArea& area);
    void SetAreas(TAreasSnapshot&& snapshot);
    TVector<TFullAreaInfo> ExtractAreas() &&;

    bool UpsertTag(const TConstDBTag& tag, const ITagsHistoryContext& context);
    bool RemoveTag(const TConstDBTag& tag);
};

class TAreaTagsManager: public TCachedEntityTagsManager<TTaggedArea, TAreaTagsHistoryManager> {
private:
    using TBase = TCachedEntityTagsManager<TTaggedArea, TAreaTagsHistoryManager>;

private:
    const ITagsHistoryContext& Context;
    TAreasSnapshot& Snapshot;

protected:
    virtual TSet<TString> GetDependencesAfter() const override {
        return {"drive_areas_history"};
    }

    virtual NDrive::IObjectSnapshot::TPtr BuildSnapshot(const TString& /*objectId*/, const NDrive::IServer* /*server*/) const override {
        return nullptr;
    }

    virtual NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Area;
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TConstDBTag>& ev) const override;
    virtual bool DoRebuildCacheUnsafe() const override;

public:
    virtual TSet<TString> GetDependencesBefore() const override {
        return {"drive_areas_history"};
    }

    TAreaTagsManager(const ITagsHistoryContext& context, const THistoryConfig& hConfig, TAreasSnapshot& snapshot)
        : TBase("area_tags", context, hConfig)
        , Context(context)
        , Snapshot(snapshot)
    {
    }
};

class TAreaManipulationConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "area_id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("area_id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TArea& object) {
        return BuildCondition(object.GetIdentifier());
    }
};

class TCarLocationFeatures {
public:
    R_OPTIONAL(TOfferDropPolicy, FeeInfo);
    R_OPTIONAL(NDrive::TLocation, Location);
    R_OPTIONAL(EDropAbility, AllowDrop);
    R_FIELD(bool, AllowRiding, false);

public:
    TCarLocationFeatures() = default;
    explicit TCarLocationFeatures(const TMaybe<TOfferDropPolicy>& feeInfo)
        : FeeInfo(feeInfo)
    {
    }

    TMaybe<ui32> GetDeltaFees(const TCarLocationFeatures& startFeatures) const;

    static TCarLocationFeatures BuildDefault() {
        return TCarLocationFeatures();
    }
};

class TRTDeviceSnapshot;

class TCarLocationContextObjectId {
private:
    const TString ObjectIdOriginal;

protected:
    const TString& GetObjectId(const IOffer* offer) const;

public:
    TCarLocationContextObjectId(const TString& objectId)
        : ObjectIdOriginal(objectId)
    {
    }
};

class TCarLocationContext: public TCarLocationContextObjectId {
private:
    R_FIELD(bool, GuaranteeFees, true);
    R_FIELD(bool, TakeFeesFromOffer, true);
    R_FIELD(bool, NeedInternalFees, false);
    R_OPTIONAL(TOperationAreaTag::TDescription, SpecialAreasInfo);
    R_OPTIONAL(NDrive::TLocation, Location);

private:
    TSet<TString> AreasGuarantee;
    TSet<TString> AreasPotentially;
    TSet<TString> TagsGuarantee;
    TSet<TString> TagsPotentially;
    bool IsRTLocation = true;
    TConstAreasSnapshot::TPtr AreasInfo = nullptr;

private:
    TCarLocationContext(const TString& objectId)
        : TCarLocationContextObjectId(objectId)
    {
    }

    TTagsFilter GetTagsFilterAllow(const IOffer* offer, const NDrive::IServer* server) const;
    TTagsFilter GetTagsFilterAllowRiding(const IOffer* offer, const NDrive::IServer* server) const;
    TTagsFilter GetTagsFilterForceAllow(const IOffer* offer, const NDrive::IServer* server) const;
    TTagsFilter GetTagsFilterDeny(const IOffer* offer, const NDrive::IServer* server) const;

public:
    TCarLocationContext(const TString& objectId, const TRTDeviceSnapshot& snapshot, TConstAreasSnapshot::TPtr areasInfo = nullptr);

    const TSet<TString>& GetTagsPotentially() const {
        return TagsPotentially;
    }

    static TCarLocationContext BuildByCoord(const TString& objectId, const TGeoCoord& c, const NDrive::IServer& server);
    static TCarLocationContext BuildByArea(const TString& objectId, const TArea& areaInfo, const NDrive::IServer& server);

    TCarLocationFeatures GetCarAreaFeatures(const bool force, const NDrive::IServer* server, const TInstant reqActuality = TInstant::Zero()) const;
    TCarLocationFeatures GetCarAreaFeatures(const bool force, const IOffer* offer, const NDrive::IServer* server, const TInstant reqActuality = TInstant::Zero()) const;
    TCarLocationFeatures GetCarAreaFeatures(const bool force, const TSet<TString>& offerTags, const NDrive::IServer* server, const TInstant reqActuality = TInstant::Zero()) const;

private:
    TCarLocationFeatures GetCarAreaFeaturesWithoutFees(bool force, const IOffer* offer, const NDrive::IServer* server) const;
};

class TAreasDB: public TDBEntitiesManagerWithPropositions<TArea, TAreaManipulationConditionConstructor> {
private:
    using TBase = TDBEntitiesManagerWithPropositions<TArea, TAreaManipulationConditionConstructor>;

public:
    using TFunction = std::function<bool(const TFullAreaInfo&)>;
    using TFunctionWithInternalType = std::function<bool(const TFullAreaInfo&, TInternalPointContext::EInternalPointType)>;

private:
    mutable TAreasSnapshot Snapshot;
    mutable TConstAreasSnapshot::TPtr ConstSnapshot = new TConstAreasSnapshot({});
    TRWMutex ConstAreasSnapshotMutex;

    TAreaTagsManager TagsManager;
    const TDBEntitiesManagerConfig Config;

protected:
    virtual TSet<TString> GetDependencesAfter() const override {
        return {"drive_areas_history"};
    }

    void DoAcceptHistoryEventBeforeRemoveUnsafe(const TObjectEvent<TArea>& ev) const override {
        Snapshot.RemoveArea(ev);
    }

    void DoAcceptHistoryEventAfterChangeUnsafe(const TObjectEvent<TArea>& ev, TArea& /*object*/) const override {
        Snapshot.UpsertArea(ev);
    }

    virtual bool DoRebuildCacheUnsafe() const override;

    virtual bool DoStart() override;
    virtual bool DoStop() override;

    bool DoRefresh(const TInstant reqActuality) const;
    void RebuildCurrentSnapshot() const;

public:
    TAreasDB(const ITagsHistoryContext& context, const TDBEntitiesManagerWithPropositionsConfig& config)
        : TBase(context, config)
        , TagsManager(context, THistoryConfig().SetDeep(TDuration::Max()), Snapshot)
        , Config(config)
    {
    }

    enum class EAreaStyle {
        AllowDelivery /* "allow_delivery" */,
        AllowDropFees /* "allow_drop_fees" */,
        AllowDropNoFees /* "allow_drop_no_fees" */,
        AllowRiding /* "allow_riding" */,
        Default /* "default" */,
        DenyDrop /* "deny_drop" */,
    };

    static EAreaStyle GetStyle(EDropAbility dropAbility, bool allowRiding, bool allowDelivery, bool withFees);

    NDrive::TLocationTags CheckGeoTags(const TGeoCoord& c, const NDrive::TLocationTags& expectedTagNames, const TInstant reqActuality) const;
    NDrive::TLocationTags GetAreaTags(const TInstant reqActuality = TInstant::Zero(), TStringBuf locationType = {}) const;
    NDrive::TLocationTags GetAreaTags(const TDuration maxAge) const;
    NDrive::TAreaIds GetAreaIds(const TInstant reqActuality = TInstant::Zero()) const;
    NDrive::TAreaIds GetAreaIds(TStringBuf locationType, const TInstant reqActuality = TInstant::Zero()) const;
    NDrive::TAreaIds GetAreaIdsInPoint(const TGeoCoord& c, TInstant actuality = TInstant::Zero(), const TInternalPointContext internalPointContext = TInternalPointContext()) const;
    NDrive::TLocationTags GetTagsInPoint(const TGeoCoord& c, TInstant actuality = TInstant::Zero(), const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;

    TAreaTagsManager& GetTagsManager() {
        return TagsManager;
    }

    const TAreaTagsManager& GetTagsManager() const {
        return TagsManager;
    }

    TConstAreasSnapshot::TPtr GetSnapshot(TInstant actuality = TInstant::Zero()) const;
    TConstAreasSnapshot::TPtr GetSnapshot(const TGeoCoord& c, TInstant actuality = TInstant::Zero()) const;

    bool ProcessAreaIds(const TSet<TString>& ids, const TFunction& actor, TInstant actuality = TInstant::Zero(), TConstAreasSnapshot::TPtr areasInfoHint = nullptr) const;
    bool ProcessAreasInPoint(const TGeoCoord& c, const TFunction& actor, TInstant actuality = TInstant::Zero(), const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    static bool ProcessAreaIdsImpl(const TSet<TString>& ids, const TFunction& actor, TConstAreasSnapshot::TPtr areasInfoHint);
    bool ProcessAreasIntersectingPolyline(const TPolyLine<TGeoCoord>& rhs, const TFunction& actor, TInstant actuality = TInstant::Zero()) const;

    bool ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, TInstant actuality = TInstant::Zero(), const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    bool ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, TDuration maxAge, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    bool ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, const TSet<TString>& tagNames, TDuration maxAge, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    bool ProcessTagsInPoint(const TGeoCoord& c, const TFunction& actor, const TSet<TString>& tagNames, TInstant actuality = TInstant::Zero(), const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const;
    bool ProcessTagsInPointWithInternalTypes(const TGeoCoord& c, const TFunctionWithInternalType& actor, const TSet<TString>& tagNames, TInstant actuality, const double precision) const;

    template <class TTag, class TActor>
    bool ProcessHardTagsInPoint(const TGeoCoord& c, const TActor& actor, const TInstant reqActuality, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const {
        TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
        for (auto&& i : *snapshot) {
            int isInside = 0;
            for (auto&& tag : i.GetTags()) {
                const TTag* tagSpecial = tag.GetTagAs<TTag>();
                if (tagSpecial) {
                    if (isInside == 0) {
                        isInside = i.IsPointInternal(c, internalContext) ? 1 : -1;
                    }
                    if (isInside == 1) {
                        if (!actor(tagSpecial)) {
                            return false;
                        }
                    }
                }
            }
        }
        return true;
    }

    template <class TTag, class TActor>
    bool ProcessHardDBTagsInPoint(const TGeoCoord& c, const TActor& actor, const TInstant reqActuality, const TInternalPointContext& internalContext = Default<TInternalPointContext>()) const {
        TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
        for (auto&& i : *snapshot) {
            int isInside = 0;
            for (auto&& tag : i.GetTags()) {
                const TTag* tagSpecial = tag.GetTagAs<TTag>();
                if (tagSpecial) {
                    if (isInside == 0) {
                        isInside = i.IsPointInternal(c, internalContext) ? 1 : -1;
                    }
                    if (isInside == 1) {
                        if (!actor(tag)) {
                            return false;
                        }
                    }
                }
            }
        }
        return true;
    }

    template <class TTag, class TActor>
    bool ProcessHardDBTags(const TActor& actor, const TInstant reqActuality = TInstant::Zero()) const {
        TConstAreasSnapshot::TPtr snapshot = GetSnapshot(reqActuality);
        for (auto&& i : *snapshot) {
            for (auto&& tag : i.GetTags()) {
                const TTag* tagSpecial = tag.GetTagAs<TTag>();
                if (tagSpecial) {
                    if (!actor(tag)) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    template <class TTag>
    bool HasHardTagsInPoint(const TGeoCoord& c, const TInstant reqActuality = TInstant::Zero()) const {
        const auto action = [](const TTag * /*tag*/) -> bool {
            return false;
        };
        return !ProcessHardTagsInPoint<TTag>(c, action, reqActuality);
    }

    bool GetHistory(const TInstant since, TVector<TAtomicSharedPtr<TObjectEvent<TArea>>>& result, const TInstant reqActuality) const {
        return HistoryManager->GetEventsAll(since, result, reqActuality);
    }

    virtual bool Process(IMessage* message) override;
    virtual bool Refresh() override;
};
