#include "drop_object_features.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/device_snapshot/manager.h>

const TString& TObjectDropPolicyCommon::GetNextLanding(const ui32 viewsCount) const {
    if (AlertLandingIds.empty()) {
        return Default<TString>();
    }
    if (AlertsCount == 0) {
        return Default<TString>();
    }
    if (viewsCount < AlertLandingIds.size()) {
        return AlertLandingIds[viewsCount];
    } else {
        return AlertLandingIds.back();
    }
}

NDrive::NProto::TOfferDropPolicy TObjectDropPolicyCommon::SerializeToProto() const {
    NDrive::NProto::TOfferDropPolicy result;
    result.SetAlertsCount(AlertsCount);
    result.SetAlertLandingIds(JoinSeq(", ", AlertLandingIds));
    result.SetPolicy(::ToString(Policy));
    result.SetAlertType(AlertType);
    return result;
}

bool TObjectDropPolicyCommon::DeserializeFromProto(const NDrive::NProto::TOfferDropPolicy& proto) {
    AlertLandingIds = StringSplitter(proto.GetAlertLandingIds()).SplitBySet(", ").SkipEmpty().ToList<TString>();
    AlertsCount = proto.GetAlertsCount();
    AlertType = proto.GetAlertType();
    if (!TryFromString(proto.GetPolicy(), Policy)) {
        return false;
    }
    return true;
}

NDrive::TScheme TObjectDropPolicyCommon::GetScheme(const NDrive::IServer* /*server*/) {
    NDrive::TScheme result;
    result.Add<TFSString>("AlertLandingIds", "Идентификаторы интроскринов");
    result.Add<TFSNumeric>("AlertsCount", "Количество предупреждений").SetPrecision(0).SetMin(0).SetMax(10).SetDefault(2);
    result.Add<TFSVariants>("Policy", "Политика обработки ситуации").InitVariants<EDropOfferPolicy>();
    result.Add<TFSString>("AlertType", "Идентификатор предупреждения");
    return result;
}

NDrive::TScheme TScaledFeePolicy::TMarker::GetScheme(const NDrive::IServer* /*server*/) {
    NDrive::TScheme result;
    result.Add<TFSNumeric>("CarsCount", "количество машин").SetMin(0).SetMax(100000).SetDefault(0);
    result.Add<TFSNumeric>("Fee", "штраф (копейки)").SetMin(0).SetMax(10000000).SetDefault(0);
    return result;
}


ui32 TScaledFeePolicy::GetFee(const NDrive::IServer& server, const TString& areaId) const {
    if (Markers.size() == 1) {
        return Markers.front().GetFee();
    }
    TConstDevicesSnapshot cds = server.GetSnapshotsManager().GetSnapshots();
    ui32 carsCount = cds.GetAreaCarsCount(areaId);
    if (WithFutures) {
        carsCount += cds.GetFutureAreaCarsCount(areaId);
    }
    return GetScaledFee(carsCount);
}

NDrive::TScheme TScaledFeePolicy::GetScheme(const NDrive::IServer* server) {
    NDrive::TScheme result;
    result.Add<TFSArray>("Markers", "маркеры").SetElement(TMarker::GetScheme(server));
    result.Add<TFSBoolean>("WithFutures", "учитывать едущие по fix машины").SetDefault(false);
    return result;
}

NDrive::TScheme TAreaDropPolicyBuilder::GetScheme(const NDrive::IServer* server) {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("OfferAttributesFilter", "Фильтр для атрибутов офферов");
    result.Add<TFSStructure>("ScaledFeePolicy", "Политика расчета переменного штрафа").SetStructure(TScaledFeePolicy::GetScheme(server));
    result.Add<TFSBoolean>("IsInternalFlowCorrector", "Для внутренних потоков").SetDefault(false);
    return result;
}
