#pragma once
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/tags/tags_filter.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/types/accessor.h>

enum class EDropOfferPolicy {
    Allow = 0 /* "allow" */,
    Deny = 1 /* "deny" */,
    FeesMax = 2 /* "fees_max" */,
    FeesFix = 3 /* "fees_fix" */,
    FeesMinute = 4 /* "fees_minute" */,
    Suspended = 5 /* "suspended" */,
};

class TDropPolicyProvider {
public:
    R_FIELD(TString, AlertLandingId);
    R_FIELD(ui32, FinishOfferFee, 0);
    R_FIELD(EDropOfferPolicy, FinishOfferFeePolicy, EDropOfferPolicy::Allow);

public:
    TDropPolicyProvider() = default;

    void Drop() {
        AlertLandingId = "";
        FinishOfferFee = 0;
        FinishOfferFeePolicy = EDropOfferPolicy::Allow;
    }

    void CopyTo(TDropPolicyProvider& acceptor) const {
        acceptor.SetAlertLandingId(GetAlertLandingId());
        acceptor.SetFinishOfferFee(GetFinishOfferFee());
        acceptor.SetFinishOfferFeePolicy(GetFinishOfferFeePolicy());
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        if (FinishOfferFee) {
            proto.SetFinishOfferFee(FinishOfferFee);
        }
        if (FinishOfferFeePolicy != EDropOfferPolicy::Allow) {
            proto.SetFinishOfferFeePolicy((ui32)FinishOfferFeePolicy);
        }
        if (!!AlertLandingId) {
            proto.SetAlertLandingId(AlertLandingId);
        }
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        if (proto.HasAlertLandingId()) {
            AlertLandingId = proto.GetAlertLandingId();
        }
        if (proto.HasFinishOfferFeePolicy()) {
            FinishOfferFeePolicy = (EDropOfferPolicy)proto.GetFinishOfferFeePolicy();
        } else {
            FinishOfferFeePolicy = EDropOfferPolicy::Allow;
        }
        if (proto.HasFinishOfferFee()) {
            FinishOfferFee = proto.GetFinishOfferFee();
        }
        return true;
    }
};

class TScaledFeePolicy {
public:
    class TMarker {
        R_FIELD(ui32, CarsCount, 0);
        R_FIELD(ui32, Fee, 0);
    public:
        TMarker() = default;
        TMarker(const ui32 carsCount, const ui32 fee)
            : CarsCount(carsCount)
            , Fee(fee)
        {
        }

        NDrive::NProto::TScaledFeePolicyMarker SerializeToProto() const {
            NDrive::NProto::TScaledFeePolicyMarker result;
            result.SetCarsCount(CarsCount);
            result.SetFee(Fee);
            return result;
        }

        bool DeserializeFromProto(const NDrive::NProto::TScaledFeePolicyMarker& info) {
            CarsCount = info.GetCarsCount();
            Fee = info.GetFee();
            return true;
        }

    public:
        static NDrive::TScheme GetScheme(const NDrive::IServer* server);
    };

private:
    R_FIELD(TVector<TMarker>, Markers);
    R_FIELD(bool, WithFutures, false);

public:
    TScaledFeePolicy() = default;

    bool DeserializeFromProto(const NDrive::NProto::TScaledFeePolicy& info) {
        Markers.clear();
        for (auto&& i : info.GetMarkers()) {
            TMarker marker;
            if (!marker.DeserializeFromProto(i)) {
                return false;
            }
            Markers.emplace_back(std::move(marker));
            if (Markers.size() > 1 && Markers[Markers.size() - 2].GetCarsCount() > Markers.back().GetCarsCount()) {
                return false;
            }
        }
        WithFutures = info.GetWithFutures();
        return Markers.size();
    }

    ui32 GetFee(const NDrive::IServer& server, const TString& areaId) const;

    ui32 GetScaledFee(const ui32 carsCount) const {
        Y_ENSURE_BT(Markers.size());
        if (Markers.front().GetCarsCount() >= carsCount) {
            return Markers.front().GetFee();
        }
        if (Markers.back().GetCarsCount() <= carsCount) {
            return Markers.back().GetFee();
        }
        for (ui32 i = 0; i + 1 < Markers.size(); ++i) {
            if (Markers[i].GetCarsCount() <= carsCount && carsCount <= Markers[i + 1].GetCarsCount()) {
                if (Markers[i].GetCarsCount() == Markers[i + 1].GetCarsCount()) {
                    return Markers[i].GetFee();
                }
                const double alpha = 1.0 * (carsCount - Markers[i].GetCarsCount()) / (Markers[i + 1].GetCarsCount() - Markers[i].GetCarsCount());
                return Markers[i + 1].GetFee() * alpha + Markers[i].GetFee() * (1 - alpha);
            }
        }
        ERROR_LOG << "Incorrect scaled fee construct" << Endl;
        return 0;
    }

    TScaledFeePolicy(const ui32 fee) {
        Markers.clear();
        Markers.emplace_back(0, fee);
    }

    NDrive::NProto::TScaledFeePolicy SerializeToProto() const {
        NDrive::NProto::TScaledFeePolicy result;
        for (auto&& i : Markers) {
            *result.AddMarkers() = i.SerializeToProto();
        }
        result.SetWithFutures(WithFutures);
        return result;
    }

    static TMaybe<TScaledFeePolicy> BuildFromProto(const NDrive::NProto::TScaledFeePolicy& info) {
        TScaledFeePolicy result;
        if (result.DeserializeFromProto(info)) {
            return result;
        }
        return {};
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer* server);
};

class TObjectDropPolicyCommon {
private:
    R_FIELD(ui32, AlertsCount, 0);
    R_FIELD(TVector<TString>, AlertLandingIds);
    R_FIELD(TString, AlertType);
    R_FIELD(EDropOfferPolicy, Policy, EDropOfferPolicy::Allow);

public:
    const TString& GetNextLanding(const ui32 viewsCount) const;
    NDrive::NProto::TOfferDropPolicy SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TOfferDropPolicy& proto);
    static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/);
};

class TOfferDropPolicy: public TObjectDropPolicyCommon {
private:
    using TBase = TObjectDropPolicyCommon;

private:
    R_FIELD(TString, FeeConstructorId);
    R_FIELD(ui32, Fee, 0);

public:
    TOfferDropPolicy() = default;
    TOfferDropPolicy(const TBase& base)
        : TBase(base)
    {
    }

    NDrive::NProto::TOfferDropPolicy SerializeToProto() const {
        NDrive::NProto::TOfferDropPolicy proto = TBase::SerializeToProto();
        proto.SetFee(Fee);
        return proto;
    }

    bool DeserializeFromProto(const NDrive::NProto::TOfferDropPolicy& proto) {
        if (!TBase::DeserializeFromProto(proto)) {
            return false;
        }
        Fee = proto.GetFee();
        return true;
    }
};

class TAreaDropPolicyBuilder: public TObjectDropPolicyCommon {
private:
    using TBase = TObjectDropPolicyCommon;

public:
    R_FIELD(TTagsFilter, OfferAttributesFilter);
    R_FIELD(TScaledFeePolicy, ScaledFeePolicy);
    R_FIELD(bool, IsInternalFlowCorrector, false);

public:
    TAreaDropPolicyBuilder& SetFee(const ui32 fee) {
        ScaledFeePolicy.MutableMarkers().clear();
        ScaledFeePolicy.MutableMarkers().emplace_back(0, fee);
        return *this;
    }

    TOfferDropPolicy GetOfferDropPolicy(const NDrive::IServer& server, const TString& areaId) const {
        TOfferDropPolicy result(*this);
        result.SetFee(ScaledFeePolicy.GetFee(server, areaId));
        return result;
    }

    TOfferDropPolicy GetOfferDropPolicy(const ui32 fee) const {
        TOfferDropPolicy result(*this);
        result.SetFee(fee);
        return result;
    }

    NDrive::NProto::TOfferDropPolicy SerializeToProto() const {
        NDrive::NProto::TOfferDropPolicy proto = TBase::SerializeToProto();
        *proto.MutableScaledFeePolicy() = ScaledFeePolicy.SerializeToProto();
        if (!OfferAttributesFilter.IsEmpty()) {
            proto.SetOfferAttributesFilter(OfferAttributesFilter.ToString());
        }
        proto.SetIsInternalFlowCorrector(IsInternalFlowCorrector);
        return proto;
    }

    bool DeserializeFromProto(const NDrive::NProto::TOfferDropPolicy& proto) {
        if (!TBase::DeserializeFromProto(proto)) {
            return false;
        }
        if (proto.HasIsInternalFlowCorrector()) {
            IsInternalFlowCorrector = proto.GetIsInternalFlowCorrector();
        }
        if (proto.HasFee()) {
            ScaledFeePolicy = TScaledFeePolicy(proto.GetFee());
        } else if (proto.HasScaledFeePolicy()) {
            if (!ScaledFeePolicy.DeserializeFromProto(proto.GetScaledFeePolicy())) {
                return false;
            }
        }
        if (proto.HasOfferAttributesFilter() && !OfferAttributesFilter.DeserializeFromString(proto.GetOfferAttributesFilter())) {
            return false;
        }
        return true;
    }

    static NDrive::TScheme GetScheme(const NDrive::IServer* server);
};
