#include "blackbox.h"

#include "auth.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/auth_client_parser/cookie.h>
#include <library/cpp/auth_client_parser/oauth_token.h>
#include <library/cpp/http/cookies/cookies.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/vector.h>

namespace {
    TNamedSignalSimple BlackboxRequest("blackbox-request");
    TNamedSignalSimple BlackboxOk("blackbox-ok");
    TNamedSignalSimple BlackboxException("blackbox-exception");
    TNamedSignalSimple BlackboxNull("blackbox-null");
    TNamedSignalSimple BlackboxTimeout("blackbox-timeout");
    TNamedSignalEnum<NBlackbox2::TSessionResp::EStatus> BlackboxStatus("blackbox", EAggregationType::Sum, "dmmm");
    TNamedSignalHistogram BlackboxTimes("blackbox-times", NRTLineHistogramSignals::IntervalsRTLineReply);
}

TBlackbox2AuthConfig::TBlackbox2AuthConfig(const TString& name)
    : IAuthModuleConfig(name)
{
}

TBlackbox2AuthConfig::~TBlackbox2AuthConfig() {
}

void TBlackbox2AuthConfig::Init(const TYandexConfig::Section* section) {
    CHECK_WITH_LOG(section);
    const auto& directives = section->GetDirectives();

    AuthMethod = directives.Value("AuthMethod", AuthMethod);
    IgnoreDeviceId = directives.Value("IgnoreDeviceId", IgnoreDeviceId);
    SelfClientId = directives.Value("SelfClientId", SelfClientId);
    EnableUserTickets = directives.Value("EnableUserTickets", EnableUserTickets);
    AcceptedScopes = StringSplitter(directives.Value("AcceptedScopes", JoinSeq(",", AcceptedScopes))).SplitBySet(" ,").SkipEmpty();
    RequiredScopes = StringSplitter(directives.Value("Scopes", JoinSeq(",", RequiredScopes))).SplitBySet(" ,").SkipEmpty();
    Environment = directives.Value("Environment", Environment);

    Url = directives.Value("BlackboxUrl", Url);
    AssertCorrectConfig(!!Url, "empty 'BlackboxUrl' field");

    CookieHost = directives.Value("CookieHost", CookieHost);
    AssertCorrectConfig(!CookieHost.empty(), "empty 'CookieHost' field");

    DestinationClientId = directives.Value("DestinationClientId", DestinationClientId);
    AssertCorrectConfig(SelfClientId == 0 || DestinationClientId != 0, "zero 'DestinationClientId' field");
    AuthHeader = directives.Value("AuthHeader", AuthHeader);
    AssertCorrectConfig(!AuthHeader.empty(), "empty 'AuthHeader' field");
}

void TBlackbox2AuthConfig::ToString(IOutputStream& os) const {
    os << "AuthMethod: " << AuthMethod << Endl;
    os << "BlackboxUrl: " << Url << Endl;
    os << "CookieHost: " << CookieHost << Endl;
    os << "IgnoreDeviceId: " << IgnoreDeviceId << Endl;
    os << "DestinationClientId: " << DestinationClientId << Endl;
    os << "SelfClientId: " << SelfClientId << Endl;
    os << "EnableUserTickets: " << EnableUserTickets << Endl;
    os << "AcceptedScopes: " << JoinSeq(",", AcceptedScopes) << Endl;
    os << "Scopes: " << JoinSeq(",", RequiredScopes) << Endl;
    os << "Environment: " << Environment << Endl;
    os << "AuthHeader: " << AuthHeader << Endl;
}

THolder<IAuthModule> TBlackbox2AuthConfig::ConstructAuthModule(const IServerBase* server) const {
    if (!server) {
        ERROR_LOG << "nullptr IServerBase" << Endl;
        return nullptr;
    }
    if (!Client) {
        auto tvm = server->GetTvmClient(SelfClientId);
        if (!tvm && SelfClientId) {
            ERROR_LOG << "cannot find TVM client for SelfClientId " << SelfClientId << Endl;
            return nullptr;
        }
        auto client = MakeAtomicShared<NDrive::TBlackboxClient>(Url, tvm);
        if (CookieHost) {
            client->SetCookieHost(CookieHost);
        }
        if (DestinationClientId) {
            client->SetDestinationClientId(DestinationClientId);
        }
        if (!RequiredScopes.empty()) {
            client->SetScopes(RequiredScopes);
        }
        if (EnableUserTickets) {
            client->EnableUserTickets();
        }
        auto guard = Guard(ClientLock);
        Client = std::move(client);
    }
    return MakeHolder<TBlackbox2AuthModule>(*this, Client);
}

TBlackbox2AuthModule::TBlackbox2AuthModule(const TBlackbox2AuthConfig& config, TAtomicSharedPtr<NDrive::TBlackboxClient> client)
    : Config(config)
    , Client(client)
{
}

IAuthInfo::TPtr TBlackbox2AuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    if (!requestContext) {
        return MakeAtomicShared<TBlackboxAuthInfo>("null RequestContext", HTTP_NOT_IMPLEMENTED);
    }
    if (!Client) {
        return MakeAtomicShared<TBlackboxAuthInfo>("null Blackbox client", HTTP_NOT_IMPLEMENTED);
    }

    const TServerRequestData& rd = requestContext->GetRequestData();
    TStringBuf auth = rd.HeaderInOrEmpty(Config.GetAuthHeader());
    TStringBuf cookie = rd.HeaderInOrEmpty("Cookie");
    TStringBuf userIp = NUtil::GetClientIp(rd);

    TStringBuf sessionId;
    if (cookie) {
        THttpCookies cookies(cookie);
        sessionId = cookies.Get("Session_id");
    }

    NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> asyncResponse;
    TInstant start = Now();
    TString errorType;
    bool cookieAuth = false;
    switch (Config.GetAuthMethod()) {
    case TBlackbox2AuthConfig::EAuthMethod::Any:
        if (auth) {
            asyncResponse = MakeOAuthRequest(auth, userIp);
        } else if (sessionId) {
            cookieAuth = true;
            asyncResponse = MakeSessionIdRequest(sessionId, userIp);
        } else {
            errorType = "NoCredentials";
        }
        break;
    case TBlackbox2AuthConfig::EAuthMethod::Cookie:
        if (sessionId) {
            cookieAuth = true;
            asyncResponse = MakeSessionIdRequest(sessionId, userIp);
        } else {
            errorType = "NoSessionId";
        }
        break;
    case TBlackbox2AuthConfig::EAuthMethod::OAuth:
        if (auth) {
            asyncResponse = MakeOAuthRequest(auth, userIp);
        } else {
            errorType = "NoOAuthToken";
        }
        break;
    }
    if (!asyncResponse.Initialized()) {
        return MakeAtomicShared<TBlackboxAuthInfo>(errorType, HTTP_UNAUTHORIZED);
    }

    BlackboxRequest.Signal(1);
    if (!asyncResponse.Wait(requestContext->GetRequestDeadline())) {
        BlackboxTimeout.Signal(1);
        return MakeAtomicShared<TBlackboxAuthInfo>(TStringBuilder() << "WaitTimeout:" << (Now() - start).MicroSeconds(), HTTP_AUTHENTICATION_TIMEOUT);
    }
    TInstant finish = Now();
    TDuration duration = finish - start;
    BlackboxTimes.Signal(duration.MilliSeconds());

    if (!asyncResponse.HasValue()) {
        BlackboxException.Signal(1);
        try {
            asyncResponse.GetValue();
        } catch (const TCodedException& e) {
            return MakeAtomicShared<TBlackboxAuthInfo>(e.GetReport().GetStringRobust(), static_cast<ui32>(e.GetCode()));
        } catch (const std::exception& e) {
            return MakeAtomicShared<TBlackboxAuthInfo>("cannot check Auth: " + FormatExc(e), HTTP_INTERNAL_SERVER_ERROR);
        }
    }
    BlackboxOk.Signal(1);

    auto bbResponse = asyncResponse.ExtractValue();
    if (!bbResponse) {
        BlackboxNull.Signal(1);
        return MakeAtomicShared<TBlackboxAuthInfo>("NullSessionIdResponse", HTTP_INTERNAL_SERVER_ERROR);
    }

    auto sessionResponse = std::dynamic_pointer_cast<NBlackbox2::TSessionResp>(bbResponse);
    auto status = sessionResponse ? sessionResponse->Status() : NBlackbox2::TSessionResp::Valid;
    BlackboxStatus.Signal(status, 1);
    if (status != NBlackbox2::TSessionResp::Valid && status != NBlackbox2::TSessionResp::NeedReset) {
        const auto message = ToString(bbResponse->Message());
        const auto error = message ? message : (TStringBuilder() << "BadSessionIdResponse" << ' ' << static_cast<int>(status));
        NDrive::TEventLog::Log("BlackboxError", NJson::TMapBuilder
            ("type", "BadSessionIdResponse")
            ("message", message)
            ("status", static_cast<int>(status))
        );
        return MakeAtomicShared<TBlackboxAuthInfo>(error, HTTP_UNAUTHORIZED);
    }

    auto info = Client->Parse(*bbResponse);
    info.IgnoreDeviceId = Config.ShouldIgnoreDeviceId();
    info.Environment = Config.GetEnvironment();

    const auto& acceptedScopes = Config.GetAcceptedScopes();
    if (!cookieAuth && !acceptedScopes.empty()) {
        auto& scopes = info.Scopes;
        std::sort(scopes.begin(), scopes.end());
        if (IsIntersectionEmpty(scopes, acceptedScopes)) {
            NDrive::TEventLog::Log("BlackboxError", NJson::TMapBuilder
                ("type", "BadScopes")
                ("scopes", NJson::ToJson(scopes))
            );
            return MakeAtomicShared<TBlackboxAuthInfo>("none of the accepted scopes found", HTTP_UNAUTHORIZED);
        }
    }

    return MakeAtomicShared<TBlackboxAuthInfo>(std::move(info));
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> TBlackbox2AuthModule::MakeOAuthRequest(TStringBuf authorization, TStringBuf userIp) const {
    constexpr TStringBuf OAuthPrefix = "OAuth ";
    const TStringBuf token = StripString(authorization.SubStr(OAuthPrefix.size()));

    NAuthClientParser::TOAuthToken parser;
    if (!parser.Parse(token)) {
        return NThreading::TExceptionFuture<TCodedException>(HTTP_BAD_REQUEST) << "IllFormedToken";
    }

    return Checked(Client)->OAuthRequest(token, userIp);
}

NThreading::TFuture<NDrive::TBlackboxClient::TResponsePtr> TBlackbox2AuthModule::MakeSessionIdRequest(TStringBuf sessionId, TStringBuf userIp) const {
    NAuthClientParser::TZeroAllocationCookie parser;
    NAuthClientParser::EParseStatus status = parser.Parse(sessionId);
    if (status != NAuthClientParser::EParseStatus::RegularMayBeValid) {
        return NThreading::TExceptionFuture<TCodedException>(HTTP_BAD_REQUEST) << "IllFormedSessionId:" << static_cast<int>(status);
    }

    return Checked(Client)->SessionIdRequest(sessionId, userIp);
}

TBlackbox2AuthConfig::TFactory::TRegistrator<TBlackbox2AuthConfig> TBlackbox2AuthConfig::Registrator("blackbox2");
