#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/processor/processor.h>

#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/yconf/conf.h>

#include <util/generic/ptr.h>
#include <util/string/vector.h>

class IAuthModule;

class IAuthInfo {
public:
    using TPtr = TAtomicSharedPtr<IAuthInfo>;

public:
    virtual ~IAuthInfo() {}

    virtual bool IsAvailable() const = 0;
    virtual const TString& GetUid() const {
        return Default<TString>();
    }
    virtual const TString& GetUsername() const {
        return Default<TString>();
    }
    virtual const TString& GetUserId() const {
        return Default<TString>();
    }

    virtual const TString& GetEmail() const {
        return Default<TString>();
    }
    virtual const TString& GetPhone() const {
        return Default<TString>();
    }

    virtual const TString& GetMessage() const {
        return Default<TString>();
    }
    virtual NJson::TJsonValue GetInfo() const {
        return {};
    }
    virtual ui32 GetCode() const {
        return 0;
    }
    virtual const TString& GetEnvironment() const {
        return Default<TString>();
    }
    inline ui32 GetCode(ui32 fallback) const {
        auto code = GetCode();
        return code ? code : fallback;
    }
};

class IAuthModuleConfig {
private:
    const TString Name;
public:
    using TPtr = TAtomicSharedPtr<IAuthModuleConfig>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IAuthModuleConfig, TString, TString>;

    TString GetName() const {
        return Name;
    }

    IAuthModuleConfig(const TString& name)
        : Name(name)
    {
    }

    virtual THolder<IAuthModule> ConstructAuthModule(const IServerBase* server) const = 0;
    virtual void Init(const TYandexConfig::Section* section) = 0;
    virtual void ToString(IOutputStream& os) const = 0;
    virtual ~IAuthModuleConfig() {}

    virtual const TSet<TString>& GetRequiredModules() const {
        return Default<TSet<TString>>();
    }
};

class IAuthModule {
public:
    using TPtr = TAtomicSharedPtr<IAuthModule>;

    virtual ~IAuthModule() {}
    virtual IAuthInfo::TPtr RestoreAuthInfo(IReplyContext::TPtr requestContext) const = 0;
};

class IAuthRequestProcessorConfig: public IRequestProcessorConfig {
private:
    using TBase = IRequestProcessorConfig;
    TString AuthModuleName;

protected:
    virtual TString GetAuthModuleName() const;

    virtual IRequestProcessor::TPtr DoConstructProcessor(IReplyContext::TPtr context, const IServerBase* server) const override final;
    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const = 0;
    virtual void DoInit(const TYandexConfig::Section* section) override;

public:
    IAuthRequestProcessorConfig(const TString& name)
        : IRequestProcessorConfig(name) {}

    virtual void ToString(IOutputStream& os) const override;
};

class TAuthRequestProcessor: public IRequestProcessor {
protected:
    using IRequestProcessor::HandlerName;
    virtual void DoAuthProcess(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) = 0;
    virtual void DoProcess(TJsonReport::TGuard& g) override final;
    virtual void CheckAuthInfo(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo);

public:
    TAuthRequestProcessor(const IAuthRequestProcessorConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr auth, const IServerBase* server);

private:
    IAuthModule::TPtr Auth;
    const IAuthRequestProcessorConfig& AuthConfig;
};

template <class T>
class TStubRequestProcessorConfig : public IAuthRequestProcessorConfig {
public:
    using IAuthRequestProcessorConfig::IAuthRequestProcessorConfig;

protected:
    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override {
        return MakeIntrusive<T>(*this, context, authModule, server);
    }
};

template <class T>
using TSimpleProcessorRegistrator = IRequestProcessorConfig::TFactory::TRegistrator<TStubRequestProcessorConfig<T>>;
