#include "database.h"

#include <library/cpp/ipmath/ipmath.h>

#include <rtline/library/json/cast.h>

NStorage::TTableRecord TExternalAccessToken::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("token", Token);
    row.Set("user_id", UserId);
    NJson::TJsonValue value(NJson::EJsonValueType::JSON_ARRAY);
    for(const auto& node: IpWhiteList) {
        value.AppendValue(node);
    }
    row.Set("ip_whitelist", value.GetStringRobust());
    if (Scope) {
        row.Set("scope", Scope);
    }
    return row;
}


bool TExternalAccessToken::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

bool TExternalAccessToken::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Token);
    NJson::TJsonValue value;
    READ_DECODER_VALUE_JSON(decoder, values, value, IpWhiteList);
    IpWhiteList.reserve(value.GetArray().size());
    for(const auto& node:value.GetArray()) {
        IpWhiteList.emplace_back(node.GetString());
    }
    READ_DECODER_VALUE(decoder, values, UserId);
    if (decoder.GetScope() > 0) {
        READ_DECODER_VALUE(decoder, values, Scope);
    }
    return true;
}

bool TExternalAccessTokensDB::Upsert(const TEntity& entity, const TString& authorId, NDrive::TEntitySession& session) const {
    bool isUpdate = false;
    NStorage::TObjectRecordsSet<TEntity> upsertedData;
    if (!TBase::Upsert(entity, session, &upsertedData, &isUpdate)) {
        session.AddErrorMessage("external_access_token_upsert", "cannot upsert");
        return false;
    }
    TRecordsSet addedHistory;
    if (!HistoryWriter.AddHistory<NStorage::TObjectRecordsSet<TEntity>>(upsertedData, authorId, EObjectHistoryAction::UpdateData, session, &addedHistory)) {
        session.AddErrorMessage("external_access_tokens_history", "unable to add history");
        return false;
    }
    if (addedHistory.size() != 1) {
        session.SetErrorInfo("external_access_tokens_history",
                            Sprintf("addedHistory.size() = %lu, expected to be 1", addedHistory.size()), EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

bool TExternalAccessTokensDB::Remove(const TString& token, const TString& authorId, NDrive::TEntitySession& session) const {
    NStorage::TObjectRecordsSet<TEntity> removedData;
    if (!TBase::Remove(token, session, &removedData)) {
        session.AddErrorMessage("external_access_tokens_remove", "cannot remove");
        return false;
    }
    TRecordsSet addedHistory;
    if (!HistoryWriter.AddHistory<NStorage::TObjectRecordsSet<TEntity>>(removedData, authorId, EObjectHistoryAction::Remove, session, &addedHistory)) {
        session.AddErrorMessage("external_access_tokens_history", "unable to add history");
        return false;
    }
    return true;
}

TExternalAccessTokensManager::TExternalAccessTokensManager(const IHistoryContext& context)
    : TokensDB(context)
{

}

TExpected<TString, IExternalAccessTokensManager::EValidationError> TExternalAccessTokensManager::Validate(const TString& token,
                                                                             const TString& ipAddressStr, const TString& scope) const {
    bool isSuccess;
    TIpv6Address address = TIpv6Address::FromString(ipAddressStr, isSuccess);
    if (!isSuccess) {
        return MakeUnexpected(EValidationError::IncorrectIp);
    }
    auto fetchResult = TokensDB.FetchInfo(token);
    if (fetchResult.empty()) {
        return MakeUnexpected(EValidationError::NotAllowed);
    }
    Y_ENSURE(fetchResult.size() == 1, "Unexpected multiple records for single token");
    TExternalAccessToken accessToken = std::move(fetchResult.begin()->second);
    if (accessToken.GetScope() && accessToken.GetScope() != scope) {
        return MakeUnexpected(EValidationError::IncorrectScope);
    }
    TVector<TString> ipWhiteList = accessToken.GetIpWhiteList();
    for (const NJson::TJsonValue& value : ipWhiteList) {
        TIpAddressRange range = TIpAddressRange::FromString(value.GetString());
        if (range.Contains(address)) {
            return accessToken.GetUserId();
        }
    }
    return MakeUnexpected(EValidationError::NotAllowed);
}

bool TExternalAccessTokensManager::Upsert(const TString& token, const TVector<TString>& ipWhiteList,
                                    const TString& userId, const TString& scope, const TString& authorId) const {
    TExternalAccessToken accessToken;
    accessToken.SetToken(token);
    accessToken.SetUserId(userId);
    accessToken.SetIpWhiteList(ipWhiteList);
    accessToken.SetScope(scope);

    auto session = TokensDB.BuildSession();
    if (!TokensDB.Upsert(accessToken, authorId, session)) {
        ERROR_LOG << "Error while adding token " << Endl;
        return false;
    }
    if (!session.Commit()) {
        ERROR_LOG << "Cannot commit upsert: " << session.GetStringReport() << Endl;
    }
    return true;
}

bool TExternalAccessTokensManager::Invalidate(const TString& token, const TString& authorId) const {
    auto session = TokensDB.BuildSession();
    if (!TokensDB.Remove(token, authorId, session)) {
        ERROR_LOG << "Unable to remove token" << Endl;
        return false;
    }
    if (!session.Commit()) {
        ERROR_LOG << "Cannot commit invalidation: " << session.GetStringReport() << Endl;
        return false;
    }
    return true;
}
