#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/common.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/history/manager.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/expected.h>

#include <util/generic/string.h>
#include <util/generic/vector.h>


class TExternalAccessToken {
    R_FIELD(TString, Token);
    R_FIELD(TVector<TString>, IpWhiteList);
    R_FIELD(TString, UserId);
    R_FIELD(TString, Scope);

public:
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, Token, -1);
        R_FIELD(i32, IpWhiteList, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, Scope, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Token = GetFieldDecodeIndex("token", decoderBase);
            IpWhiteList = GetFieldDecodeIndex("ip_whitelist", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            Scope = GetFieldDecodeIndex("scope", decoderBase);
        }
    };

    bool Parse(const NStorage::TTableRecord& row);

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TExternalAccessTokensHistoryManager : public TDatabaseHistoryManager<TExternalAccessToken> {
private:
    using TBase = TDatabaseHistoryManager<TExternalAccessToken>;

public:
    TExternalAccessTokensHistoryManager(const IHistoryContext& context)
        : TBase(context, "external_access_tokens_history")
    {
    }
};

class TExternalAccessTokensDB : public TDBEntities<TExternalAccessToken> {
private:
    using TEntity = TExternalAccessToken;
    using TBase = TDBEntities<TEntity>;

public:
    TExternalAccessTokensDB(const IHistoryContext& context)
        : TBase(context.GetDatabase())
        , HistoryWriter(context)
    {
    }

    TString GetTableName() const override {
        return "external_access_tokens";
    }

    TString GetColumnName() const override {
        return "token";
    }

    TString GetMainId(const TEntity& entity) const override {
        return entity.GetToken();
    }

    bool Upsert(const TEntity& entity, const TString& authorId, NDrive::TEntitySession& session) const;
    bool Remove(const TString& token, const TString& authorId, NDrive::TEntitySession& session) const;

private:
    TExternalAccessTokensHistoryManager HistoryWriter;
};

class IExternalAccessTokensManager {
public:
    enum class EValidationError {
        IncorrectIp /* "incorrect_ip" */,
        NotAllowed /* "not_allowed" */,
        IncorrectScope /* "incorrect_scope" */
    };

    virtual ~IExternalAccessTokensManager() {}
    virtual TExpected<TString, EValidationError> Validate(const TString& token, const TString& ipAddress, const TString& scope = "") const = 0;
    virtual bool Upsert(const TString& token, const TVector<TString>& ipWhiteList, const TString& userId, const TString& scope, const TString& authorId) const = 0;
    virtual bool Invalidate(const TString& token, const TString& authorId) const = 0;
};

class TExternalAccessTokensManager : public IExternalAccessTokensManager {
public:
    TExternalAccessTokensManager(const IHistoryContext& context);
    TExpected<TString, EValidationError> Validate(const TString& token, const TString& ipAddress, const TString& scope = "") const override;
    bool Upsert(const TString& token, const TVector<TString>& ipWhiteList, const TString& userId, const TString& scope, const TString& authorId) const override;
    bool Invalidate(const TString& token, const TString& authorId) const override;

private:
    const TExternalAccessTokensDB TokensDB;
};

class TExternalAccessTokensManagerConfig : public TDBEntitiesManagerConfig {
public:
    using TDBEntitiesManagerConfig::TDBEntitiesManagerConfig;
};
