#include "external.h"

#include <drive/library/cpp/network/data/data.h>

TExternalAuthInfo::TExternalAuthInfo(TMaybe<TString> maybeAuthId, NJson::TJsonValue info)
    : MaybeAuthId(std::move(maybeAuthId))
    , Info(std::move(info))
{
}

bool TExternalAuthInfo::IsAvailable() const {
    return !!MaybeAuthId;
}

const TString& TExternalAuthInfo::GetUserId() const {
    return MaybeAuthId.GetOrElse(Default<TString>());
}

ui32 TExternalAuthInfo::GetCode() const {
    return IsAvailable();
}

NJson::TJsonValue TExternalAuthInfo::GetInfo() const {
    return Info;
}

TExternalAuthModule::TExternalAuthModule(const IExternalAccessTokensManager* tokensManager, const TString& scope)
    : TokensManager(tokensManager)
    , Scope(scope)
{
}

IAuthInfo::TPtr TExternalAuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    auto ipAddress = NUtil::GetClientIp(requestContext->GetRequestData());

    TStringBuf token = requestContext->GetRequestData().HeaderInOrEmpty("Authorization");
    constexpr TStringBuf necessaryPrefix = "Bearer ";
    if (!token.StartsWith(necessaryPrefix)) {
        return new TExternalAuthInfo(NothingObject, NJson::TMapBuilder("error_message", "Incorrect Authorization header format"));
    }
    token.Skip(necessaryPrefix.size());
    if (!TokensManager) {
        return new TExternalAuthInfo(NothingObject, NJson::TMapBuilder("error_message", "External access token manager is inaccessible"));
    }
    auto result = TokensManager->Validate(TString{token}, TString{ipAddress}, Scope);
    if (result) {
        return new TExternalAuthInfo(result.GetValue(), NJson::TJsonValue(NJson::EJsonValueType::JSON_MAP));
    } else {
        return new TExternalAuthInfo(NothingObject, NJson::TMapBuilder("error_message", "Invalid credentials")
                                                           ("token", token)
                                                           ("ip_address", ipAddress)
                                                           ("validation_error", ToString(result.GetError())));
    }
}

THolder<IAuthModule> TExternalAuthConfig::ConstructAuthModule(const IServerBase* server) const {
    return MakeHolder<TExternalAuthModule>(server->GetExternalAccessTokensManager(), Scope);
}

void TExternalAuthConfig::ToString(IOutputStream& os) const {
    os << "Scope: " << Scope << Endl;
}

void TExternalAuthConfig::Init(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    Scope = directives.Value("Scope", Scope);
}

TExternalAuthConfig::TFactory::TRegistrator<TExternalAuthConfig> TExternalAuthConfig::Registrator("external_access_token");

