#pragma once

#include <drive/backend/auth/common/auth.h>

class TFakeAuthInfo: public IAuthInfo {
private:
    const TString AuthId;
    bool IsAuthIdUserId;

public:
    TFakeAuthInfo(const TString& authId, bool isAuthIdUserId = true)
        : AuthId(authId), IsAuthIdUserId(isAuthIdUserId)
    {
    }

    virtual bool IsAvailable() const override {
        return true;
    }

    virtual const TString& GetUserId() const override {
        if (IsAuthIdUserId) {
            return AuthId;
        } else {
            return IAuthInfo::GetUserId();
        }
    }

    virtual const TString& GetUid() const override {
        if (IsAuthIdUserId) {
            return IAuthInfo::GetUid();
        } else {
            return AuthId;
        }
    }
};

class TFakeAuthConfig;

class TFakeAuthModule: public IAuthModule {
private:
    const TFakeAuthConfig* Config = nullptr;

public:
    TFakeAuthModule(const TFakeAuthConfig* config)
        : Config(config)
    {
    }

    virtual IAuthInfo::TPtr RestoreAuthInfo(IReplyContext::TPtr requestContext) const;
};

class TFakeAuthConfig: public IAuthModuleConfig {
private:
    TString DefaultUserId;
    TMaybe<bool> IsAuthIdUserId;

    static TFactory::TRegistrator<TFakeAuthConfig> Registrator;

public:
    TFakeAuthConfig(const TString& name)
        : IAuthModuleConfig(name)
    {
    }

    const TString& GetDefaultUserId() const {
        return DefaultUserId;
    }

    bool GetIsAuthIdUserId() const {
        return IsAuthIdUserId.GetOrElse(true);
    }

    virtual THolder<IAuthModule> ConstructAuthModule(const IServerBase* server) const override {
        if (server->GetSettings().GetValue<bool>("authorization.fake.enabled").GetOrElse(true)) {
            return MakeHolder<TFakeAuthModule>(this);
        } else {
            return nullptr;
        }
    }

    virtual void Init(const TYandexConfig::Section* section) override {
        CHECK_WITH_LOG(section);
        const auto& directives = section->GetDirectives();
        DefaultUserId = directives.Value("DefaultUserId", DefaultUserId);
        if (auto isAuthIdUserIdIt = directives.find("IsAuthIdUserId"); directives.end() != isAuthIdUserIdIt) {
            IsAuthIdUserId = FromString<bool>(isAuthIdUserIdIt->second);
        }
    }

    virtual void ToString(IOutputStream& os) const override {
        if (DefaultUserId) {
            os << "DefaultUserId: " << DefaultUserId << Endl;
        }
        if (!IsAuthIdUserId.Empty()) {
            os << "IsAuthIdUserId: " << ::ToString(*IsAuthIdUserId) << Endl;
        }
    }
};
