#include "meta.h"

#include <drive/backend/logging/evlog.h>

#include <rtline/library/json/exception.h>

#include <util/string/split.h>

NJson::TJsonValue TExceptionAuthInfo::GetInfo() const {
    return NJson::GetExceptionInfo(Exception);
}

TMetaAuthInfo::TMetaAuthInfo(TInfos&& infos)
    : Infos(std::move(infos))
{
    Message = GetMessageImpl();
}

bool TMetaAuthInfo::IsAvailable() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return true;
        }
    }
    return false;
}

const TString& TMetaAuthInfo::GetEmail() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetEmail();
        }
    }
    for (auto&& [name, info] : Infos) {
        if (info) {
            return info->GetEmail();
        }
    }
    return Default<TString>();
}

const TString& TMetaAuthInfo::GetPhone() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetPhone();
        }
    }
    for (auto&& [name, info] : Infos) {
        if (info) {
            return info->GetPhone();
        }
    }
    return Default<TString>();
}

const TString& TMetaAuthInfo::GetUid() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetUid();
        }
    }
    for (auto&& [name, info] : Infos) {
        if (info) {
            return info->GetUid();
        }
    }
    return Default<TString>();
}

const TString& TMetaAuthInfo::GetUsername() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetUsername();
        }
    }
    for (auto&& [name, info] : Infos) {
        if (info) {
            return info->GetUsername();
        }
    }
    return Default<TString>();
}

const TString& TMetaAuthInfo::GetUserId() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetUserId();
        }
    }
    for (auto&& [name, info] : Infos) {
        const auto& userId = info ? info->GetUserId() : Default<TString>();
        if (userId) {
            return userId;
        }
    }
    return Default<TString>();
}

const TString& TMetaAuthInfo::GetMessage() const {
    return Message;
}

TString TMetaAuthInfo::GetMessageImpl() const {
    TStringBuilder result;
    for (auto&& [name, info] : Infos) {
        if (!result.empty()) {
            result << ", ";
        }
        result << name << ": " << (info ? info->GetMessage() : "null");
    }
    return result;
}

NJson::TJsonValue TMetaAuthInfo::GetInfo() const {
    NJson::TJsonValue result;
    for (auto&& [name, info] : Infos) {
        result.InsertValue(name, info ? info->GetInfo() : NJson::JSON_NULL);
    }
    return result;
}

ui32 TMetaAuthInfo::GetCode() const {
    for (auto&& [name, info] : Infos) {
        if (info && info->IsAvailable()) {
            return info->GetCode();
        }
    }
    for (auto&& [name, info] : Infos) {
        if (info) {
            return info->GetCode();
        }
    }
    return 0;
}

IAuthInfo::TPtr TMetaAuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    TMetaAuthInfo::TInfos infos;
    for (auto&& [name, module] : Submodules) {
        auto guard = NDrive::BuildEventGuard("RestoreAuthInfo:" + name);
        if (!module) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "NullAuthModule")
                    ("name", name)
                );
            } else {
                ERROR_LOG << "null AuthModule: " << name << Endl;
            }
            continue;
        }
        IAuthInfo::TPtr info;
        try {
            info = module->RestoreAuthInfo(requestContext);
        } catch (const std::exception& e) {
            info = MakeAtomicShared<TExceptionAuthInfo>(
                std::current_exception(),
                TString{e.what()}
            );
        }
        if (!info) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "NullAuthInfo")
                    ("name", name)
                );
            }
            continue;
        }
        if (!info->IsAvailable()) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "UnavailableAuthInfo")
                    ("name", name)
                    ("info", info->GetInfo())
                );
            }
            infos.emplace(name, std::move(info));
            continue;
        }
        return info;
    }
    return MakeAtomicShared<TMetaAuthInfo>(std::move(infos));
}

THolder<IAuthModule> TMetaAuthConfig::ConstructAuthModule(const IServerBase* server) const {
    TMetaAuthModule::TSubmodules submodules;
    for (auto&& name : Submodules) {
        if (name == GetName()) {
            ERROR_LOG << "self-inclusion in MetaAuthConfig: " << GetName() << Endl;
            continue;
        }

        const IAuthModuleConfig* config = server ? server->GetAuthModuleInfo(name) : nullptr;
        if (!config) {
            ERROR_LOG << "null AuthModuleConfig: " << name << Endl;
            continue;
        }
        submodules.emplace_back(name, config->ConstructAuthModule(server));
    }
    return MakeHolder<TMetaAuthModule>(std::move(submodules));
}

void TMetaAuthConfig::Init(const TYandexConfig::Section* section) {
    CHECK_WITH_LOG(section);
    const auto& directives = section->GetDirectives();
    Submodules = StringSplitter(directives.Value("Submodules", JoinStrings(Submodules, ",")))
        .SplitBySet(", ")
        .SkipEmpty();
}

void TMetaAuthConfig::ToString(IOutputStream& os) const {
    os << "Submodules: " << JoinStrings(Submodules, ",") << Endl;
}

TMetaAuthConfig::TFactory::TRegistrator<TMetaAuthConfig> TMetaAuthConfig::Registrator("meta");
