#include "public_key.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/head/head_account.h>

#include <drive/library/cpp/openssl/rsa.h>

TPublicKeyAuthModule::TPublicKeyAuthModule(const NDrive::IServer* server, const TPublicKeyAuthConfig& config)
    : DriveApi(server->GetDriveAPI())
    , Config(config)
{
    CHECK_WITH_LOG(DriveApi);
}

IAuthInfo::TPtr TPublicKeyAuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    TString sessionKey(requestContext->GetRequestData().HeaderInOrEmpty("Session-Key"));
    TString userId(requestContext->GetRequestData().HeaderInOrEmpty("UserId"));

    if (!userId) {
        userId = requestContext->GetRequestData().HeaderInOrEmpty("DeviceId");
    }

    if (!userId || !sessionKey) {
        DEBUG_LOG << "No UserId or Session-Key provided" << Endl;
    }

    sessionKey = Base64Decode(sessionKey);

    const TShortSessionsManager& manager = DriveApi->GetShortSessionsManager();
    auto sessionInfo = manager.GetUserSession(userId);

    if (!sessionInfo.Defined()) {
        sessionInfo = manager.GetUserSession("default");
    }

    if (!sessionInfo.Defined() || sessionInfo->CheckExpired(TInstant::Now())) {
        return new TPublicKeyAuthInfo("session_expired");
    }

    TString publicKey = manager.GetPublicKey(userId);
    if (!publicKey) {
        return new TPublicKeyAuthInfo("no_auth_data");
    }

    NOpenssl::TRSAPublicKey verifier;
    if (!verifier.Init(publicKey)) {
        return new TPublicKeyAuthInfo("incorrect_public_key");
    }

    if (!verifier.VerifySHA(userId + " " + sessionInfo->GetSessionKey(), sessionKey)) {
        return new TPublicKeyAuthInfo("verify_failed");
    }
    if (Config.GetExternalUserId()) {
        return new TPublicKeyAuthInfo(true, Config.GetExternalUserId(), false);
    }
    return new TPublicKeyAuthInfo(true, userId, true);
}

THolder<IAuthModule> TPublicKeyAuthConfig::ConstructAuthModule(const IServerBase* server) const {
    return MakeHolder<TPublicKeyAuthModule>(&server->GetAsSafe<NDrive::IServer>(), *this);
}

TPublicKeyAuthConfig::TFactory::TRegistrator<TPublicKeyAuthConfig> TPublicKeyAuthConfig::Registrator("public_key");
