#pragma once

#include <drive/backend/auth/common/auth.h>

#include <rtline/util/types/accessor.h>

class TDriveAPI;

namespace NDrive {
    class IServer;
}

class TPublicKeyAuthInfo: public IAuthInfo {
public:
    R_READONLY(bool, Authorized, false);
    R_READONLY(bool, RemapUser, true);

public:
    TPublicKeyAuthInfo(const TString& message = "")
        : Message(message)
    {}

    TPublicKeyAuthInfo(const bool authorized, const TString& userId, bool remapUser)
        : Authorized(authorized)
        , RemapUser(remapUser)
        , UserId(userId)
    {
    }

    virtual bool IsAvailable() const override {
        return Authorized;
    }

    virtual NJson::TJsonValue GetInfo() const override {
        NJson::TJsonValue result;
        result["type"] = "public_key_auth";
        if (Message) {
            result["message"] = Message;
        }
        if (RemapUser) {
            result["remap_user"] = RemapUser;
        }
        if (UserId) {
            result["user_id"] = UserId;
        }
        return result;
    }

    virtual const TString& GetMessage() const override {
        return Message;
    }

    virtual const TString& GetUserId() const override {
        return UserId;
    }

private:
    TString Message;
    TString UserId;
};

class TPublicKeyAuthConfig : public IAuthModuleConfig {
    R_READONLY(TString, ExternalUserId);
public:
    TPublicKeyAuthConfig(const TString& name)
        : IAuthModuleConfig(name) {}

    virtual THolder<IAuthModule> ConstructAuthModule(const IServerBase* server) const override;

    virtual void Init(const TYandexConfig::Section* section) override {
        ExternalUserId = section->GetDirectives().Value("ExternalUserId", ExternalUserId);
    }

    virtual void ToString(IOutputStream& os) const override {
        os << "ExternalUserId: " << ExternalUserId << Endl;
    }

private:
    static TFactory::TRegistrator<TPublicKeyAuthConfig> Registrator;
};

class TPublicKeyAuthModule: public IAuthModule {
public:
    TPublicKeyAuthModule(const NDrive::IServer* server, const TPublicKeyAuthConfig& config);

    virtual IAuthInfo::TPtr RestoreAuthInfo(IReplyContext::TPtr requestContext) const override;

private:
    const TDriveAPI* DriveApi;
    TPublicKeyAuthConfig Config;
};
