#include "tvm.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/blackbox/client.h>
#include <drive/library/cpp/network/data/data.h>

NJson::TJsonValue TTvmAuthInfo::GetInfo() const {
    NJson::TJsonValue result;
    if (UserId) {
        result["user_id"] = UserId;
    }
    if (BlackboxInfo.PassportUid) {
        result["passport_uid"] = BlackboxInfo.PassportUid;
    }
    if (BlackboxInfo.Login) {
        result["login"] = BlackboxInfo.Login;
    }
    if (BlackboxInfo.IsPlusUser) {
        result["is_plus"] = BlackboxInfo.IsPlusUser;
    }
    if (BlackboxInfo.IsYandexoid) {
        result["is_yandexoid"] = BlackboxInfo.IsYandexoid;
    }
    return result;
}

TTvmAuthModule::TTvmAuthModule(TAtomicSharedPtr<NDrive::TBlackboxClient> blackbox, TAtomicSharedPtr<NTvmAuth::TTvmClient> client, const TTvmAuthConfig& config)
    : Blackbox(blackbox)
    , Client(client)
    , AcceptedClientIds(config.GetAcceptedClientIds())
    , Name(config.GetName())
    , UserId(config.GetUserId())
{
}

IAuthInfo::TPtr TTvmAuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    if (!requestContext) {
        return nullptr;
    }
    const auto& rd = requestContext->GetRequestData();
    const auto& headers = rd.HeadersIn();
    const auto p = headers.find("X-Ya-Service-Ticket");
    if (p == headers.end()) {
        return MakeAtomicShared<TTvmAuthInfo>("header X-Ya-Service-Ticket is missing");
    }
    const TString& serializedTicket = p->second;

    const auto up = headers.find("X-Ya-User-Ticket");
    const TStringBuf serializedUserTicket = up != headers.end() ? up->second : TStringBuf();
    try {
        if (!Client) {
            return MakeAtomicShared<TTvmAuthInfo>("TVM client is missing");
        }
        const NTvmAuth::TCheckedServiceTicket serviceTicket = Client->CheckServiceTicket(serializedTicket);
        if (serviceTicket.GetStatus() != NTvmAuth::ETicketStatus::Ok) {
            return MakeAtomicShared<TTvmAuthInfo>(
                TStringBuilder() << "incorrect service ticket status: " << NTvmAuth::StatusToString(serviceTicket.GetStatus())
            );
        }
        NTvmAuth::TTvmId clientId = serviceTicket.GetSrc();
        if (!AcceptedClientIds.contains(clientId)) {
            return MakeAtomicShared<TTvmAuthInfo>("ClientId " + ToString(clientId) + " is forbidden");
        }

        NDrive::TBlackboxInfo blackboxInfo;
        const auto uid = headers.find("X-Uid");
        if (serializedUserTicket) {
            NTvmAuth::TCheckedUserTicket userTicket = Client->CheckUserTicket(serializedUserTicket);
            if (userTicket.GetStatus() != NTvmAuth::ETicketStatus::Ok) {
                return MakeAtomicShared<TTvmAuthInfo>(
                    TStringBuilder() << "incorrect user ticket status: " << NTvmAuth::StatusToString(userTicket.GetStatus())
                );
            }
            blackboxInfo.PassportUid = ToString(userTicket.GetDefaultUid());
        } else if (uid != headers.end()) {
            blackboxInfo.PassportUid = uid->second;
        }

        const auto login = headers.find("X-Yandex-Login");
        if (login != headers.end()) {
            blackboxInfo.Login = login->second;
        }
        const auto taxiFlags = headers.find("X-YaTaxi-Pass-Flags");
        if (taxiFlags != headers.end() && taxiFlags->second.find("ya-plus") != TString::npos) {
            blackboxInfo.IsPlusUser = true;
        }

        if (blackboxInfo.PassportUid) {
            if (Blackbox) {
                auto userIp = NUtil::GetClientIp(rd);
                auto asyncBlackboxResponse = Blackbox->UidInfoRequest(blackboxInfo.PassportUid, userIp);
                Y_ENSURE(asyncBlackboxResponse.Wait(requestContext->GetRequestDeadline()), "blackbox UidInfoRequest wait timeout");
                auto blackboxResponse = asyncBlackboxResponse.GetValue();
                if (blackboxResponse) {
                    blackboxInfo = Blackbox->Parse(*blackboxResponse);
                } else {
                    NDrive::TEventLog::Log("BlackboxError", NJson::TMapBuilder
                        ("type", "NullSessionIdResponse")
                    );
                }
            }
            return MakeAtomicShared<TTvmAuthInfo>(true, std::move(blackboxInfo));
        } else if (UserId) {
            return MakeAtomicShared<TTvmAuthInfo>(true, UserId);
        } else {
            return MakeAtomicShared<TTvmAuthInfo>("neither UserTicket nor default user are provided");
        }
    } catch (const TCodedException& e) {
        return MakeAtomicShared<TTvmAuthInfo>(
            e.GetReport().GetStringRobust(),
            static_cast<ui32>(e.GetCode())
        );
    } catch (const std::exception& e) {
        return MakeAtomicShared<TTvmAuthInfo>(
            "cannot check ServiceTicket: " + FormatExc(e),
            HTTP_INTERNAL_SERVER_ERROR
        );
    }
}

THolder<IAuthModule> TTvmAuthConfig::ConstructAuthModule(const IServerBase* server) const {
    Y_ENSURE(server);
    auto client = server->GetTvmClient(SelfClientId, BlackboxEnv);
    Y_ENSURE(client, "cannot GetTvmClient for " << SelfClientId << "/" << BlackboxEnv);
    if (BlackboxUrl && !Blackbox) {
        auto tvm = BlackboxClientId ? server->GetTvmClient(BlackboxClientId) : client;
        auto blackbox = MakeAtomicShared<NDrive::TBlackboxClient>(BlackboxUrl, tvm);
        auto guard = Guard(BlackboxLock);
        Blackbox = std::move(blackbox);
    }
    return MakeHolder<TTvmAuthModule>(Blackbox, client, *this);
}

void TTvmAuthConfig::Init(const TYandexConfig::Section* section) {
    if (!section) {
        WARNING_LOG << "nullptr YandexConfig section" << Endl;
        return;
    }

    const auto& directives = section->GetDirectives();
    {
        StringSplitter(directives.Value("AcceptedClientIds", JoinStrings(AcceptedClientIds.begin(), AcceptedClientIds.end(), ","))).Split(',').SkipEmpty().ParseInto(&AcceptedClientIds);
        SelfClientId = directives.Value("SelfClientId", SelfClientId);
        BlackboxClientId = directives.Value("BlackboxClientId", BlackboxClientId);
        BlackboxEnv = directives.Value("BlackboxEnv", BlackboxEnv);
        BlackboxUrl = directives.Value("BlackboxUrl", BlackboxUrl);
        UserId = directives.Value("UserId", UserId);
    }
}

void TTvmAuthConfig::ToString(IOutputStream& os) const {
    os << "AcceptedClientIds: " << JoinStrings(AcceptedClientIds.begin(), AcceptedClientIds.end(), ",") << Endl;
    os << "SelfClientId: " << SelfClientId << Endl;
    os << "BlackboxClientId: " << BlackboxClientId << Endl;
    os << "BlackboxEnv:" << BlackboxEnv << Endl;
    os << "BlackboxUrl:" << BlackboxUrl << Endl;
    os << "UserId: " << UserId << Endl;
}

TTvmAuthConfig::TFactory::TRegistrator<TTvmAuthConfig> TTvmAuthConfig::Registrator("tvm");
