#pragma once

#include <drive/backend/auth/common/auth.h>
#include <drive/library/cpp/blackbox/info.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/mutex.h>

namespace NDrive {
    class TBlackboxClient;
}

class TTvmAuthInfo: public IAuthInfo {
private:
    const bool Available;
    const TString UserId;
    const TString ErrorMessage;
    const NDrive::TBlackboxInfo BlackboxInfo;

    ui32 Code = 0;

public:
    TTvmAuthInfo(bool available, const TString& clientId)
        : Available(available)
        , UserId(clientId)
    {
    }
    TTvmAuthInfo(bool available, NDrive::TBlackboxInfo&& blackboxInfo)
        : Available(available)
        , BlackboxInfo(std::move(blackboxInfo))
    {
    }
    TTvmAuthInfo(const TString& errorMessage, ui32 code = 0)
        : Available(false)
        , ErrorMessage(errorMessage)
        , Code(code)
    {
    }

    virtual NJson::TJsonValue GetInfo() const override;

    virtual ui32 GetCode() const override {
        return Code;
    }

    virtual bool IsAvailable() const override {
        return Available;
    }

    virtual const TString& GetMessage() const override {
        return ErrorMessage;
    }

    virtual const TString& GetUid() const override {
        return GetPassportUid();
    }
    virtual const TString& GetUsername() const override {
        return GetLogin();
    }
    virtual const TString& GetUserId() const override {
        return UserId;
    }

    virtual const TString& GetEmail() const override {
        return GetDefaultEmail();
    }
    virtual const TString& GetPhone() const override {
        return GetDefaultPhone();
    }

    const NDrive::TBlackboxInfo& GetBlackboxInfo() const {
        return BlackboxInfo;
    }
    const TString& GetPassportUid() const {
        return BlackboxInfo.PassportUid;
    }
    const TString& GetDefaultPhone() const {
        return BlackboxInfo.DefaultPhone;
    }
    const TString& GetDefaultEmail() const {
        return BlackboxInfo.DefaultEmail;
    }
    const TString& GetLogin() const {
        return BlackboxInfo.Login;
    }
};

class TTvmAuthConfig;

class TTvmAuthModule: public IAuthModule {
public:
    using TClientId = NTvmAuth::TTvmId;
    using TClientIds = TSet<TClientId>;

public:
    TTvmAuthModule(TAtomicSharedPtr<NDrive::TBlackboxClient> blackbox, TAtomicSharedPtr<NTvmAuth::TTvmClient> client, const TTvmAuthConfig& config);

    virtual IAuthInfo::TPtr RestoreAuthInfo(IReplyContext::TPtr requestContext) const override;

private:
    TAtomicSharedPtr<NDrive::TBlackboxClient> Blackbox;
    TAtomicSharedPtr<NTvmAuth::TTvmClient> Client;
    TClientIds AcceptedClientIds;
    TString Name;
    TString UserId;
};

class TTvmAuthConfig: public IAuthModuleConfig {
public:
    using TClientId = TTvmAuthModule::TClientId;
    using TClientIds = TTvmAuthModule::TClientIds;

public:
    using IAuthModuleConfig::IAuthModuleConfig;

    const TClientIds& GetAcceptedClientIds() const {
        return AcceptedClientIds;
    }
    TClientId GetSelfClientId() const {
        return SelfClientId;
    }
    const TString& GetUserId() const {
        return UserId;
    }

    virtual THolder<IAuthModule> ConstructAuthModule(const IServerBase* server) const override;

    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;

private:
    TClientIds AcceptedClientIds;
    TClientId SelfClientId = 0;
    TClientId BlackboxClientId = 0;
    NTvmAuth::EBlackboxEnv BlackboxEnv = NTvmAuth::EBlackboxEnv::Prod;
    TString BlackboxUrl;
    TString UserId;

    mutable TAtomicSharedPtr<NDrive::TBlackboxClient> Blackbox;
    TMutex BlackboxLock;

private:
    static TFactory::TRegistrator<TTvmAuthConfig> Registrator;
};
