#include "yang.h"

#include <util/stream/file.h>

TYangAuthModule::TYangAuthModule(const TYangAuthConfig& config, TAtomicSharedPtr<TYangClient> yangClient)
    : DefaultUserId(config.GetDefaultUserId())
    , CheckIsActivePair(config.GetCheckIsActivePair())
    , YangClient(yangClient) {
}

IAuthInfo::TPtr TYangAuthModule::RestoreAuthInfo(IReplyContext::TPtr requestContext) const {
    if (!requestContext) {
        return nullptr;
    }

    auto guard = NDrive::BuildEventGuard("RestoreYangAuthInfo");

    TString assignmentId = "";
    TString secretId = "";
    if (!requestContext->GetCgiParameters().Has("assignmentId") || !requestContext->GetCgiParameters().Has("secretId")) {
        TBlob postData = requestContext->GetBuf();
        TMemoryInput inp(postData.Data(), postData.Size());
        NJson::TJsonValue jsonValue(NJson::JSON_NULL);
        if (!NJson::ReadJsonTree(&inp, &jsonValue)) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "InputReadJsonTree")
                    ("data", postData.AsStringBuf())
                );
            } else {
                ERROR_LOG << "cannot parse json post" << postData.AsStringBuf() << Endl;
            }
            return nullptr;
        }
        if (!jsonValue.Has("assignmentId") || !jsonValue.Has("secretId") || !jsonValue["assignmentId"].IsString() || !jsonValue["secretId"].IsString()) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "IncorrectInputData")
                    ("data", jsonValue)
                );
            } else {
                ERROR_LOG << "incorrect input post data" << jsonValue.GetStringRobust() << Endl;
            }
            return nullptr;
        }
        assignmentId = jsonValue["assignmentId"].GetString();
        secretId = jsonValue["secretId"].GetString();
    } else {
        assignmentId = requestContext->GetCgiParameters().find("assignmentId")->second;
        secretId = requestContext->GetCgiParameters().find("secretId")->second;
    }

    if (assignmentId == "" || secretId == "") {
        if (guard) {
            guard->AddEvent(NJson::TMapBuilder
                ("event", "EmptyInputData")
                ("assignmentId", assignmentId)
                ("secretId", secretId)
            );
        } else {
            ERROR_LOG << "empty input data assignmentId=" << assignmentId << " secretId=" << secretId << Endl;
        }
        return nullptr;
    }

    if (!CheckIsActivePair) {
        return MakeAtomicShared<TYangAuthInfo>(true, DefaultUserId);
    }

    try {
        NJson::TJsonValue jsonContent = YangClient->RequestAssignmentSync(assignmentId);

        bool isActive = (jsonContent.Has("status") && jsonContent["status"].IsString() && jsonContent["status"].GetString() == "ACTIVE");
        if (!isActive) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "ParseStatus")
                    ("assignmentId", assignmentId)
                    ("secretId", secretId)
                    ("status", jsonContent["status"])
                );
            } else {
                ERROR_LOG << "non-active pair for assignmentId=" << assignmentId << " secretId=" << secretId << " status=" << jsonContent["status"].GetStringRobust() << Endl;
            }
            return nullptr;
        }

        bool isSecretIdValid = false;
        if (!jsonContent.Has("tasks") || !jsonContent["tasks"].IsArray()) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "ParseTasks")
                    ("assignmentId", assignmentId)
                    ("secretId", secretId)
                    ("tasks", jsonContent["tasks"])
                );
            } else {
                ERROR_LOG << "no 'tasks' array in response for assignmentId=" << assignmentId << " secretId=" << secretId << " tasks=" << jsonContent["tasks"].GetStringRobust() << Endl;
            }
            return nullptr;
        }
        for (auto&& entry : jsonContent["tasks"].GetArray()) {
            if (!entry.Has("input_values") || !entry["input_values"].IsMap()) {
                continue;
            }
            auto inputValues = entry["input_values"];
            if (inputValues.Has("secret") && inputValues["secret"].IsString()) {
                auto trueSecretId = inputValues["secret"].GetString();
                if (secretId == trueSecretId) {
                    isSecretIdValid = true;
                    break;
                }
            }
        }

        if (!isSecretIdValid) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "CheckSecretId")
                    ("assignmentId", assignmentId)
                    ("secretId", secretId)
                    ("tasks", jsonContent["tasks"])
                );
            } else {
                ERROR_LOG << "invalid secretId for assignmentId=" << assignmentId << " secretId=" << secretId << Endl;
            }
            return nullptr;
        }

        auto workerId = jsonContent.Has("user_id") && jsonContent["user_id"].IsString() ? jsonContent["user_id"].GetString() : "";
        return MakeAtomicShared<TYangAuthInfo>(true, DefaultUserId, workerId);
    } catch (const TYangRequestError& exception) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "UnsuccessfulReply")
                    ("assignmentId", assignmentId)
                    ("secretId", secretId)
                    ("content", exception.what())
                );
            } else {
                ERROR_LOG << "unsuccessful reply for assignmentId=" << assignmentId << " secretId=" << secretId << Endl;
            }
            return nullptr;
    } catch (const TYangJsonParseException& exception) {
            if (guard) {
                guard->AddEvent(NJson::TMapBuilder
                    ("event", "OutputReadJsonTree")
                    ("assignmentId", assignmentId)
                    ("secretId", secretId)
                    ("content", exception.what())
                );
            } else {
                ERROR_LOG << "non-json response for assignmentId=" << assignmentId << " secretId=" << secretId << Endl << exception.what() << Endl;
            }
            return nullptr;
    } catch (const std::exception& e) {
        if (guard) {
            guard->AddEvent(NJson::TMapBuilder
                ("event", "ExceptionMessage")
                ("assignmentId", assignmentId)
                ("secretId", secretId)
                ("error", FormatExc(e))
            );
        } else {
            ERROR_LOG << "cannot check yang auth for assignmentId=" << assignmentId << " secretId=" << secretId << ":" << FormatExc(e) << Endl;
        }
        return nullptr;
    }
}

void TYangAuthConfig::Init(const TYandexConfig::Section* section) {
    if (!section) {
        WARNING_LOG << "nullptr YandexConfig section" << Endl;
        return;
    }

    const auto& directives = section->GetDirectives();
    DefaultUserId = directives.Value("DefaultUserId", DefaultUserId);
    CheckIsActivePair = directives.Value("CheckIsActivePair", CheckIsActivePair);

    {
        const TYandexConfig::TSectionsMap children = section->GetAllChildren();
        auto it = children.find("YangClient");
        if (it != children.end()) {
            ClientConfig.Init(it->second);
        }
    }
}

void TYangAuthConfig::ToString(IOutputStream& os) const {
    os << "DefaultUserId: " << DefaultUserId << Endl;
    os << "CheckIsActivePair: " << CheckIsActivePair << Endl;

    os << "<YangClient>" << Endl;
    ClientConfig.ToString(os);
    os << "</YangClient>" << Endl;
}

TYangAuthConfig::TFactory::TRegistrator<TYangAuthConfig> TYangAuthConfig::Registrator("yang");
