#include "alerts_child_seats.h"

#include "info_text.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/api/sensor/interface.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TAlertsChildSeatsConfig::TFactory::TRegistrator<TAlertsChildSeatsConfig> TAlertsChildSeatsConfig::Registrator("alerts_child_seats");

bool TAlertsChildSeats::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    TVector<TString> carsIMEI;
    TMap<TString, TString> carHRReport;
    TSet<TString> carsSendingSignals;

    {
        bool getObjectsSuccessful = false;
        TSet<TString> candidateCarIds;
        for (ui32 i = 0; i < 10; ++i) {
            if (server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetIdsFromCache({ Config->GetObservedTagName() }, candidateCarIds, Now() - TDuration::Minutes(1)))
            {
                getObjectsSuccessful = true;
                break;
            }
        }
        if (!getObjectsSuccessful) {
            ERROR_LOG << "Unable to get cars with tag" << Endl;
            return true;
        }

        TVector<TTaggedDevice> taggedCars;
        TSet<TString> tagNames;
        {
            auto registeredTags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
            for (auto&& it : registeredTags) {
                tagNames.emplace(it.first);
            }
        }
        for (ui32 i = 0; i < 10; ++i) {
            if (server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCache(candidateCarIds, tagNames, taggedCars, Now() - TDuration::Minutes(1))) {
                break;
            }
        }

        for (auto&& taggedCar : taggedCars) {
            bool isBookingAvailable = true;
            for (auto&& tag : taggedCar.GetTags()) {
                if (tag->GetTagPriority(0)) {
                    isBookingAvailable = false;
                    break;
                }
            }
            if (!isBookingAvailable) {
                candidateCarIds.erase(taggedCar.GetId());
            }
        }

        auto carObjects = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(candidateCarIds).GetResult();
        for (auto&& car : carObjects) {
            carsIMEI.push_back(car.second.GetIMEI());
            carHRReport[car.second.GetIMEI()] = car.second.GetHRReport();
        }
    }

    auto sensorAPI = server->GetSensorApi();
    auto asyncSensors = sensorAPI->GetSensor(carsIMEI, VEGA_NRF_VISIBLE_MARKS_BF);
    asyncSensors.Wait();
    if (!asyncSensors.HasValue()) {
        ERROR_LOG << "Unable to get the " << VEGA_NRF_VISIBLE_MARKS_BF << " sensor value: " << NThreading::GetExceptionMessage(asyncSensors) << Endl;
        return true;
    }
    TMap<TString, NDrive::ISensorApi::TMultiSensor> sensors = asyncSensors.ExtractValue();

    TVector<TString> reportLines;

    for (auto&& sensor : sensors) {
        carsSendingSignals.insert(sensor.first);
        if (sensor.second.size() != 1) {
//            reportLines.emplace_back(
//                "Некорректное количество присланной информации с сенсоров для " + carHRReport[sensor.first] + ". Ожидалось 1 значение, а получено " + ToString(sensor.second.size()) + "."
//            );
            continue;
        }

        const NDrive::TSensor& sensorValue = sensor.second.back();
        if (std::holds_alternative<double>(sensorValue.Value)) {
            if (fabs(std::get<double>(sensorValue.Value)) < 1e-6) {
                reportLines.emplace_back(carHRReport[sensor.first] + " не получает хорошего сигнала от кресла уже " + ToString(Now() - sensorValue.Since));
            }
        } else {
//            reportLines.emplace_back("Некорректный тип сенсора для " + carHRReport[sensor.first]);
        }

        if (sensorValue.Timestamp + TDuration::Minutes(10) < TInstant::Now()) {
//            reportLines.emplace_back("Сенсор для машины " + carHRReport[sensor.first] + " уже более 10 минут не присылал никаких сигналов.");
        }
    }

//    for (auto&& carIMEI : carsIMEI) {
//        if (!carsSendingSignals.contains(carIMEI)) {
//            reportLines.emplace_back(
//                "Машина " + carHRReport[carIMEI] + " отмечена тегом " + Config->GetObservedTagName() + ", но не присылает сигналы по сенсору " + ToString(VEGA_NRF_VISIBLE_MARKS_BF) + "."
//            );
//        }
//    }

    if (!reportLines.empty()) {
        NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()),
            "👶 Есть " + ToString(reportLines.size()) + " машин, подозрительных на проблемы с детским креслом согласно сенсору " + ToString(VEGA_NRF_VISIBLE_MARKS_BF), reportLines);
    }
    return true;
}

IBackgroundProcess* TAlertsChildSeatsConfig::Construct() const {
    return new TAlertsChildSeats(this);
}
