#include "alerts_devices.h"

#include "info_text.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/autocode/client.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TAlertsDevicesConfig::TFactory::TRegistrator<TAlertsDevicesConfig> TAlertsDevicesConfig::Registrator("alerts_devices");

void TAlertsDevices::SerializeToProto(NDrive::NProto::TDeviceAlertsState& proto) const {
    for (auto&& i : ProblemsSignal) {
        auto& notifyInfo = *proto.AddProblemSignal();
        notifyInfo.SetDeviceId(i.first);
        notifyInfo.SetNotificationInstant(i.second.Seconds());
    }

    for (auto&& i : ProblemsVoltage) {
        auto& notifyInfo = *proto.AddProblemVoltage();
        notifyInfo.SetDeviceId(i.first);
        notifyInfo.SetNotificationInstant(i.second.Seconds());
    }

    proto.SetTimestampLastCallNoTags(TimestampLastCallNoTags.Seconds());

    DEBUG_LOG << "TAlertsDevices::SerializeToProto: " << ProblemsSignal.size() << "/" << ProblemsVoltage.size() << Endl;
}

bool TAlertsDevices::DeserializeFromProto(const NDrive::NProto::TDeviceAlertsState& proto) {
    ProblemsSignal.clear();
    for (auto&& i : proto.GetProblemSignal()) {
        ProblemsSignal.emplace(i.GetDeviceId(), TInstant::Seconds(i.GetNotificationInstant()));
    }

    ProblemsVoltage.clear();
    for (auto&& i : proto.GetProblemVoltage()) {
        ProblemsVoltage.emplace(i.GetDeviceId(), TInstant::Seconds(i.GetNotificationInstant()));
    }

    TimestampLastCallNoTags = TInstant::Seconds(proto.GetTimestampLastCallNoTags());
    return true;
}

bool TAlertsDevices::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    NDrive::INotifier::TPtr notifier = server->GetNotifier(Config->GetNotifierName());
    CHECK_WITH_LOG(notifier);

    FillProblemsReport(server, notifier);

    DEBUG_LOG << "Background process TAlertsDevices iteration finished" << Endl;
    return true;
}

bool TAlertsDevices::FillProblemsReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const {
    const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
    CHECK_WITH_LOG(server->GetDriveAPI()->HasAutocodeClient() && server->GetDriveAPI()->GetAutocodeClient().HasEvacuationClient());

    TVector<TTaggedDevice> devicesProblem;
    {
        TSet<TString> tagsList({TTagIncorrectMoving::TagName, TTagIncorrectRidingZone::TagName, TTagNoSignal::TagName, TTagLowVoltage::TagName});
        if (!manager.GetDeviceTags().GetObjectsFromCache(tagsList, devicesProblem, tagsList, Now())) {
            ERROR_LOG << "Cannot receive objects from cache" << Endl;
            return true;
        }
    }
    TSet<TString> devices;
    for (auto&& i : devicesProblem) {
        devices.emplace(i.GetId());
    }
    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(devices);

    TVector<TDBTag> removeTags;
    TVector<TDBTag> addTags;
    for (auto&& i : devicesProblem) {
        TString carReport;
        auto deviceId = gCars.GetResult().find(i.GetId());
        TString carNumber;
        if (deviceId == gCars.GetResult().end()) {
            carReport = i.GetId();
        } else {
            carReport = deviceId->second.GetHRReport() + "\n";
            carNumber = deviceId->second.GetNumber();
        }
        bool noSent = true;
        for (auto&& tag : i.GetTags()) {
            if (tag->GetName() == TTagNoSignal::TagName) {
                if (Config->GetNoSignalNotificationEnabled()) {
                    auto signalIt = ProblemsSignal.find(i.GetId());
                    if (signalIt == ProblemsSignal.end() || signalIt->second + Config->GetRegularAbsentAlert() < Now()) {
                        carReport += tag->GetHRDescription() + "\n";
                        if (!ProblemsSignal.emplace(i.GetId(), Now()).second) {
                            signalIt->second = Now();
                        }
                        noSent = false;
                    }
                }
            } else if (tag->GetName() == TTagLowVoltage::TagName) {
                auto signalIt = ProblemsVoltage.find(i.GetId());
                if (signalIt == ProblemsVoltage.end() || signalIt->second + Config->GetRegularAbsentAlert() < Now()) {
                    carReport += tag->GetHRDescription() + "\n";
                    if (!ProblemsVoltage.emplace(i.GetId(), Now()).second) {
                        signalIt->second = Now();
                    }
                    noSent = false;
                }
            } else if (tag->GetName() == TTagIncorrectMoving::TagName) {
                carReport += "&#8239;&#8212; " + tag->GetHRDescription() + "\n";
                NDrive::NAutocode::TEvacuationInfo evacuationInfo;
                TMessagesCollector errors;
                if (server->GetDriveAPI()->GetAutocodeClient().GetCarEvacuationStatus(carNumber, evacuationInfo, errors) && evacuationInfo.IsEvacuated()) {
                    carReport += "&#8239;&#8212; машина эвакуирована, данные Автокода:\n";
                    carReport += evacuationInfo.GetHRReport(/* defaultReport = */ "Нет данных") + "\n";

                    if (!!Config->GetEvacuationTag()) {
                        if (!Config->GetDryRunMode()) {
                            auto evacuationTag = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTag(Config->GetEvacuationTag(), Config->GetType());
                            if (!evacuationTag) {
                                ERROR_LOG << GetId() << ": cannot create tag " << Config->GetEvacuationTag() << Endl;
                                return false;
                            }
                            NDrive::TEntitySession session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                            if (!manager.GetDeviceTags().AddTag(evacuationTag, Config->GetType(), deviceId->second.GetId(), server, session, EUniquePolicy::SkipIfExists)) {
                                ERROR_LOG << "Cannot add evacuation tag on car: " << deviceId->second.GetId() << Endl;
                                return false;
                            }
                            if (!session.Commit()) {
                                ERROR_LOG << "Cannot commit session for evacuation tag on car " << deviceId->second.GetId() << ": " << session.GetStringReport() << Endl;
                                return false;
                            }
                            carReport += "&#8239;&#8212; добавлен тег эвакуации\n";
                        } else {
                            carReport += "&#8239;&#8212; необходимо добавить тег " + Config->GetEvacuationTag() + "\n";
                        }
                    }

                    auto sessionBuilder = manager.GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
                    if (!sessionBuilder) {
                        ERROR_LOG << "Cannot use billing sessions builder" << Endl;
                        return false;
                    }
                    auto lastSession = sessionBuilder->GetLastObjectSession(deviceId->second.GetId());
                    if (lastSession) {
                        auto userId = lastSession->GetUserId();
                        auto fetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
                        const TDriveUserData* userData = fetchResult.GetResultPtr(userId);
                        carReport += "&#8239;&#8212; <a href=\"";
                        carReport += TCarsharingUrl().SessionPage(lastSession->GetSessionId());
                        carReport += "\">последняя сессия</a>; водитель: ";
                        carReport += userData ? userData->GetHRReport() : userId;
                    }
                }
                removeTags.push_back(tag);
                noSent = false;
            } else {
                carReport += tag->GetHRDescription() + "\n";
                removeTags.push_back(tag);
                noSent = false;
            }
        }
        if (!noSent) {
            notifier->Notify(carReport);
        }
    }

    for (auto it = ProblemsSignal.begin(); it != ProblemsSignal.end();) {
        if (it->second + Config->GetRegularAbsentAlert() + Config->GetRegularAbsentAlert() < Now()) {
            it = ProblemsSignal.erase(it);
        } else {
            ++it;
        }
    }

    if (!Config->GetDryRunMode()) {
        for (ui32 i = 0; i < 10; ++i) {
            NDrive::TEntitySession session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!manager.GetDeviceTags().RemoveTags(removeTags, "robot_devices_alerts", server, session, false) || !session.Commit()) {
                ERROR_LOG << "Cannot remove problem tags: " << session.GetStringReport() << Endl;
            } else {
                break;
            }
        }
    }
    return true;
}


IBackgroundProcess* TAlertsDevicesConfig::Construct() const {
    return new TAlertsDevices(this);
}
