#include "alerts_special.h"

#include "info_text.h"

#include <drive/backend/background/common/common.h>

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>
#include <rtline/library/storage/structured.h>

#include <util/charset/utf8.h>

namespace {
    TString GetStorageUserKey(const TString& path, const TString& userId, const TString& type) {
        if (!!type) {
            return path + "/" + type + "/" + userId;
        }
        return path + "/" + userId;
    }
}

TAlertsSpecialConfig::TFactory::TRegistrator<TAlertsSpecialConfig> TAlertsSpecialConfig::Registrator("alerts_special");

void TAlertsSpecial::SerializeToProto(NDrive::NProto::TSpecialAlertsState& /*proto*/) const {
}

bool TAlertsSpecial::DeserializeFromProto(const NDrive::NProto::TSpecialAlertsState& /*proto*/) {
    return true;
}

bool TAlertsSpecial::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    NDrive::INotifier::TPtr notifier = server->GetNotifier(Config->GetNotifierName());
    CHECK_WITH_LOG(notifier);

    if (Config->CheckAction(ESpecialAlertActions::CheckLongUsage)) {
        if (!FillLongUsageReport(server, notifier)) {
            ERROR_LOG << "Cannot execute FillLongUsageReport" << Endl;
        }
    }
    if (Config->CheckAction(ESpecialAlertActions::CheckServiceNoTags)) {
        if (!FillServiceNoTagsReport(server, notifier)) {
            ERROR_LOG << "Cannot execute FillServiceNoTagsReport" << Endl;
        }
    }
    if (Config->CheckAction(ESpecialAlertActions::CheckDebt)) {
        if (!FillDebtReport(server, notifier)) {
            ERROR_LOG << "Cannot execute FillDebtReport" << Endl;
        }
    }

    DEBUG_LOG << "Background process TAlertsSpecial iteration finished" << Endl;
    return true;
}

bool TOrderItem::FromTableRecord(const NStorage::TTableRecord& r) {
    double startedAt;
    TString type;
    if (!r.TryGet("user_id", UserId) || !r.TryGet("order_id", OrderId) || !r.TryGet("type", type) || !r.TryGet("started_at", startedAt)) {
        return false;
    }
    State = StateTextToId(type);
    r.TryGet("car_id", CarId);
    StartedAt = TInstant::Seconds((ui64)startedAt);
    return true;
}

bool TAlertsSpecial::FillLongUsageReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const {
    TInstant runStart = Now();
    NDrive::TEventLog::Log("LongUsageReportRunStart", NJson::TMapBuilder
        ("run_start", runStart.Seconds())
    );
    TVector<TOrderItem> orderItems;
    {
        const TDeviceTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
        const TCarTagsHistoryManager& tagsHistoryManager = tagsManager.GetHistoryManager();
        auto sessionBuilder = tagsHistoryManager.GetSessionsBuilder("billing");
        if (!sessionBuilder) {
            ERROR_LOG << "Cannot use billing sessions builder" << Endl;
            return false;
        }

        TInstant now = ModelingNow();
        auto actualSessions = sessionBuilder->GetSessionsActual(now - Max(TDuration::Minutes(10), Config->GetPeriod()), TInstant::Max());

        for (auto&& user : actualSessions) {
            for (auto&& session : user.second) {
                if (session->GetClosed()) {
                    continue;
                }

                TBillingSession::TBillingEventsCompilation eventsCompilation;
                TBillingSession::TBillingCompilation compilation;
                if (session->FillCompilation(eventsCompilation) && session->FillCompilation(compilation)) {
                    TString phaseName;
                    TInstant startInstant;
                    TInstant currentInstant;
                    TSet<TString> tagsInPoint;
                    IOffer::TPtr offerPtr;
                    if (eventsCompilation.GetCurrentPhaseInfo(phaseName, startInstant, currentInstant, tagsInPoint) && !!(offerPtr = compilation.GetCurrentOffer())) {
                        if (!Config->GetLongUsageOfferTagsFilter().IsEmpty() && !CheckOfferTagsFilter(offerPtr, Config->GetLongUsageOfferTagsFilter(), server)) {
                            continue;
                        }
                        TOrderItem item;
                        const TDuration paymentDuration = offerPtr->CalcChargableDuration(startInstant, currentInstant, phaseName, tagsInPoint);
                        item.SetCarId(session->GetObjectId()).SetOrderId(session->GetSessionId()).SetStartedAt(startInstant)
                            .SetCurrentInstant(currentInstant).SetPaymentDuration(paymentDuration).SetState(TOrderItem::StateTextToId(phaseName))
                            .SetUserId(user.first).SetFromScanner(offerPtr->GetFromScanner()).SetOfferName(offerPtr->GetName());
                        if (!!item.GetOrderId() && item.GetState() != TOrderItem::None) {
                            orderItems.emplace_back(std::move(item));
                        }
                    }
                }
            }
        }
    }

    TSet<TString> allUserIds;
    TSet<TString> carIds;
    for (auto&& i : orderItems) {
        allUserIds.insert(i.GetUserId());
        if (i.GetCarId()) {
            carIds.insert(i.GetCarId());
        }
    }
    AddTagPerformers(orderItems, server);

    TSet<TString> userIds;
    ExcludeUsersByDowntime(allUserIds, userIds, ToString(ESpecialAlertActions::CheckLongUsage));

    auto usersInfo = server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
    auto carsInfo = server->GetDriveAPI()->GetCarsData()->FetchInfo(carIds);

    auto& billingManager = server->GetDriveAPI()->GetBillingManager();
    auto session = billingManager.BuildSession(true);
    auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks(MakeVector(userIds), session);
    if (!tasks) {
        return false;
    }
    auto payments = billingManager.GetPaymentsManager().GetSessionsPayments(*tasks, session, true);
    if (!payments) {
        return false;
    }
    {
        for (auto&& i : orderItems) {
            NDrive::TEventLog::TUserIdGuard userIdGuard(i.GetUserId());
            if (!userIds.contains(i.GetUserId())) {
                NDrive::TEventLog::Log("LongUsageReportSkipByDowntime", NJson::TMapBuilder
                    ("run_start", runStart.Seconds())
                    ("session_id", i.GetOrderId())
                    ("state", ToString(i.GetState()))
                    ("payment_duration", i.GetPaymentDuration().Seconds())
                );
                continue;
            }

            auto it = usersInfo.GetResult().find(i.GetUserId());
            TString userReport = i.GetUserId();
            TString userStatus;
            if (it != usersInfo.GetResult().end()) {
                userReport = it->second.GetHRReport();
                userStatus = it->second.GetStatus();
            }

            auto itPayment = payments->find(i.GetOrderId());
            if (itPayment == payments->end()) {
                continue;
            }
            auto optionalDebt = itPayment->second.GetDebt();
            ui32 debt = 0.01 * optionalDebt;
            const bool userHasDebt = debt > 0;

            if (!Config->NeedCheck(i, userHasDebt)) {
                if (userHasDebt) {
                    NDrive::TEventLog::Log("LongUsageReportSkipByNeedCheck", NJson::TMapBuilder
                        ("run_start", runStart.Seconds())
                        ("session_id", i.GetOrderId())
                        ("state", ToString(i.GetState()))
                        ("payment_duration", i.GetPaymentDuration().Seconds())
                    );
                }
                continue;
            }

            DEBUG_LOG << "Notify user: " << i.GetUserId() << ", type: " << i.GetState() << ", started_at: " << i.GetStartedAt() << Endl;

            TString carReport = "-";
            if (i.GetCarId()) {
                auto it = carsInfo.GetResult().find(i.GetCarId());
                carReport = i.GetCarId();
                if (it != carsInfo.GetResult().end()) {
                    carReport = it->second.GetHRReport();
                }
            }

            TStringStream ssReport;
            ssReport << i.GetTypeEmoji() << " ";
            ssReport << userReport;
            if (Config->HasSpecialOffer(i.GetOfferName())) {
                ssReport << "(" << i.GetOfferName() << ")";
            }
            ssReport << ", авто " << carReport << " в состоянии ";
            ssReport << "<a href=\"" << TCarsharingUrl().ClientSessions(i.GetUserId()) << "\">" << i.GetState() << "</a>";
            ssReport << " оплачивает ";
            if (i.GetFromScanner()) {
                ssReport << "после взятия радаром ";
            }
            ssReport << "уже\n";
            ssReport << "&#8239;&#8212; " << FormatDuration(i.GetPaymentDuration());
            if (i.GetDuration() != i.GetPaymentDuration()) {
                ssReport << " (без учета льготного периода: " << FormatDuration(i.GetDuration()) << ")";
            }
            if (userHasDebt) {
                ssReport << "\n&#8239;&#8212; имеет задолженность";
                if (debt > 0) {
                    ssReport << " " << debt << "&#8239руб.";
                }
            }
            if (i.GetPerformTag()) {
                ssReport << "\n&#8239;&#8212; выполняет работу " << i.GetPerformTag();
            }

            NJson::TJsonValue eventLogInfo;
            eventLogInfo.InsertValue("run_start", runStart.Seconds());
            eventLogInfo.InsertValue("session_id", i.GetOrderId());
            eventLogInfo.InsertValue("debt", debt);
            eventLogInfo.InsertValue("offer", i.GetOfferName());
            eventLogInfo.InsertValue("state", ToString(i.GetState()));
            eventLogInfo.InsertValue("start", i.GetStartedAt().Seconds());
            eventLogInfo.InsertValue("payment_duration", i.GetPaymentDuration().Minutes());
            eventLogInfo.InsertValue("scanner", i.GetFromScanner());
            NDrive::INotifier::TMessage message(ssReport.Str());
            message.SetAdditionalInfo(eventLogInfo);
            notifier->Notify(message);
        }
    }

    CleanUsersDowntime(ToString(ESpecialAlertActions::CheckLongUsage));
    NDrive::TEventLog::Log("LongUsageReportRunFinish", NJson::TMapBuilder
        ("run_start", runStart.Seconds())
    );
    return true;
}

bool TAlertsSpecial::FillDebtReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const {
    auto sessionBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
    if (!sessionBuilder) {
        ERROR_LOG << "Cannot use billing sessions builder" << Endl;
        return false;
    }

    TInstant now = ModelingNow();
    auto actualSessions = sessionBuilder->GetSessionsActual(now - TDuration::Seconds(1), now + TDuration::Seconds(1));

    TMap<TString, std::pair<double, TString>> debtByUsers;
    TSet<TString> allUserIds;
    TVector<TString> actualUserIds;
    for (auto&& user : actualSessions) {
        actualUserIds.emplace_back(user.first);
    }
    auto& billingManager = server->GetDriveAPI()->GetBillingManager();
    auto session = billingManager.BuildSession(true);
    auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks(actualUserIds, session);
    if (!tasks) {
        return false;
    }
    auto payments = billingManager.GetPaymentsManager().GetSessionsPayments(*tasks, session, true);
    if (!payments) {
        return false;
    }
    {
        for (auto&& user : actualSessions) {
            TVector<TPaymentsData> userPayments;
            for (auto&& userSession : user.second) {
                auto itPayment = payments->find(userSession->GetSessionId());
                if (itPayment != payments->end()) {
                    userPayments.emplace_back(itPayment->second);
                }
            }
            for (auto&& userSession : user.second) {
                if (userSession->GetClosed()) {
                    continue;
                }
                if (!Config->GetDebtOfferTagsFilter().IsEmpty()) {
                    TBillingSession::TBillingCompilation compilation;
                    if (userSession->FillCompilation(compilation) && compilation.GetCurrentOffer()) {
                        if (!CheckOfferTagsFilter(compilation.GetCurrentOffer(), Config->GetDebtOfferTagsFilter(), server)) {
                            continue;
                        }
                    }
                }
                auto optionalDebt = billingManager.GetDebt(userPayments);
                ui32 debt = 0.01 * optionalDebt;
                if (debt >= Config->GetDebtWarningLimit()) {
                    debtByUsers.emplace(user.first, std::make_pair(debt, userSession->GetLastEvent() ? (*userSession->GetLastEvent())->GetName() : "unknown"));
                    allUserIds.emplace(user.first);
                }
            }
        }
    }

    TSet<TString> userIds;
    ExcludeUsersByDowntime(allUserIds, userIds, ToString(ESpecialAlertActions::CheckDebt));
    if (userIds.empty()) {
        return true;
    }

    auto usersInfo = server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
    for (auto&& i : debtByUsers) {
        if (!userIds.contains(i.first)) {
            continue;
        }
        auto it = usersInfo.GetResultPtr(i.first);
        const TString userReport = it ? it->GetHRReport() : i.first;

        TStringStream ss;
        ss << ESpeciaAlertTypes::Debt << " ";
        ss << "Пользователь " << userReport << " еще едет ";
        ss << "(" << i.second.second << "), а уже должен " << std::floor(i.second.first) << "&#8239руб.";
        notifier->Notify(NDrive::INotifier::TMessage(ss.Str()));
    }

    CleanUsersDowntime(ToString(ESpecialAlertActions::CheckDebt));

    return true;
}

void TAlertsSpecial::ExcludeUsersByDowntime(const TSet<TString>& userIds, TSet<TString>& result, const TString& type) const {
    for (auto&& userId: userIds) {
        TString key = GetStorageUserKey(Config->GetStorageDowntimePath(), userId, type);
        TString value;
        DEBUG_LOG << "ExcludeUsersByDowntime: getting downtime for " << key << Endl;
        if (Storage->GetValue(key, value)) {
            TInstant downtime = TInstant::Seconds(FromString<ui64>(value));
            DEBUG_LOG << "ExcludeUsersByDowntime: get downtime " << downtime << " for user " << key << Endl;
            if (Now() < downtime) {
                DEBUG_LOG << "ExcludeUsersByDowntime: exclude by downtime user " << userId << Endl;
                continue;
            } else {
                DEBUG_LOG << "ExcludeUsersByDowntime: remove downtime for " << userId << Endl;
            }
        }
        result.emplace(userId);
    }
}

void TAlertsSpecial::CleanUsersDowntime(const TString& type) const {
    TVector<TString> userIds;
    Storage->GetNodes(Config->GetStorageDowntimePath() + "/" + type, userIds, false);

    TInstant now = Now();
    for (auto&& userId: userIds) {
        TString key = GetStorageUserKey(Config->GetStorageDowntimePath(), userId, type);
        TString value;
        if (Storage->GetValue(key, value) && TInstant::Seconds(FromString<ui64>(value)) < now) {
            Storage->RemoveNode(key);
            DEBUG_LOG << "CleanUsersDowntime: remove downtime for " << userId << Endl;
        }
    }
}

TVector<TString> TAlertsSpecial::GetPerformerTags(const NDrive::IServer* server) const {
    const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
    TVector<TString> performerTags;
    TVector<TString> tagTypes = Config->GetPerformerTagTypes();
    for (auto&& type : tagTypes) {
        auto tagDescriptions = tagsManager.GetTagsMeta().GetTagsByType(type);
        for (auto&& it : tagDescriptions) {
            performerTags.emplace_back(it->GetName());
        }
    }
    return performerTags;
}

void TAlertsSpecial::AddTagPerformers(TVector<TOrderItem>& orderItems, const NDrive::IServer* server) const {
    TMap<TString, TDBTag> performersInfo;
    TVector<TDBTag> tags;
    TVector<TString> performerTags = GetPerformerTags(server);

    const TDeviceTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!tagsManager.RestorePerformerTags(performerTags, {}, tags, session)) {
            ERROR_LOG << GetId() << ": cannot RestorePerformerTags: " << session.GetStringReport() << Endl;
        }
    }

    for (auto&& tag : tags) {
        performersInfo.emplace(tag->GetPerformer(), tag);
    }
    DEBUG_LOG << "Found " << performersInfo.size() << " performers" << Endl;

    for (auto&& orderItem : orderItems) {
        auto it = performersInfo.find(orderItem.GetUserId());
        TDBTag tg;
        if (it != performersInfo.end() && it->second.GetObjectId() == orderItem.GetCarId()) {
            orderItem.SetPerformTag(it->second.GetData()->GetName());
        }
    }
}

bool TAlertsSpecial::FillServiceNoTagsReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const {
    TRecordsSet records;
    {
        auto transaction = server->GetDriveAPI()->GetDatabase().CreateTransaction(true);
        TStringStream request;
        request << "SELECT id, status ";
        request << "FROM car ";
        request << "WHERE ";
        const TVector<TString>& status = Config->GetWatchCarNoTagsStatus();
        if (!(status.ysize() == 1 && status[0] == "*")) {
            request << "status IN ('" << JoinVectorIntoString(status, "', '") << "')";
        }

        auto result = transaction->Exec(request.Str(), &records);
        if (!result->IsSucceed()) {
            notifier->Notify(NDrive::INotifier::TMessage("Проблемы с сервисом", transaction->GetErrors().GetStringReport()));
            return false;
        }
    }

    TSet<TString> carIdsSet;
    for (auto&& record : records.GetRecords()) {
        TString carId;
        if (record.TryGet("id", carId)) {
            carIdsSet.emplace(carId);
        }
    }
    DEBUG_LOG << "Fetching info for " << carIdsSet.size() << " cars" << Endl;
    auto carsInfo = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(carIdsSet);

    const TDeviceTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    for (ui32 att = 0; att <= 10; ++att) {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        if (!tagsManager.RestoreTags(carIdsSet, TVector<TString>(), tags, session)) {
            if (att == 10) {
                ERROR_LOG << "Cannot restore tags data: " << session.GetStringReport() << Endl;
                notifier->Notify(NDrive::INotifier::TMessage("Ошибка сервера. Не получается восстановить теги"));
                return false;
            }
            continue;
        } else {
            for (auto&& i : tags) {
                carIdsSet.erase(i.GetObjectId());
            }
            break;
        }
    }
    for (auto&& carId : carIdsSet) {
        auto it = carsInfo.GetResult().find(carId);
        TString carReport = carId;
//        TString status = "unknown";
        if (it != carsInfo.GetResult().end()) {
            carReport = it->second.GetHRReport();
//            status = it->second.GetStatus();
        }

        TStringStream ssReport;
        ssReport << "Машина " << carReport << " находится без тегов";
        notifier->Notify(NDrive::INotifier::TMessage(ssReport.Str()));
    }

    return true;
}

TInstant TAlertsSpecial::ChangeInstantHour(const TInstant& origin, ui32 hour) const {
    struct tm tmOrigin;
    origin.LocalTime(&tmOrigin);
    tmOrigin.tm_sec = tmOrigin.tm_min = 0;
    tmOrigin.tm_hour = hour;
    tmOrigin.tm_isdst = -1;
    TInstant result = TInstant::Seconds(::mktime(&tmOrigin));
    return result;
}


IBackgroundProcess* TAlertsSpecialConfig::Construct() const {
    return new TAlertsSpecial(this);
}


ITelegramCommandsProcessor* TAlertUsersTelegramProcessorConfig::Construct(TActiveTelegramBot* bot) const {
    return new TAlertUsersTelegramProcessor(*this, bot);
}

bool TAlertUsersTelegramProcessor::ParseDowntimeArgs(const TBotCommand& command, TString& userId, TDuration& downtime) const {
    TGUID uuid;
    if (ExtractUserId(command.GetReplyToMessage(), userId)) {
        if (command.GetArgs().size() > 0) {
            TryFromString<TDuration>(command.GetArg(0), downtime);
        }
    } else if (command.GetArgs().size() > 0 && GetUuid(command.GetArg(0), uuid)) {
        userId = command.GetArg(0);
        if (command.GetArgs().size() > 1) {
            TryFromString<TDuration>(command.GetArg(1), downtime);
        }
    } else {
        return false;
    }
    return true;
}

bool TAlertUsersTelegramProcessor::ProcessCommand(const TBotCommand& command, const TString& chatId, const IServerBase* server) {
    Storage = server->GetVersionedStorage(Config.GetStorageName());
    if (!Storage) {
        return false;
    }
    ECommand commandCode = ECommand::None;
    if (command.Is("downtime")) {
        commandCode = ECommand::Downtime;
    } else if (command.Is("no_answer")) {
        commandCode = ECommand::NoAnswer;
    }

    if (commandCode == ECommand::Downtime || commandCode == ECommand::NoAnswer) {
        TString userId;
        TDuration downtime = Config.GetDefaultDowntime();
        if (commandCode == ECommand::NoAnswer) {
            downtime = Config.GetDefaultNoAnswerDowntime();
        }

        if (!ParseDowntimeArgs(command, userId, downtime)) {
            Bot->Notify(NDrive::INotifier::TMessage("Некорректные аргументы команды " + command.GetCommand()), chatId);
            return false;
        }

        if (userId) {
            auto g = server->GetAsPtrSafe<NDrive::IServer>()->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            TString userReport = userId;
            TString tagsReport = "&#8212;";
            auto&& it = g.GetResult().find(userId);
            if (it != g.GetResult().end()) {
                userReport = it->second.GetHRReport();
            }
            const TInstant limit = Now() + downtime;
            TStringStream message;
            message << "Оповещения о " << userReport << " отключены до " << limit.FormatLocalTime("%F %T");
            message << ", теги: " << tagsReport;

            ESpecialAlertActions downtimeType = ESpecialAlertActions::CheckLongUsage;
            ESpeciaAlertTypes alertType;
            if (command.GetReplyToMessage().Has("text")) {
                TString text = command.GetReplyToMessage()["text"].GetStringRobust();
                TVector<TString> parts(SplitString(text, " ", 2));
                if (!parts.empty()) {
                    TString entities = ConvertUTF8ToEntities(parts[0]);
                    if (TryFromString(entities, alertType)) {
                        switch (alertType) {
                        case ESpeciaAlertTypes::CarsharingRide:
                        case ESpeciaAlertTypes::CarsharingReservationPaid:
                        case ESpeciaAlertTypes::CarsharingParking:
                            downtimeType = ESpecialAlertActions::CheckLongUsage;
                            break;
                        case ESpeciaAlertTypes::Debt:
                            downtimeType = ESpecialAlertActions::CheckDebt;
                            break;
                        }
                    } else {
                        WARNING_LOG << "Unknown message type: " << text << Endl;
                    }
                } else {
                    WARNING_LOG << "Unknown message type: " << text << Endl;
                }
            }

            const TString key = GetStorageUserKey(Config.GetStoragePath(), userId, ToString(downtimeType));
            Storage->SetValue(key, ToString<ui64>(limit.Seconds()), false, false);
            DEBUG_LOG << "Execute: downtime: " << userId << " until: "<< limit << Endl;
            Bot->Notify(NDrive::INotifier::TMessage(message.Str()), chatId);
            return true;
        }
    }

    return false;
}

TString TAlertUsersTelegramProcessor::ConvertUTF8ToEntities(const TString& str) {
    TStringStream result;
    size_t length = 0;
    while (length < str.size()) {
        size_t len;
        wchar32 wc;
        if (SafeReadUTF8Char(wc, len, (const unsigned char*)str.c_str() + length, (const unsigned char*)str.c_str() + str.size()) != RECODE_OK) {
            return result.Str();
        }
        result << "&#" << ToString((ui64)wc) << ";";
        length += len;
    }
    return result.Str();
}


bool TAlertUsersTelegramProcessor::ExtractUserId(const NJson::TJsonValue& message, TString& userId) const {
    if (!message.Has("entities") || !message["entities"].IsArray()) {
        return false;
    }

    for (auto&& it: message["entities"].GetArray()) {
        if (!it.Has("url")) {
            continue;
        }
        TVector<TString> parts(SplitString(it["url"].GetString(), "/"));
        for (ui32 i = 0; i + 1 < parts.size(); ++i) {
            if (parts[i] == "clients") {
                userId = parts[i + 1];
                return true;
            }
        }
    }

    return false;
}

bool TAlertsSpecial::CheckOfferTagsFilter(IOffer::TPtr offer, const TTagsFilter& filter, const NDrive::IServer* server) const {
    auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
    if (action) {
        auto offerBuilder = action->GetAs<IOfferBuilderAction>();
        if (offerBuilder) {
            if (!filter.IsMatching(offerBuilder->GetGrouppingTags())) {
                return false;
            }
        }
    }
    return true;
}

TAlertUsersTelegramProcessorConfig::TFactory::TRegistrator<TAlertUsersTelegramProcessorConfig> TAlertUsersTelegramProcessorConfig::Registrator("alert_users");
