#pragma once
#include <drive/backend/background/manager/regular.h>
#include <drive/backend/background/telegram/telegram.h>
#include <drive/backend/background/telegram_bot/telegram.h>

#include <drive/backend/data/alerts/config.h>
#include <drive/backend/data/proto/alerts.pb.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/tags/tags_filter.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/util/network/neh.h>

enum class ESpecialAlertActions {
    CheckLongUsage = 0 /* "long_usage" */,
    CheckServiceNoTags = 1 /* "service_no_tags" */,
    CheckDebt = 2 /* "debt" */,
};

/*
  All the values should be different, because it is used to determine message type
 */
enum class ESpeciaAlertTypes {
    CarsharingRide = 0 /* "&#128663;" */,
    CarsharingReservationPaid = 1 /* "&#128176;" */,
    CarsharingParking = 2 /* "&#127359;&#65039;" */,
    Debt = 4 /* "&#128179;" */
};

class TOrderItem {
public:
    enum EState {
        None /* "none" */,
        Parking /* "parking" */,
        Reservation /* "reservation" */,
        Riding /* "riding" */
    };

    R_FIELD(TString, UserId, Default<TString>());
    R_FIELD(TString, OrderId, Default<TString>());
    R_FIELD(EState, State, EState::None);
    R_FIELD(TInstant, StartedAt, TInstant::Zero());
    R_FIELD(TString, CarId, Default<TString>());
    R_FIELD(TString, PerformTag, Default<TString>());
    R_FIELD(TDuration, PaymentDuration, TDuration::Zero());
    R_FIELD(TString, OfferName, Default<TString>());
    R_FIELD(TInstant, CurrentInstant, TInstant::Zero());
    R_FIELD(bool, FromScanner, false);

public:
    TOrderItem() = default;
    TOrderItem(const TString& userId, const TString& orderId, const TString& type, TInstant startedAt, const TString& carId)
        : UserId(userId)
        , OrderId(orderId)
        , State(StateTextToId(type))
        , StartedAt(startedAt)
        , CarId(carId)
    {
    }

    TDuration GetDuration() const {
        return CurrentInstant - StartedAt;
    }

    static EState StateTextToId(const TString& type) {
        if (type == "carsharing_ride" || type == "old_state_riding") {
            return EState::Riding;
        } else if (type == "carsharing_reservation_paid" || type == "carsharing_reservation" || type == "old_state_reservation") {
            return EState::Reservation;
        } else if (type == "carsharing_parking" || type == "old_state_parking") {
            return EState::Parking;
        }
        WARNING_LOG << "Unknown order state: " << type << Endl;
        return EState::None;
    }

    bool FromTableRecord(const NStorage::TTableRecord& r);

    TString GetTypeEmoji() const {
        switch (State) {
            case Parking:
                return ToString(ESpeciaAlertTypes::CarsharingParking);
            case Reservation:
                return ToString(ESpeciaAlertTypes::CarsharingReservationPaid);
            case Riding:
                return ToString(ESpeciaAlertTypes::CarsharingRide);
            case None:
                break;
        }
        return Default<TString>();
    }
};

class TAlertsSpecialConfig : public IBackgroundRegularProcessConfig {
    using IBackgroundRegularProcessConfig::IBackgroundRegularProcessConfig;
    static TFactory::TRegistrator<TAlertsSpecialConfig> Registrator;
private:
    class TSpecialOfferConfig {
    public:
        R_READONLY(TString, OfferName);
        R_READONLY(TDuration, WatchInterval, TDuration::Hours(4));
        R_READONLY(bool, NeedDebtCheck, false);

    public:
        virtual void Init(const TYandexConfig::Section* section) {
            AssertCorrectConfig(section->GetDirectives().GetValue("OfferName", OfferName), "no 'OfferName' field");
            WatchInterval = section->GetDirectives().Value("WatchInterval", WatchInterval);
            NeedDebtCheck = section->GetDirectives().Value("NeedDebtCheck", NeedDebtCheck);
        }

        virtual void ToString(IOutputStream& os) const {
            os << "OfferName:" << OfferName << Endl;
            os << "WatchInterval:" << WatchInterval << Endl;
            os << "NeedDebtCheck:" << NeedDebtCheck << Endl;
        }

        bool NeedCheck(const TOrderItem& order, bool hasDebt) const {
            if (NeedDebtCheck && !hasDebt) {
                return false;
            }
            return order.GetPaymentDuration() > WatchInterval;
        }
    };
private:
    R_READONLY(TString, NotifierName);
    R_READONLY(TDuration, WatchInterval, TDuration::Hours(2));
    R_READONLY(TDuration, ScannerWatchInterval, TDuration::Hours(1));

    R_READONLY(ui32, FreePeriodHourBegin, 0);
    R_READONLY(ui32, FreePeriodHourEnd, 6);

    R_READONLY(ui32, DebtWarningLimit, 1024);

    TVector<TString> PerformerTagTypes = {
        "car_service_tag",
        "simple_car_tag",
    };
    TMap<TString, TSpecialOfferConfig> SpecialOfferConfigs;

    R_READONLY(TVector<TString>, WatchCarNoTagsStatus, { "service" });

    R_READONLY(TDuration, WatchCarNoTagsPeriod, TDuration::Minutes(20));
    R_READONLY(ui32, CleanUsersDowntimeIter, 20);
    R_READONLY(TString, StorageDowntimePath, "/alert_users_downtime");

    R_READONLY(TTagsFilter, LongUsageOfferTagsFilter);
    R_READONLY(TTagsFilter, DebtOfferTagsFilter);

    TSet<ESpecialAlertActions> Actions;

public:
    const TVector<TString>& GetPerformerTagTypes() const {
        return PerformerTagTypes;
    }

    bool CheckAction(const ESpecialAlertActions action) const {
        return Actions.contains(action);
    }

    bool HasSpecialOffer(const TString& offerName) const {
        return SpecialOfferConfigs.find(offerName) != SpecialOfferConfigs.end();
    }

    bool NeedCheck(const TOrderItem& order, bool hasDebt) const {
        if (order.GetFromScanner() && order.GetState() == TOrderItem::Reservation) {
            return order.GetPaymentDuration() > GetScannerWatchInterval();
        }
        auto it = SpecialOfferConfigs.find(order.GetOfferName());
        if (it != SpecialOfferConfigs.end()) {
            return it->second.NeedCheck(order, hasDebt);
        }
        if (order.GetState() == TOrderItem::Riding && !hasDebt) {
            return false;
        }
        return order.GetPaymentDuration() > GetWatchInterval();
    }

    std::pair<ui32, ui32> GetFreePeriod() const {
        return std::pair<ui32, ui32>(FreePeriodHourBegin, FreePeriodHourEnd);
    }

    virtual IBackgroundProcess* Construct() const override;

    virtual void Init(const TYandexConfig::Section* section) override {
        IBackgroundRegularProcessConfig::Init(section);
        AssertCorrectConfig(section->GetDirectives().GetValue("NotifierName", NotifierName), "no 'NotifierName' field");

        WatchInterval = section->GetDirectives().Value("WatchInterval", WatchInterval);
        ScannerWatchInterval = section->GetDirectives().Value("ScannerWatchInterval", ScannerWatchInterval);

        FreePeriodHourBegin = section->GetDirectives().Value<ui32>("FreePeriodHourBegin", FreePeriodHourBegin);
        FreePeriodHourEnd = section->GetDirectives().Value<ui32>("FreePeriodHourEnd", FreePeriodHourEnd);

        DebtWarningLimit = section->GetDirectives().Value<ui32>("DebtWarningLimit", DebtWarningLimit);

        section->GetDirectives().FillArray("PerformerTagTypes", PerformerTagTypes);
        section->GetDirectives().FillArray("WatchCarNoTagsStatus", WatchCarNoTagsStatus);

        WatchCarNoTagsPeriod = section->GetDirectives().Value("WatchCarNoTagsPeriod", WatchCarNoTagsPeriod);
        CleanUsersDowntimeIter = section->GetDirectives().Value("CleanUsersDowntimeIter", CleanUsersDowntimeIter);
        StorageDowntimePath = section->GetDirectives().Value("StorageDowntimePath", StorageDowntimePath);

        TString debtOfferTagsFilterStr;
        debtOfferTagsFilterStr = section->GetDirectives().Value("DebtOfferTagsFilter", debtOfferTagsFilterStr);
        if (debtOfferTagsFilterStr) {
            AssertCorrectConfig(DebtOfferTagsFilter.DeserializeFromString(debtOfferTagsFilterStr), "can't parse DebtOfferTagsFilter field");
        }
        TString longUsageOfferTagsFilterStr;
        longUsageOfferTagsFilterStr = section->GetDirectives().Value("LongUsageOfferTagsFilter", longUsageOfferTagsFilterStr);
        if (longUsageOfferTagsFilterStr) {
            AssertCorrectConfig(LongUsageOfferTagsFilter.DeserializeFromString(longUsageOfferTagsFilterStr), "can't parse LongUsageOfferTagsFilter field");
        }

        TVector<ESpecialAlertActions> actions;
        if (section->GetDirectives().TryFillArray<ESpecialAlertActions, char, false>("Actions", actions)) {
            Actions = TSet<ESpecialAlertActions>(actions.begin(), actions.end());
        }

        TYandexConfig::TSectionsMap sections = section->GetAllChildren();
        auto offerSections = sections.equal_range("SpecialOffer");
        for (; offerSections.first != offerSections.second; ++offerSections.first) {
            TSpecialOfferConfig offerConfig;
            offerConfig.Init(offerSections.first->second);
            AssertCorrectConfig(SpecialOfferConfigs.insert(std::make_pair(offerConfig.GetOfferName(), offerConfig)).second, "Special offer config for " + offerConfig.GetOfferName() + " was duplicated");
        }
    }

    virtual void ToString(IOutputStream& os) const override {
        IBackgroundRegularProcessConfig::ToString(os);
        os << "NotifierName: " << NotifierName << Endl;
        os << "WatchInterval: " << WatchInterval << Endl;
        os << "ScannerWatchInterval: " << ScannerWatchInterval << Endl;
        os << "FreePeriodHourBegin: " << FreePeriodHourBegin << Endl;
        os << "FreePeriodHourEnd: " << FreePeriodHourEnd << Endl;
        os << "DebtWarningLimit: " << DebtWarningLimit << Endl;
        os << "WatchCarNoTagsStatus: " << JoinSeq(",", WatchCarNoTagsStatus) << Endl;
        os << "WatchCarNoTagsPeriod: " << WatchCarNoTagsPeriod << Endl;
        os << "CleanUsersDowntimeIter: " << CleanUsersDowntimeIter << Endl;
        os << "StorageDowntimePath: " << StorageDowntimePath << Endl;
        os << "PerformerTagTypes: " << JoinSeq(",", PerformerTagTypes) << Endl;
        os << "DebtOfferTagsFilter" << DebtOfferTagsFilter.ToString() << Endl;
        os << "LongUsageOfferTagsFilter" << LongUsageOfferTagsFilter.ToString() << Endl;
        os << "Actions: " << JoinSeq(",", Actions) << Endl;
        for (auto&& offer : SpecialOfferConfigs) {
            os << "<SpecialOffer>" << Endl;
            offer.second.ToString(os);
            os << "</SpecialOffer>" << Endl;
        }
    }
};


class TAlertsSpecial: public ISerializableProtoBackgroundProcess<NDrive::NProto::TSpecialAlertsState, IBackgroundRegularProcessImpl<NDrive::IServer>> {
private:
    using TBase = ISerializableProtoBackgroundProcess<NDrive::NProto::TSpecialAlertsState, IBackgroundRegularProcessImpl<NDrive::IServer>>;

    const TAlertsSpecialConfig* Config = nullptr;

private:
    virtual void SerializeToProto(NDrive::NProto::TSpecialAlertsState& proto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TSpecialAlertsState& proto) override;

    void ExcludeUsersByDowntime(const TSet<TString>& userIds, TSet<TString>& result, const TString& type) const;
    void CleanUsersDowntime(const TString& type) const;
    void AddTagPerformers(TVector<TOrderItem>& orderItems, const NDrive::IServer* server) const;
    TVector<TString> GetPerformerTags(const NDrive::IServer* server) const;
    bool CheckOfferTagsFilter(IOffer::TPtr offer, const TTagsFilter& filter, const NDrive::IServer* server) const;

protected:
    bool DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const override;

    bool FillLongUsageReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const;
    bool FillDebtReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const;
    bool FillServiceNoTagsReport(const NDrive::IServer* server, NDrive::INotifier::TPtr notifier) const;

    TInstant ChangeInstantHour(const TInstant& origin, ui32 hour) const;

public:
    TAlertsSpecial(const TAlertsSpecialConfig* config)
        : TBase(*config)
        , Config(config) {
    }

    virtual void Stop() override {
    }

    virtual void Start() override {
    }

};


class TAlertUsersTelegramProcessorConfig: public ITelegramCommandsProcessorConfig {
private:
    TString StorageName;
    TString StoragePath = "/alert_users_downtime";
    TDuration DefaultDowntime = TDuration::Seconds(3600);
    TDuration DefaultNoAnswerDowntime = TDuration::Hours(12);
    TSet<TString> ReportAboutTags;
public:
    static TFactory::TRegistrator<TAlertUsersTelegramProcessorConfig> Registrator;

    virtual void ToString(IOutputStream& os) const override {
        os << "StorageName: " << StorageName << Endl;
        os << "StoragePath: " << StoragePath << Endl;
        os << "DefaultDowntime: " << DefaultDowntime.Seconds() << Endl;
        os << "DefaultNoAnswerDowntime: " << DefaultNoAnswerDowntime.Seconds() << Endl;
        os << "ReportAboutTags: " << JoinSeq(",", ReportAboutTags) << Endl;
    }

    virtual void Init(TYandexConfig::Section* section) override {
        AssertCorrectConfig(section->GetDirectives().GetValue("StorageName", StorageName), "no 'StorageName' field");
        StoragePath = section->GetDirectives().Value("StoragePath", StoragePath);
        DefaultDowntime = section->GetDirectives().Value("DefaultDowntime", DefaultDowntime);
        DefaultNoAnswerDowntime = section->GetDirectives().Value("DefaultNoAnswerDowntime", DefaultNoAnswerDowntime);
        TVector<TString> tags;
        if (section->GetDirectives().FillArray("ReportAboutTags", tags)) {
            ReportAboutTags.insert(tags.begin(), tags.end());
        }
    }

    const TString& GetStorageName() const {
        return StorageName;
    }

    const TString& GetStoragePath() const {
        return StoragePath;
    }

    const TDuration GetDefaultDowntime() const {
        return DefaultDowntime;
    }

    const TDuration GetDefaultNoAnswerDowntime() const {
        return DefaultNoAnswerDowntime;
    }

    const TSet<TString>& GetReportAboutTags() const {
        return ReportAboutTags;
    }

    ITelegramCommandsProcessor* Construct(TActiveTelegramBot* bot) const override;
};


class TAlertUsersTelegramProcessor: public IChatTelegramCommandsProcessor {
public:
    enum class ECommand {
        None,
        Downtime,
        NoAnswer
    };

private:

    class TAlertUsersState {
    public:
        TMap<TString, TInstant> UsersDowntime;
    };

    NRTLine::IVersionedStorage::TPtr Storage;

    mutable TAlertUsersState AlertUsersState;

    const TAlertUsersTelegramProcessorConfig Config;

    bool ExtractUserId(const NJson::TJsonValue& message, TString& userId) const;
    bool ParseDowntimeArgs(const TBotCommand& command, TString& userId, TDuration& downtime) const;
    TString ConvertUTF8ToEntities(const TString& str);

public:

    TAlertUsersTelegramProcessor(const TAlertUsersTelegramProcessorConfig& config, TActiveTelegramBot* bot)
        : IChatTelegramCommandsProcessor(bot)
        , Config(config)
    {
    }

    bool ProcessCommand(const TBotCommand& command, const TString& chatId, const IServerBase* server) override;
};

