#include "alerts_traces.h"

#include "info_text.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/graph/geometry_graph/common/types.h>
#include <rtline/library/storage/abstract.h>

TAlertsTracesConfig::TFactory::TRegistrator<TAlertsTracesConfig> TAlertsTracesConfig::Registrator("alerts_traces");

void TAlertsTraces::SerializeToProto(NDrive::NProto::TTraceAlertsState& /*proto*/) const {
    DEBUG_LOG << "TAlertsTraces::SerializeToProto " << Endl;
}

bool TAlertsTraces::DeserializeFromProto(const NDrive::NProto::TTraceAlertsState& /*proto*/) {
    return true;
}

bool TAlertsTraces::SetsIntersect(const TSet<TString>& s1, const TSet<TString>& s2) const {
    for (auto&& s : s1) {
        if (s2.contains(s)) {
            return true;
        }
    }
    return false;
}

TMaybe<TAlertsTraceRule> TAlertsTraces::HasApplicableRule(const TDBTag& tag, const TString& modelCode, const TSet<TString>& areas) const {
    for (auto&& rule : Config->GetRules()) {
        if (!!rule.GetTagName() && tag->GetName() != rule.GetTagName()) {
            continue;
        }
        if (!rule.GetIncludeModelCodes().empty() && !rule.GetIncludeModelCodes().contains(modelCode)) {
            continue;
        }
        if (!rule.GetExcludeModelCodes().empty() && rule.GetExcludeModelCodes().contains(modelCode)) {
            continue;
        }

        if (!areas.empty()) {
            if (!rule.GetIncludeAreas().empty() && !SetsIntersect(areas, rule.GetIncludeAreas())) {
                continue;
            }
            if (!rule.GetExcludeAreas().empty() && SetsIntersect(areas, rule.GetExcludeAreas())) {
                continue;
            }
        }

        // applicable rule, check condition
        if (tag->GetName() == TTagIncorrectVelocity::TagName) {
            auto impl = tag.GetTagAs<TTagIncorrectVelocity>();
            double velocityDelta = Yensured(impl)->GetVDelta();
            velocityDelta = TSpeedWithUnit::MS(velocityDelta).KmH();
            if (velocityDelta < rule.GetVelocityDelta()) {
                return {};
            }
            return rule;
        }
    }

    return {};
}

bool TAlertsTraces::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    NDrive::INotifier::TPtr notifier = server->GetNotifier(Config->GetNotifierName());
    CHECK_WITH_LOG(notifier);
    TMap<TString, NDrive::INotifier::TPtr> notifiers;
    for (auto&& rule : Config->GetRules()) {
        notifiers[rule.GetName()] = server->GetNotifier(rule.GetNotifierName());
        CHECK_WITH_LOG(notifiers[rule.GetName()]);
    }

    const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

    TVector<TDBTag> tracesProblem;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!manager.GetTraceTags().RestoreTags(TSet<TString>(), { TTagIncorrectVelocity::TagName }, tracesProblem, session)) {
            ERROR_LOG << "Cannot receive objects from cache" << Endl;
            return true;
        }
    }

    TSet<TString> users;
    TSet<TString> devices;
    TMap<TString, std::pair<TString, TString>> remapToData;
    TMap<TString, TString> traceReports;
    for (auto&& tag : tracesProblem) {
        auto tagData = tag.GetTagAs<TTraceTag>();
        Y_ENSURE(tagData, "cannot cast " << tag.GetTagId() << " as TraceTag");
        remapToData[tag.GetObjectId()].first = tagData->GetDeviceId();
        remapToData[tag.GetObjectId()].second = tagData->GetUserId();
        traceReports.emplace(tag.GetObjectId(), Sprintf("<a href=\"%s\">Трек</a>", TCarsharingUrl().TrackPage(tagData->GetSessionId()).data()));
        users.emplace(tagData->GetUserId());
        devices.emplace(tagData->GetDeviceId());
        DEBUG_LOG << "Found problem " << tagData->GetDeviceId() << " user " << tagData->GetUserId() << Endl;
        break;
    }
    auto gCars = server->GetDriveAPI()->GetCarsData()->FetchInfo(devices);
    auto gUsers = server->GetDriveAPI()->GetUsersData()->FetchInfo(users);
    TDevicesSnapshot snapshots = server->GetSnapshotsManager().GetSnapshots(devices);

    TVector<TDBTag> tracesTags;
    for (auto&& i : tracesProblem) {
        auto dataRemap = remapToData[i.GetObjectId()];

        auto traceReportIt = traceReports.find(i.GetObjectId());
        const TString traceReport = (traceReportIt == traceReports.end()) ? "" : traceReportIt->second;

        auto carInfo = gCars.GetResult().find(dataRemap.first);
        TString carModelCode;
        TString carReport;
        if (carInfo == gCars.GetResult().end()) {
            carReport = dataRemap.first;
        } else {
            carReport = carInfo->second.GetHRReport();
            carModelCode = carInfo->second.GetModel();
        }

        auto userInfo = gUsers.GetResult().find(dataRemap.second);
        TString userReport;
        if (userInfo == gUsers.GetResult().end()) {
            userReport = dataRemap.second;
        } else {
            userReport = userInfo->second.GetHRReport();
        }

        TSet<TString> areas;
        auto snapshotIt = snapshots.GetSnapshots().find(dataRemap.first);
        if (snapshotIt != snapshots.GetSnapshots().end()) {
            NDrive::TLocation location;
            snapshotIt->second.GetLocation(location, TDuration::Minutes(1));
            areas = server->GetDriveAPI()->GetTagsInPoint({ location.Longitude, location.Latitude });
        }

        {
            TStringStream ss;
            ss << carReport << Endl;
            ss << userReport << Endl;
            if (!!traceReport) {
                ss << traceReport << Endl;
            }
            ss << i->GetHRDescription() << Endl;

            auto rule = HasApplicableRule(i, carModelCode, areas);

            if (rule.Defined()) {
                ss << "Правило: " << rule.GetRef().Print();
                notifier->Notify(ss.Str());
            } else {
                DEBUG_LOG << "Rule not found " << i.GetObjectId() << Endl;
            }
            tracesTags.push_back(i);
        }
    }

    if (!Config->GetDryRunMode()) {
        auto session = manager.GetTraceTags().BuildTx<NSQL::Writable>();
        if (!manager.GetTraceTags().RemoveTags(tracesTags, "robot_alerts_traces", server, session, false)) {
            ERROR_LOG << GetId() << ": cannot remove tags " << session.GetStringReport() << Endl;
            return true;
        }
        if (!session.Commit()) {
            ERROR_LOG << GetId() << ": cannot commit transaction " << session.GetStringReport() << Endl;
            return true;
        }
    }

    return true;
}

IBackgroundProcess* TAlertsTracesConfig::Construct() const {
    return new TAlertsTraces(this);
}
