#pragma once

#include <drive/backend/background/manager/regular.h>

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/alerts/config.h>
#include <drive/backend/data/proto/alerts.pb.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/util/network/neh.h>

class TAlertInfo;

class TAlertsTraceRule {
public:
    R_FIELD(TString, Name);
    R_FIELD(i32, Priority, 0);
    R_FIELD(TSet<TString>, IncludeModelCodes, {});
    R_FIELD(TSet<TString>, ExcludeModelCodes, {});
    R_FIELD(TSet<TString>, IncludeAreas, {});
    R_FIELD(TSet<TString>, ExcludeAreas, {});
    R_FIELD(TString, TagName);
    R_FIELD(ui32, VelocityDelta, 0);
    R_FIELD(TString, NotifierName);

private:
    void InitSetField(const TYandexConfig::Section* section, const TString& name, TSet<TString>& result) {
        TVector<TString> vec;
        section->GetDirectives().FillArray(name, vec);
        result.insert(vec.begin(), vec.end());
    }

public:
    bool operator < (const TAlertsTraceRule& o) const {
        if (Priority > o.Priority && Name < o.Name) {
            return true;
        }
        return false;
    }

    TString Print() const {
        TStringStream ss;
        ss << "name=" << Name << " priority=" << Priority << " model_codes=" << JoinSeq(",", IncludeModelCodes) << "/" << JoinSeq(",", ExcludeModelCodes)
            << " areas=" << JoinSeq(",", IncludeAreas) << "/" << JoinSeq(",", ExcludeAreas) << " vdelta=" << VelocityDelta;
        return ss.Str();
    }

    void Init(const TYandexConfig::Section* section, i32 priority = 0) {
        AssertCorrectConfig(section->GetDirectives().GetValue("TagName", TagName), "no 'TagName' field");
        Priority = section->GetDirectives().Value("Priority", priority);
        InitSetField(section, "IncludeModelCodes", IncludeModelCodes);
        InitSetField(section, "ExcludeModelCodes", ExcludeModelCodes);
        InitSetField(section, "IncludeAreas", IncludeAreas);
        InitSetField(section, "ExcludeAreas", ExcludeAreas);
        VelocityDelta = section->GetDirectives().Value("VelocityDelta", VelocityDelta);
        section->GetDirectives().Value("NotifierName", NotifierName);
    }

    void ToString(IOutputStream& os) const {
        os << "TagName: " << TagName << Endl;
        os << "Priority: " << Priority << Endl;
        os << "IncludeModelCodes: " << JoinSeq(",", IncludeModelCodes);
        os << "ExcludeModelCodes: " << JoinSeq(",", ExcludeModelCodes);
        os << "IncludeAreas: " << JoinSeq(",", IncludeAreas);
        os << "ExcludeAreas: " << JoinSeq(",", ExcludeAreas);
        os << "VelocityDelta: " << VelocityDelta << Endl;
        os << "NotifierName: " << NotifierName << Endl;
    }
};

class TAlertsTracesConfig: public IBackgroundRegularProcessConfig {
private:
    TString NotifierName;
    TVector<TAlertsTraceRule> Rules;

    static TFactory::TRegistrator<TAlertsTracesConfig> Registrator;
public:

    using IBackgroundRegularProcessConfig::IBackgroundRegularProcessConfig;

    const TString& GetNotifierName() const {
        return NotifierName;
    }

    const TVector<TAlertsTraceRule>& GetRules() const {
        return Rules;
    }

    virtual IBackgroundProcess* Construct() const override;

    virtual void Init(const TYandexConfig::Section* section) override {
        IBackgroundRegularProcessConfig::Init(section);
        AssertCorrectConfig(section->GetDirectives().GetValue("NotifierName", NotifierName), "no 'NotifierName' field");
        Rules.clear();

        auto children = section->GetAllChildren();
        auto rulesIt = children.find("Rules");
        if (rulesIt != children.end()) {
            i32 count = 0;
            for (auto&& ruleIt : rulesIt->second->GetAllChildren()) {
                TAlertsTraceRule rule;
                rule.SetNotifierName(NotifierName);
                rule.SetName(ruleIt.first);
                rule.Init(ruleIt.second, rulesIt->second->GetAllChildren().size() - count);
                ++count;
                Rules.emplace_back(rule);
            }
        }
        TAlertsTraceRule defaultRule;
        defaultRule.SetName("_default");
        defaultRule.SetPriority(-1);
        defaultRule.SetNotifierName(NotifierName);
        Rules.emplace_back(defaultRule);

        std::sort(Rules.begin(), Rules.end());
    }

    virtual void ToString(IOutputStream& os) const override {
        IBackgroundRegularProcessConfig::ToString(os);
        os << "NotifierName: " << NotifierName << Endl;
        if (!Rules.empty()) {
            os << "<Rules>" << Endl;
            for (auto&& rule : Rules) {
                rule.ToString(os);
            }
            os << "</Rules>" << Endl;
        }
    }
};

class TAlertsTraces: public ISerializableProtoBackgroundProcess<NDrive::NProto::TTraceAlertsState, IBackgroundRegularProcessImpl<NDrive::IServer>> {
private:
    using TBase = ISerializableProtoBackgroundProcess<NDrive::NProto::TTraceAlertsState, IBackgroundRegularProcessImpl<NDrive::IServer>>;
    virtual void SerializeToProto(NDrive::NProto::TTraceAlertsState& proto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TTraceAlertsState& proto) override;

    bool SetsIntersect(const TSet<TString>& s1, const TSet<TString>& s2) const;
    TMaybe<TAlertsTraceRule> HasApplicableRule(const TDBTag& tag, const TString& modelCode, const TSet<TString>& areas) const;

    const TAlertsTracesConfig* Config = nullptr;

protected:
    bool DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const override;

public:
    TAlertsTraces(const TAlertsTracesConfig* config)
        : TBase(*config)
        , Config(config)
    {
    }

    virtual void Stop() override {
    }

    virtual void Start() override {
    }
};
