#include "watcher_traces.h"

#include "info_text.h"

#include <drive/backend/data/alerts/alerts_info.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TTracesWatcherConfig::TFactory::TRegistrator<TTracesWatcherConfig> TTracesWatcherConfig::Registrator("traces_watcher");

bool TTracesWatcher::Deserialize(const TBlob& data) {
    InfoById.clear();
    TBuffer buf(data.AsCharPtr(), data.Size());
    TBufferInput bInput(buf);
    NDrive::NProto::TTraceChecker proto;
    ui32 sizeBlock = 0;
    TVector<char> bufProto;
    while (bInput.Read(&sizeBlock, sizeof(sizeBlock)) == sizeof(sizeBlock) && sizeBlock) {
        bufProto.reserve(sizeBlock);
        bInput.Read(&bufProto[0], sizeBlock);
        if (!proto.ParseFromArray(&bufProto[0], sizeBlock)) {
            ERROR_LOG << GetId() << ": cannot parse robot state" << Endl;
            return false;
        }
        TTraceAccumulator accum;
        if (!accum.DeserializeFromProto(proto)) {
            continue;
        }
        InfoById.emplace(accum.GetTraceId(), std::move(accum));
        proto.Clear();
    }
    INFO_LOG << "TTracesWatcher::Deserialize: " << InfoById.size() << Endl;
    return true;
}

TBlob TTracesWatcher::Serialize() const {
    TBuffer buf;
    TBufferOutput bOut(buf);
    for (auto&& i : InfoById) {
        NDrive::NProto::TTraceChecker proto;
        i.second.SerializeToProto(proto);
        const TString protoStr = proto.SerializeAsString();
        const ui32 sizeProto = protoStr.size();
        bOut.Write(&sizeProto, sizeof(sizeProto));
        bOut.Write(protoStr.c_str(), protoStr.size());
    }
    INFO_LOG << "TTracesWatcher::Serialize: " << InfoById.size() << Endl;
    return TBlob::FromBuffer(buf);
}

bool TTracesWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

    TTracesAccessor traces(server, Config->GetRTLineAPIName());
    TMessagesCollector message;
    if (!traces.Execute(message, Config->GetWatchPeriod())) {
        ERROR_LOG << "Cannot receive data: " << message.GetStringReport() << Endl;
        return true;
    }

    TWatchingInfo wInfo(traces, Config->GetAlertsConfig());
    if (!wInfo.Initialize()) {
        ERROR_LOG << "Cannot fetch data for incorrect traces detection" << Endl;
        return true;
    } else {
        TSet<TString> idsCurrent;
        for (auto&& i : wInfo.GetAlerts()) {
            idsCurrent.insert(i.GetTraceId());
            auto it = InfoById.emplace(i.GetTraceId(), i.GetTraceId()).first;
            it->second.Actualize(true);
            it->second.SetAlertData(i);
        }

        for (auto&& [id, info] : InfoById) {
            if (!idsCurrent.contains(id)) {
                info.Actualize(false);
            }
        }

        auto carsManager = server->GetDriveAPI()->GetCarsData();
        if (!carsManager) {
            ERROR_LOG << GetId() << ": Need cars db config" << Endl;
            return true;
        }

        auto session = server->GetDriveAPI()->BuildSession(Config->GetDryRunMode());
        for (auto it = InfoById.begin(); it != InfoById.end(); ) {
            if (it->second.GetState() == EAccumulatorState::Pause) {
                if (it->second.GetFirstCheckInstant() + Config->GetDurationTraceCheck() + Config->GetDurationTraceCheckPause() < ModelingNow()) {
                    it = InfoById.erase(it);
                } else {
                    ++it;
                }
                continue;
            }

            if (it->second.GetFirstCheckInstant() + Config->GetDurationTraceCheck() >= ModelingNow()) {
                ++it;
                continue;
            }

            if (it->second.GetReliability() >= Config->GetReliabilityForAlert() && it->second.HasAlertData()) {
                it->second.SetState(EAccumulatorState::Pause);
                const TAlertInfo& aInfo = it->second.GetAlertData();
                if (aInfo.GetDeviceId()) {
                    auto fechedCars = carsManager->FetchInfo(aInfo.GetDeviceId(), session);
                    if (!fechedCars) {
                        ERROR_LOG << GetId() << ": Fail to fetch cars info: " << session.GetStringReport() << Endl;
                        return true;
                    }
                    if (fechedCars.empty()) {
                        WARNING_LOG << GetId() << ": skip unknown car(" << aInfo.GetDeviceId() << ")" << Endl;
                        continue;
                    }
                } else {
                    WARNING_LOG << GetId() << ": skip empty car data" << Endl;
                    continue;
                }
                TVector<IAlertTag::TPtr> tags = aInfo.ConstructTags();
                if (aInfo.IsCarTag()) {
                    for (auto&& tag : tags) {
                        TVector<TDBTag> conflictingTags;
                        if (tag->HasConflictingTags()) {
                            if (!manager.GetDeviceTags().RestoreEntityTags(aInfo.GetDeviceId(), tag->GetConflictingTags(), conflictingTags, session)) {
                                ERROR_LOG << "Cannot restore tags for: " << aInfo.GetDeviceId() << Endl;
                                return false;
                            }
                        }
                        DEBUG_LOG << "Car " << aInfo.GetDeviceId() << " has " << conflictingTags.size() << " conflicting tags: " << JoinSeq(",", tag->GetConflictingTags()) << " " << tag->HasConflictingTags() << " with " << tag->GetName() << Endl;
                        if (conflictingTags.empty()) {
                            DEBUG_LOG << "Adding tag " << tag->GetName() << " on car " << aInfo.GetDeviceId() << Endl;
                            if (!Config->GetDryRunMode()) {
                                if (!manager.GetDeviceTags().AddTag(tag, "robot_trace_watcher", aInfo.GetDeviceId(), server, session)) {
                                    ERROR_LOG << "Error on add device tag: " << session.GetReport() << Endl;
                                    return true;
                                }
                            }
                        } else {
                            DEBUG_LOG << "Adding tag " << tag->GetName() << " on car " << aInfo.GetDeviceId() << " conflicts with existing tag: "
                                << conflictingTags.front()->GetName() << "; skip adding" << Endl;
                        }
                    }
                } else {
                    for (auto&& tag : tags) {
                        DEBUG_LOG << "Adding tag " << tag->GetName() << " on user " << aInfo.GetUserId() << Endl;
                        if (!Config->GetDryRunMode()) {
                            if (!manager.GetTraceTags().AddTag(tag, aInfo.GetUserId(), aInfo.GetTraceId(), server, session)) {
                                ERROR_LOG << "Error on add trace tag: " << session.GetReport() << Endl;
                                return true;
                            }
                        }
                    }
                }
                TVector<TString> tagsDescriptions;
                for (auto&& tag : tags) {
                    tagsDescriptions.push_back(tag->GetHRDescription());
                }
                INFO_LOG << "setting " << JoinStrings(tagsDescriptions, ",") << " for " << aInfo.GetReport().GetStringRobust() << Endl;
                ++it;
            } else {
                it = InfoById.erase(it);
            }
        }
        if (!session.Commit()) {
            ERROR_LOG << session.GetStringReport() << Endl;
        }
    }

    DEBUG_LOG << "Background process TTracesWatcher iteration finished" << Endl;
    return true;
}

IBackgroundProcess* TTracesWatcherConfig::Construct() const {
    return new TTracesWatcher(this);
}
