#pragma once
#include <drive/backend/background/manager/regular.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/data/alerts/alerts_info.h>
#include <drive/backend/data/alerts/config.h>
#include <drive/backend/data/proto/alerts.pb.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/util/network/neh.h>

class TTracesWatcherConfig: public IBackgroundRegularProcessConfig {
private:
    TString RTLineAPIName;
    TAlertsConfig AlertsConfig;
    TDuration WatchPeriod = TDuration::Minutes(5);

    TDuration DurationTraceCheck = TDuration::Minutes(1);
    TDuration DurationTraceCheckPause = TDuration::Minutes(4);

    double ReliabilityForAlert = 0.7;

    static TFactory::TRegistrator<TTracesWatcherConfig> Registrator;
public:

    using IBackgroundRegularProcessConfig::IBackgroundRegularProcessConfig;

    TDuration GetDurationTraceCheck() const {
        return DurationTraceCheck;
    }

    TDuration GetDurationTraceCheckPause() const {
        return DurationTraceCheckPause;
    }

    double GetReliabilityForAlert() const {
        return ReliabilityForAlert;
    }

    TDuration GetWatchPeriod() const {
        return WatchPeriod;
    }

    const TString& GetRTLineAPIName() const {
        return RTLineAPIName;
    }

    const TAlertsConfig& GetAlertsConfig() const {
        return AlertsConfig;
    }

    virtual IBackgroundProcess* Construct() const override;

    virtual void Init(const TYandexConfig::Section* section) override {
        IBackgroundRegularProcessConfig::Init(section);
        AlertsConfig.Init(section);
        RTLineAPIName = section->GetDirectives().Value("RTLineAPIName", RTLineAPIName);
        WatchPeriod = section->GetDirectives().Value("WatchPeriod", WatchPeriod);

        DurationTraceCheck = section->GetDirectives().Value("DurationTraceCheck", DurationTraceCheck);
        DurationTraceCheckPause = section->GetDirectives().Value("DurationTraceCheckPause", DurationTraceCheckPause);
        ReliabilityForAlert = section->GetDirectives().Value("ReliabilityForAlert", ReliabilityForAlert);
        CHECK_WITH_LOG(GetPeriod() <= DurationTraceCheck);
    }

    virtual void ToString(IOutputStream& os) const override {
        IBackgroundRegularProcessConfig::ToString(os);
        AlertsConfig.ToString(os);
        os << "RTLineAPIName: " << RTLineAPIName << Endl;
        os << "WatchPeriod: " << WatchPeriod << Endl;

        os << "DurationTraceCheck: " << DurationTraceCheck << Endl;
        os << "DurationTraceCheckPause: " << DurationTraceCheckPause << Endl;
        os << "ReliabilityForAlert: " << ReliabilityForAlert << Endl;
    }
};

class TTracesWatcher: public IBackgroundRegularProcessImpl<NDrive::IServer> {
private:
    using TBase = IBackgroundRegularProcessImpl<NDrive::IServer>;
    enum class EAccumulatorState {
        Accumulate = 0,
        Pause = 1
    };

    class TTraceAccumulator {
    private:
        TString TraceId;
        TInstant FirstCheckInstant = Now();
        ui32 ActualSignal = 0;
        ui32 NoActualSignal = 0;
        EAccumulatorState State = EAccumulatorState::Accumulate;
        THolder<TAlertInfo> AlertData;
    public:

        bool HasAlertData() const {
            return !!AlertData;
        }

        TAlertInfo& GetAlertData() const {
            CHECK_WITH_LOG(!!AlertData);
            return *AlertData;
        }

        TTraceAccumulator& SetAlertData(const TAlertInfo& data) {
            AlertData.Reset(new TAlertInfo(data));
            return *this;
        }

        EAccumulatorState GetState() const {
            return State;
        }

        TTraceAccumulator& SetState(const EAccumulatorState value) {
            CHECK_WITH_LOG(State < value);
            State = value;
            return *this;
        }

        TTraceAccumulator() = default;

        TTraceAccumulator(const TString& traceId)
            : TraceId(traceId)
        {

        }

        void Actualize(const bool isActual) {
            if (isActual) {
                ++ActualSignal;
            } else {
                ++NoActualSignal;
            }
        }

        double GetReliability() const {
            if (NoActualSignal + ActualSignal == 0) {
                return 0;
            }
            return (1.0 * ActualSignal) / (NoActualSignal + ActualSignal);
        }

        void SerializeToProto(NDrive::NProto::TTraceChecker& proto) const {
            proto.SetFirstCheckTimestamp(FirstCheckInstant.Seconds());
            proto.SetTraceId(TraceId);
            proto.SetActualSignal(ActualSignal);
            proto.SetNoActualSignal(NoActualSignal);
            proto.SetState((ui32)State);
        }

        bool DeserializeFromProto(const NDrive::NProto::TTraceChecker& proto) {
            FirstCheckInstant = TInstant::Seconds(proto.GetFirstCheckTimestamp());
            TraceId = proto.GetTraceId();
            ActualSignal = proto.GetActualSignal();
            NoActualSignal = proto.GetNoActualSignal();
            State = (EAccumulatorState)proto.GetState();
            return true;
        }

        const TInstant& GetFirstCheckInstant() const {
            return FirstCheckInstant;
        }

        const TString& GetTraceId() const {
            return TraceId;
        }

    };

    virtual bool Deserialize(const TBlob& data) override;
    virtual TBlob Serialize() const override;

    TDuration Period;
    const TTracesWatcherConfig* Config = nullptr;
    mutable TMap<TString, TTraceAccumulator> InfoById;

protected:
    bool DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const override;

public:
    TTracesWatcher(const TTracesWatcherConfig* config)
        : TBase(*config)
        , Config(config) {
    }

    virtual void Stop() override {
    }

    virtual void Start() override {
    }

};
