#include "processor.h"

#include "config.h"

#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

bool TBanActualizer::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const auto api = Yensured(server->GetDriveAPI());
    const auto& userData = *Yensured(api->GetUsersData());
    auto robotUserId = GetRobotUserId(server);

    TVector<TDBTag> dbTags;
    TMap<TString, ui32> tagPoints;
    {
        TVector<TString> tagNames;
        auto allTags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
        for (auto tdIt : allTags) {
            const auto description = tdIt.second->GetAs<TUserProblemTag::TDescription>();
            if (!!description) {
                tagPoints[tdIt.first] = description->GetPoint();
                tagNames.push_back(tdIt.first);
            }
        }
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, tagNames, dbTags, session)) {
            ERROR_LOG << "BAN ACTUALIZER FAILED: can't restore user tags" << Endl;
            return true;
        }
    }

    TSet<TString> bannedUsers;
    {
        auto queryOption = NSQL::TQueryOptions()
            .AddGenericCondition("status", NDrive::UserStatusBlocked)
        ;
        auto session = userData.BuildSession(true);
        auto optionalUsers = userData.Fetch(session, queryOption);
        if (!optionalUsers) {
            ERROR_LOG << GetId() << ": cannot get banned users: " << session.GetStringReport() << Endl;
            return true;
        }
        for (auto&& user : *optionalUsers) {
            bannedUsers.insert(user.GetUserId());
        }
    }
    TMap<TString, ui32> userToPoints;
    for (auto&& userId : bannedUsers) {
        userToPoints[userId] = 0;
    }
    for (auto&& tag : dbTags) {
        auto userProblemTag = tag.GetTagAs<TUserProblemTag>();
        if (userProblemTag) {
            userToPoints[tag.GetObjectId()] += tagPoints[userProblemTag->GetName()];
        }
    }

    if (!server->GetUserRegistrationManager()) {
        ERROR_LOG << "no registration manager configured" << Endl;
        return false;
    }
    auto registrar = server->GetUserRegistrationManager();

    if (!registrar) {
        ERROR_LOG << "BAN ACTUALIZER FAILED: no registration manager configured" << Endl;
        return false;
    }
    auto banThreshold = registrar->GetBanThreshold();

    TSet<TString> banCandidates;

    for (auto&& userIt : userToPoints) {
        bool userIsBanned = bannedUsers.contains(userIt.first);
        bool userShouldBeBanned = userIt.second >= banThreshold;
        if (userIsBanned && !userShouldBeBanned) {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!registrar->UnbanUser(userIt.first, robotUserId, session) || !session.Commit()) {
                ERROR_LOG << GetId() << ": BAN ACTUALIZER ACTUALIZATION FAILED: unban " << userIt.first << ": " << session.GetStringReport() << Endl;
            }
            INFO_LOG << "BAN ACTUALIZER is unbanning user " << userIt.first << Endl;
        } else if (!userIsBanned && userShouldBeBanned) {
            banCandidates.insert(userIt.first);
        }
    }

    TVector<TDBTag> performedTagsByBanCandidates;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        TVector<TString> banCandidatesVector;
        for (auto&& candidate : banCandidates) {
            banCandidatesVector.emplace_back(candidate);
        }
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestorePerformerTags(banCandidatesVector, performedTagsByBanCandidates, session)) {
            ERROR_LOG << "BAN ACTUALIZER can't restore performed tags" << Endl;
            return false;
        }
    }

    for (auto&& tag : performedTagsByBanCandidates) {
        auto userId = tag->GetPerformer();
        auto banCandidateIt = banCandidates.find(userId);
        if (banCandidateIt != banCandidates.end()) {
            banCandidates.erase(banCandidateIt);
        }
    }

    for (auto&& userId : banCandidates) {
        auto session = userData.BuildTx<NSQL::Writable>();
        auto usersFetchResult = userData.FetchInfo(userId, session);
        auto userPtr = usersFetchResult.GetResultPtr(userId);
        if (!userPtr || userPtr->GetStatus() != NDrive::UserStatusActive) {
            continue;
        }

        if (!registrar->BanUser(userId, robotUserId, NBans::EReason::Auto, session)) {
            ERROR_LOG << GetId() << ": BAN ACTUALIZER ACTUALIZATION FAILED: ban " << userId << ": " << session.GetStringReport() << Endl;
            continue;
        } else {
            INFO_LOG << "BAN ACTUALIZER banned user: " << userId << Endl;
        }
        if (!session.Commit()) {
            ERROR_LOG << GetId() << ": banning " << userId << ": cannot commit session: " << session.GetStringReport() << Endl;
        }
    }

    return true;
}

TBanActualizer::TBanActualizer(const TBanActualizerConfig* config)
    : TBase(*config)
{
}
