#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/yt/common/writer.h>

void TBillingWatcher::SerializeToProto(NDrive::NProto::TEmptyProcessorData& /*proto*/) const {
}

bool TBillingWatcher::DeserializeFromProto(const NDrive::NProto::TEmptyProcessorData& /*proto*/) {
    return true;
}

bool TBillingWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    ui32 tasksAtOnce = server->GetSettings().GetValueDef<ui32>("billing.tasks_at_once", Config.GetTasksAtOnce());
    server->GetDriveAPI()->GetBillingManager().WaitBillingCycle(tasksAtOnce, 1);
    return true;
}

TBillingWatcher::TBillingWatcher(const TBillingWatcherConfig& config)
    : TBase(config)
    , Config(config)
{
}

void TRefundWatcher::SerializeToProto(NDrive::NProto::TEmptyProcessorData& /*proto*/) const {
    INFO_LOG << "TRefundWatcher::SerializeToProto" << Endl;
}

bool TRefundWatcher::DeserializeFromProto(const NDrive::NProto::TEmptyProcessorData& /*proto*/) {
    INFO_LOG << "TRefundWatcher::DeserializeFromProto" << Endl;
    return true;
}

bool TRefundWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    bool billingActiveFlag;
    if (server->GetSettings().GetValue("billing_active", billingActiveFlag) && !billingActiveFlag) {
        return true;
    }
    server->GetDriveAPI()->GetBillingManager().WaitRefundCycle();
    return true;
}

TRefundWatcher::TRefundWatcher(const TRefundWatcherConfig& config)
    : TBase(config)
    , Config(config)
{
    Y_UNUSED(Config);
}

void TBillingQueueSwitcher::SerializeToProto(NDrive::NProto::TEmptyProcessorData& /*proto*/) const {
}

bool TBillingQueueSwitcher::DeserializeFromProto(const NDrive::NProto::TEmptyProcessorData& /*proto*/) {
    return true;
}

void TBillingQueueSwitcher::Stop() {
}

void TBillingQueueSwitcher::Start() {
}

TMap<TString, TPaymentsData> GetSessionsPayments(const TBillingManager& billingManager, const TBillingQueueSwitcher::EQueueField field, const EBillingQueue skipQueue, TSet<EBillingQueue> targetQueues) {
    auto session = billingManager.BuildSession(true);
    NSQL::TQueryOptions options;
    if (!targetQueues.empty()) {
        auto it = targetQueues.find(skipQueue);
        if (it != targetQueues.end()) {
            targetQueues.erase(it);
        }
        if (targetQueues.empty()) {
            return {};
        }
        TSet<TString> queues;
        for (auto&& queue : targetQueues) {
            queues.emplace(ToString(queue));
        }
        options.SetGenericCondition(ToString(field), queues);
    } else {
        TSet<TString> queues = { ToString(skipQueue) };
        options.SetGenericCondition(ToString(field), NSQL::TNot<TSet<TString>>(queues));
    }

    auto optionalTasks = billingManager.GetActiveTasksManager().GetTasks(session, options.PrintConditions(*session.GetTransaction()));
    if (!optionalTasks) {
        ERROR_LOG << "Task select fails TRTBillingQueueSwitcher::GetBillingTasks : " << session.GetStringReport() << Endl;
        return {};
    }
    auto payments = billingManager.GetPaymentsManager().GetSessionsPayments(*optionalTasks, session, true);
    if (!payments) {
        return {};
    }
    return *payments;
}

void TBillingQueueSwitcher::SoftSwitchQueue(const TBillingManager& billingManager, const TSet<EBillingQueue>& queues, const TString& userId, const ui32 tasksAtOnce, const ui32 minPriority) const {
    if (queues.empty()) {
        return;
    }
    const EBillingQueue currentQueue = billingManager.GetConfig().GetActiveQueue();
    ui32 tasksCount = 0;
    auto actuality = ModelingNow();
    auto payments = GetSessionsPayments(billingManager, EQueueField::NextQueue, currentQueue, queues);
    for (auto&& [sessionId, task] : payments) {
        if (tasksCount >= tasksAtOnce) {
            break;
        }
        if (!billingManager.NeedIgnoreByDiscretization(task.GetSnapshot(), task.GetBillingTask()) && (ui32)billingManager.GetTaskPriority(task, actuality) < minPriority) {
            continue;
        }
        if (billingManager.SwitchQueue(task.GetSnapshot(), task.GetBillingTask(), userId, {}, currentQueue)) {
            tasksCount++;
        }
    }
    TUnistatSignalsCache::SignalAdd("billing-queue-switcher", "switched", tasksCount);
}

void TBillingQueueSwitcher::HardSwitchQueue(const TBillingManager& billingManager, const TSet<EBillingQueue>& queues, const TString& userId, const ui32 tasksAtOnce, const ui32 minPriority) const {
    if (queues.empty()) {
        return;
    }
    const EBillingQueue currentQueue = billingManager.GetConfig().GetActiveQueue();
    ui32 tasksCount = 0;
    auto actuality = ModelingNow();
    auto payments = GetSessionsPayments(billingManager, EQueueField::Queue, currentQueue, queues);
    for (auto&& [sessionId, task] : payments) {
        if (tasksCount >= tasksAtOnce) {
            break;
        }
        if (!billingManager.NeedIgnoreByDiscretization(task.GetSnapshot(), task.GetBillingTask()) && (ui32)billingManager.GetTaskPriority(task, actuality) < minPriority) {
            continue;
        }
        if (billingManager.SwitchQueue(task.GetSnapshot(), task.GetBillingTask(), userId, currentQueue, currentQueue)) {
            tasksCount++;
        }
    }
    TUnistatSignalsCache::SignalAdd("billing-queue-switcher", "switched-forcibly", tasksCount);
}

TSet<EBillingQueue> TBillingQueueSwitcher::GetQueues(const NDrive::IServer* server, const TString& settingName) const {
    TSet<EBillingQueue> result;
    auto stringQueues = SplitString(server->GetSettings().GetValueDef<TString>(settingName, ""), ",");
    for(auto& item : stringQueues) {
        EBillingQueue queue = EBillingQueue::Prestable;
        if (TryFromString<EBillingQueue>(item, queue)) {
            result.emplace(queue);
        }
    }
    return result;
}

bool TBillingQueueSwitcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const ui32 tasksAtOnce = server->GetSettings().GetValueDef<ui32>("billing.queue_switcher.tasks_at_once", Config.GetTasksAtOnce());
    const auto& manager = server->GetDriveAPI()->GetBillingManager();
    const auto softQueues = GetQueues(server, "billing.queue_switcher.soft_update_queues");
    ui32 minPriority = server->GetSettings().GetValueDef<ui32>("billing.min_task_priority", (ui32)TBillingLogic::ETaskPriority::RegularDebts);
    minPriority = server->GetSettings().GetValueDef<ui32>("billing.queue_switcher.min_task_priority", minPriority);
    const TString robotUserId = GetRobotUserId(server);
    SoftSwitchQueue(manager, softQueues, robotUserId, tasksAtOnce, minPriority);
    const auto hardQueues = GetQueues(server, "billing.queue_switcher.hard_update_queues");
    HardSwitchQueue(manager, hardQueues, robotUserId, tasksAtOnce, minPriority);
    return true;
}

TBillingQueueSwitcher::TBillingQueueSwitcher(const TBillingQueueSwitcherConfig& config)
    : TBase(config)
    , Config(config)
{
}

