#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <search/fetcher/fetcher.h>

#include <util/string/subst.h>

bool TCallerTelegramBotProcessor::NormalizePhoneNumber(const TString& number, TString& result) {
    bool prefix = true;
    result = "9";
    for (size_t i = 0; i < number.size(); ++i) {
        if (prefix) {
            if (number.at(i) == ' ' || number.at(i) == '+') {
                continue;
            }
            if (number.at(i) >= '0' && number.at(i) <= '9') {
                if (number.at(i) == '7') {
                    result += '8';
                } else {
                    result += number.at(i);
                }
                prefix = false;
            } else {
                DEBUG_LOG << "Invalid phone '" << number << "': wrong symbol at pos. " << i << Endl;
                return false;
            }
        } else {
            if (number.at(i) == ' ' || number.at(i) == '-') {
                continue;
            } else if (number.at(i) >= '0' && number.at(i) <= '9') {
                result += number.at(i);
            } else {
                DEBUG_LOG << "Invalid phone '" << number << "': wrong symbol at pos. " << i << Endl;
                return false;
            }
        }
    }

    if (result.size() + 1 != sizeof("981112223344")) {
        DEBUG_LOG << "Invalid phone '" << number << "': wrong result length '" << result << "': " << result.size() << Endl;
        return false;
    }
    return true;
}

bool TCallerTelegramBotProcessor::ProcessCommand(const TBotCommand& command, const TString& chatId, const IServerBase* /*server*/) {
    if (command.Is("call")) {
        if (command.GetArgs().empty()) {
            Bot->Notify(NDrive::INotifier::TMessage("Некорректные аргументы команды " + command.GetCommand()), chatId);
            return false;
        }
        TString calling;
        if (!NormalizePhoneNumber(command.GetArg(0), calling)) {
            Bot->Notify(NDrive::INotifier::TMessage("Некорректный номер телефона: " + command.GetArg(0)), chatId);
            return false;
        }
        TCaller caller;
        if (command.GetArgs().size() > 1) {
            if (Config.GetCallers().empty()) {
                Bot->Notify(NDrive::INotifier::TMessage("Ошибка при вызове: не заданы вызывающие номера"), chatId);
                return false;
            }
            if (command.GetArg(1) == "*") {
                // random
                ui32 callerIndex = TInstant::Now().Seconds() % Config.GetCallers().size();
                caller = Config.GetCallers()[callerIndex];
            } else {
                // stub
                caller = Config.GetCallers().front();
            }
        } else {
            caller = Config.GetDefaultCaller();
        }

        TString callerUrl = Config.GetCallerUrlTmpl();
        SubstGlobal(callerUrl, "{{caller}}", caller.GetPhone());
        SubstGlobal(callerUrl, "{{caller_name}}", CGIEscapeRet(caller.GetName()));
        SubstGlobal(callerUrl, "{{calling}}", calling);
        DEBUG_LOG << "Calling: " << callerUrl << Endl;

        if (!GetDryRunMode()) {
            TSelfFlushLogFramePtr logFramePtr;
            THttpFetcher fetcher(logFramePtr, "1");
            if (fetcher.AddRequestUrl(callerUrl.data(), TDuration::Seconds(2)) < 0) {
                Bot->Notify(NDrive::INotifier::TMessage("Ошибка при cоздании запроса"), chatId);
                return false;
            }
            fetcher.Run(THttpFetcher::TAbortOnTimeout());

            const IRemoteRequestResult* result = fetcher.GetRequestResult(0);
            if (!result) {
                Bot->Notify(NDrive::INotifier::TMessage("Ошибка при запросе: пустой ответ"), chatId);
                return false;
            }
            if (result->StatusCode() != 200) {
                Bot->Notify(NDrive::INotifier::TMessage("Ошибка при запросе, статус: " + ToString(result->StatusCode())), chatId);
                return false;
            }
        }
        TStringStream ss;
        ss << "Вызываю " << calling << " с номера " << caller.GetPhone() << " (" << caller.GetName() << ")";
        if (GetDryRunMode()) {
            ss << " (dry run mode)";
        }
        Bot->Notify(NDrive::INotifier::TMessage(ss.Str()), chatId);

        return true;
    }

    return false;
}
