#include "processor.h"

#include "config.h"

#include <drive/backend/background/common/common.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/events.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/tags/tags.h>

#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    enum class ETagAction {
        Add,
        Unknown
    };
}

void TCarMarkers::SerializeToProto(NDrive::NProto::TDeviceProblemsState& /*proto*/) const {
    INFO_LOG << "TCarMarkers::SerializeToProto" << Endl;
}

bool TCarMarkers::DeserializeFromProto(const NDrive::NProto::TDeviceProblemsState& /*proto*/) {
    INFO_LOG << "TCarMarkers::DeserializeFromProto" << Endl;
    return true;
}

class TTagSyncInfo {
    R_FIELD(ETagAction, Action, ETagAction::Unknown);
    R_FIELD(TString, Comment);
public:
    TTagSyncInfo() = default;
    TTagSyncInfo(const ETagAction action)
        : Action(action) {

    }
    TTagSyncInfo(const ETagAction action, const TString& comment)
        : Action(action)
        , Comment(comment) {

    }
};

bool UpdateRidingTagForObjects(const TVector<TDBTag>& tags, const TString& robotUserId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto conditionBuilder = [](const TDBTag& tag) -> NStorage::TTableRecord {
        return NStorage::TRecordBuilder
            ("tag_id", tag.GetTagId())
            ("tag", tag->GetName())
        ;
    };

    auto updateBuilder = [](const TDBTag& tag) -> NStorage::TTableRecord {
        return NStorage::TRecordBuilder
            ("snapshot", tag->GetStringSnapshot())
        ;
    };

    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagsExtCondition(tags, robotUserId, conditionBuilder, updateBuilder, session)) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }

    return true;
}

bool SyncTagForObjects(TMap<TString, TTagSyncInfo>&& carIds, const TString& robotUserId, const NDrive::IServer* server, const TString& tagName, const TVector<TDBTag>& currentTags = TVector<TDBTag>()) {
    auto td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
    if (!td) {
        ERROR_LOG << "Incorrect tag name: " << tagName << Endl;
        return false;
    }
    NDrive::ITag::TPtr tag = NDrive::ITag::TFactory::Construct(td->GetType());
    if (!tag) {
        ERROR_LOG << "Incorrect tag type: " << td->GetType() << Endl;
        return false;
    }
    tag->SetName(tagName);

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    TVector<TDBTag> tags;
    if (!currentTags.empty()) {
        tags = currentTags;
    } else {
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {tagName}, tags, session)) {
            return false;
        }
    }
    TVector<TDBTag> tagsRemove;
    for (auto&& i : tags) {
        auto it = carIds.find(i.GetObjectId());
        if (it == carIds.end()) {
            i->SetObjectSnapshot(nullptr);
            tagsRemove.emplace_back(i);
        } else {
            carIds.erase(it);
        }
    }
    for (auto&& i : carIds) {
        if (i.second.GetAction() != ETagAction::Add) {
            continue;
        }
//        tag->SetComment(i.second.GetComment());
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, robotUserId, i.first, server, session, EUniquePolicy::SkipIfExists)) {
            ERROR_LOG << session.GetStringReport() << Endl;
            return false;
        }
    }
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTags(tagsRemove, robotUserId, server, session, false)) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }
    if (!session.Commit()) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return false;
    }
    return true;
}

bool TCarMarkers::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    auto robotUserId = GetRobotUserId(server);
    auto snapshots = server->GetSnapshotsManager().GetSnapshots();
    const TSet<TString> ridingTagsCorrection = Config->GetRidingTagsCorrection();
    if (!ridingTagsCorrection.empty()) {
        auto builder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
        auto sessions = builder->GetSessionsActual();
        TVector<TDBTag> ridingTags;
        for (auto&& i : sessions) {
            TMaybe<TCarTagHistoryEvent> evLast = i->GetLastEvent();
            if (!evLast) {
                continue;
            }
            if ((*evLast)->GetName() != "old_state_riding") {
                continue;
            }
            NDrive::IObjectSnapshot::TPtr dsPtr = snapshots.GetSnapshotPtr(evLast->TConstDBTag::GetObjectId());
            const THistoryDeviceSnapshot* ds = snapshots.GetSnapshot(evLast->TConstDBTag::GetObjectId());
            const THistoryDeviceSnapshot* currentDs = (*evLast)->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (ds && currentDs) {
                TSet<TString> currentIntersection;
                SetIntersection(currentDs->GetLocationTagsArray().begin(), currentDs->GetLocationTagsArray().end(),
                    ridingTagsCorrection.begin(), ridingTagsCorrection.end(),
                    std::inserter(currentIntersection, currentIntersection.begin()));

                TSet<TString> pointIntersection;
                SetIntersection(ds->GetLocationTagsArray().begin(), ds->GetLocationTagsArray().end(),
                    ridingTagsCorrection.begin(), ridingTagsCorrection.end(),
                    std::inserter(pointIntersection, pointIntersection.begin()));

                TSet<TString> removedTags;
                SetDifference(currentIntersection.begin(), currentIntersection.end(),
                    pointIntersection.begin(), pointIntersection.end(),
                    std::inserter(removedTags, removedTags.begin()));

                TSet<TString> newTags;
                SetDifference(pointIntersection.begin(), pointIntersection.end(),
                    currentIntersection.begin(), currentIntersection.end(),
                    std::inserter(newTags, newTags.begin()));

                if (!removedTags.empty() || !newTags.empty()) {
                    TDBTag copyTag = evLast->Clone(server->GetDriveAPI()->GetTagsHistoryContext());
                    if (!!copyTag) {
                        ridingTags.emplace_back(std::move(copyTag));
                        ridingTags.back()->SetObjectSnapshot(dsPtr);
                    } else {
                        ERROR_LOG << GetId() << ": Clone failed for " << (*evLast)->GetName() << Endl;
                    }
                }

            } else {
                ERROR_LOG << "Incorrect snapshots for object " << evLast->TConstDBTag::GetObjectId() << Endl;
            }
        }
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!UpdateRidingTagForObjects(ridingTags, robotUserId, server, session) || !session.Commit()) {
            ERROR_LOG << "cannot do shite: " << session.GetStringReport() << Endl;
        }
    }
    if (!!Config->GetFullTankTagName()) {
        TMap<TString, TTagSyncInfo> carIdsFuel;
        for (auto&& i : snapshots) {
            double fuelLevel;
            if (i.second.GetFuelLevel(fuelLevel, Config->GetSensorFreshness())) {
                if (fuelLevel >= Config->GetFuelLevel()) {
                    TTagSyncInfo info(ETagAction::Add, "fuel level: " + ToString(fuelLevel));
                    carIdsFuel.emplace(i.first, info);
                } else if (fuelLevel >= Config->GetFuelLevel() - Config->GetSensorPrecision()) {
                    carIdsFuel.emplace(i.first, ETagAction::Unknown);
                }
            } else {
                carIdsFuel.emplace(i.first, ETagAction::Unknown);
            }
        }

        SyncTagForObjects(std::move(carIdsFuel), robotUserId, server, Config->GetFullTankTagName());
    }

    if (auto pusher = server->GetNotifier(Config->GetPackOfferPushNotifier())) {
        const TDriveAPI* api = server->GetDriveAPI();
        const TDeviceTagsManager& tags = api->GetTagsManager().GetDeviceTags();

        TString distancePushContentTemplate;
        TString durationPushContentTemplate;
        {
            auto session = tags.BuildSession(true);
            if (!server->GetSettings().GetValue(Config->GetPackOfferDistanceThresholdPushTemplate(), distancePushContentTemplate)) {
                WARNING_LOG << "cannot acquire value " << Config->GetPackOfferDistanceThresholdPushTemplate() << " from settings" << Endl;
            }
            if (!server->GetSettings().GetValue(Config->GetPackOfferDurationThresholdPushTemplate(), durationPushContentTemplate)) {
                WARNING_LOG << "cannot acquire value " << Config->GetPackOfferDurationThresholdPushTemplate() << " from settings" << Endl;
            }
        }
        TString distancePushTagName = server->GetSettings().GetValue<TString>("pack_offer.distance_threshold.push_tag_name").GetOrElse({});
        TString durationPushTagName = server->GetSettings().GetValue<TString>("pack_offer.duration_threshold.push_tag_name").GetOrElse({});

        auto builder = tags.GetHistoryManager().GetSessionsBuilderSafe("billing", Now());
        Y_ENSURE(builder);
        auto sessions = builder->GetSessionsActualByObjects();
        for (auto&&[objectId, billingSession] : sessions) {
            if (!billingSession) {
                DEBUG_LOG << "empty session for " << objectId << Endl;
                continue;
            }
            if (billingSession->GetClosed()) {
                DEBUG_LOG << "closed session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }

            auto compilation = billingSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
            if (!compilation) {
                ERROR_LOG << "no default compilation in session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }

            auto offer = compilation->GetCurrentOffer();
            if (!offer) {
                ERROR_LOG << "no offer in default compilation in session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }
            auto packOffer = dynamic_cast<TPackOffer*>(offer.Get());
            if (!packOffer) {
                continue;
            }
            auto fixPointOffer = dynamic_cast<TFixPointOffer*>(offer.Get());

            auto state = compilation->GetCurrentOfferState();
            if (!state) {
                ERROR_LOG << "no offer state in default compilation in session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }
            auto packOfferState = dynamic_cast<TPackOfferState*>(state.Get());
            if (!packOfferState) {
                ERROR_LOG << "unable to cast offer state in default compilation in session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }

            auto now = Now();
            auto snapshot = MakeAtomicShared<TEventsSnapshot>();

            bool sendPushDistance = false;
            if (packOffer->GetDistancePushThreshold()) {
                const double distance = packOffer->GetMileageLimit() - packOfferState->GetRemainingDistance();
                sendPushDistance = distance > packOffer->GetDistancePushThreshold();
            } else if (packOffer->GetRemainingDistancePushThreshold()) {
                sendPushDistance = packOfferState->GetRemainingDistance() < packOffer->GetRemainingDistancePushThreshold();
            }

            auto locale = offer->GetLocale();
            TString distancePushContent = distancePushContentTemplate;
            if (sendPushDistance &&
                !packOfferState->IsDistanceThresholdPushSent() &&
                packOfferState->GetRemainingDistance() > 0 &&
                distancePushContent
            ) {
                SubstGlobal(distancePushContent, "REMAININGDISTANCE", server->GetLocalization()->DistanceFormatKm(locale, packOfferState->GetRemainingDistance()));
                SubstGlobal(distancePushContent, "OVERRUNPRICE", server->GetLocalization()->FormatPrice(locale, packOffer->GetDiscountedOverrunPrice()));
                SubstGlobal(distancePushContent, "OFFERNAME", packOffer->GetName());
                TEventsSnapshot::TPushInfo info;
                info.Content = distancePushContent;
                info.Type = TPackOffer::DistanceThresholdPushName;
                info.Timestamp = now;
                snapshot->AddPush(std::move(info));
            }

            bool sendPushDuration = false;
            if (packOffer->GetDurationPushThreshold()) {
                TDuration duration = packOffer->GetDuration() - packOfferState->GetRemainingTime();
                sendPushDuration = duration > packOffer->GetDurationPushThreshold();
            } else if (packOffer->GetRemainingDurationPushThreshold()) {
                sendPushDuration = packOfferState->GetRemainingTime() < packOffer->GetRemainingDurationPushThreshold();
            }
            TString durationPushContent = durationPushContentTemplate;
            if (sendPushDuration &&
                !packOfferState->IsDurationThresholdPushSent() &&
                packOfferState->GetRemainingTime() &&
                durationPushContent
            ) {
                SubstGlobal(durationPushContent, "REMAININGDURATION", server->GetLocalization()->FormatDuration(locale, packOfferState->GetRemainingTime()));
                SubstGlobal(durationPushContent, "OVERTIMEPRICE", server->GetLocalization()->FormatPrice(locale, packOffer->GetDiscountedOvertimePrice()));
                SubstGlobal(durationPushContent, "OFFERNAME", packOffer->GetName());
                TEventsSnapshot::TPushInfo info;
                info.Content = durationPushContent;
                info.Type = TPackOffer::DurationThresholdPushName;
                info.Timestamp = now;
                snapshot->AddPush(std::move(info));
            }

            if (snapshot->GetPushes().empty()) {
                continue;
            }

            TMaybe<TConstDBTag> tag = billingSession->GetLastEvent();
            if (!tag || !(*tag)) {
                ERROR_LOG << "no last event tag in session " << billingSession->GetSessionId() << " for " << objectId << Endl;
                continue;
            }

            TDBTag copyTag = tag->Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            if (!copyTag) {
                ERROR_LOG << GetId() << ": Clone failed for  " << objectId << Endl;
                continue;
            }
            copyTag->SetObjectSnapshot(snapshot);

            auto object = api->GetCarsData()->GetObject(objectId);

            const TString& userId = copyTag->GetPerformer();
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto userFetchResult = api->GetUsersData()->FetchInfo(userId, session);
            const TDriveUserData* userInfo = userFetchResult.GetResultPtr(userId);
            if (!userInfo) {
                ERROR_LOG << "cannot fetch user info for user_id " << userId << Endl;
                continue;
            }
            auto pushByTag = [&] {
                for (auto&& push : snapshot->GetPushes()) {
                    TString tagName;
                    if (push.Type == TPackOffer::DistanceThresholdPushName) {
                        tagName = distancePushTagName;
                    } else if (push.Type == TPackOffer::DurationThresholdPushName) {
                        tagName = durationPushTagName;
                    } else {
                        ERROR_LOG << "cannot determine tag name for " << push.Type << Endl;
                        return false;
                    }
                    auto tag = api->GetTagsManager().GetTagsMeta().CreateTag(tagName);
                    if (!tag) {
                        ERROR_LOG << "cannot create tag " << tagName << Endl;
                        return false;
                    }
                    auto pushTag = std::dynamic_pointer_cast<TUserPushTag>(tag);
                    if (!pushTag) {
                        ERROR_LOG << "cannot cast tag " << tagName << " to UserPushTag" << Endl;
                        return false;
                    }
                    if (!fixPointOffer) {
                        auto additionalInfo = NJson::TJsonValue(NJson::JSON_MAP);
                        additionalInfo["status"] = "extend_pack";
                        additionalInfo["number"] = object ? object->GetNumber() : "unknown";
                        pushTag->SetAdditionalInfo(std::move(additionalInfo));
                    }
                    pushTag->SetMessageText(push.Content);
                    if (!api->GetTagsManager().GetUserTags().AddTag(tag, GetId(), userId, server, session)) {
                        ERROR_LOG << "cannot add tag " << tagName << " to " << userId << ": " << session.GetStringReport() << Endl;
                        return false;
                    }
                }
                return true;
            };
            if (distancePushTagName && durationPushTagName) {
                if (!pushByTag()) {
                    continue;
                }
            } else {
                for (auto&& push : snapshot->GetPushes()) {
                    NDrive::INotifier::TMessage message(push.Content);
                    NDrive::INotifier::TResult::TPtr notification = pusher->Notify(message, NDrive::INotifier::TContext().SetRecipients({ *userInfo }));
                    if (notification) {
                        NOTICE_LOG << "sent push " << push.Type << " to " << userId << "in session " << billingSession->GetSessionId() << Endl;
                    } else {
                        ERROR_LOG << "cannot send push " << push.Type << " to " << userId << "in session " << billingSession->GetSessionId() << Endl;
                    }
                }
            }
            if (!UpdateRidingTagForObjects({copyTag}, robotUserId, server, session)) {
                ERROR_LOG << "cannot update tag " << copyTag.GetTagId() << Endl;
                continue;
            }
            if (!session.Commit()) {
                ERROR_LOG << "cannot commit transaction: " << session.GetStringReport() << Endl;
                continue;
            }
        }
    } else {
        INFO_LOG << "PackOffer pusher is not set " << Config->GetPackOfferPushNotifier() << Endl;
    }

    return true;
}

TCarMarkers::TCarMarkers(const TCarMarkersConfig* config)
    : TBase(*config)
    , Config(config)
{
}
