#include "processor.h"

#include "config.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>

TMaybe<TVector<TString>> TDBServiceProcessor::GetPerformedTagIds(const TString& tagTable, NDrive::TEntitySession& session) const {
    TRecordsSet records;
    auto query = TStringBuilder() << "SELECT tag_id FROM " << tagTable << " WHERE performer IS NOT NULL AND performer != ''";
    auto queryResult = session->Exec(query, &records);
    if (!queryResult || !queryResult->IsSucceed()) {
        return {};
    }
    TVector<TString> result;
    for (auto&& record : records) {
        auto tagId = record.Get("tag_id");
        if (tagId) {
            result.push_back(tagId);
        }
    }
    return result;
}

void TDBServiceProcessor::RemoveCommon(const TString& query, const TString& target, const TInstant instantBorder, const NDrive::IServer* server, const TString& primaryKey) const {
    const ui32 recordsCountLimit = 1000 * Config->GetCleanerDiscretizationRecordsCount();
    while (true) {
        TRecordsSet records;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            if (!session->Exec(
                "SELECT " + primaryKey + " FROM " + target + " WHERE "
                + query
                + " ORDER BY " + primaryKey + " LIMIT " + ::ToString(recordsCountLimit), &records)->IsSucceed()) {
                NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()), "Не получилось проверить устаревшие записи для " + target + ": " + session.GetStringReport());
                return;
            }
        }

        if (records.empty()) {
            return;
        }

        NDrive::TEventLog::Log("DbServiceDiscoverOldRecords", NJson::TMapBuilder
            ("query", query)
            ("table", target)
            ("since", NJson::ToJson(instantBorder))
            ("count", records.size())
        );

        TVector<TString> idsAll;
        TString maxId;
        for (ui32 i = 0; i < records.size(); ++i) {
            TString idLocal;
            if (!records.GetRecords()[i].TryGet(primaryKey, idLocal)) {
                NDrive::INotifier::Notify(
                    server->GetNotifier(Config->GetNotifier()),
                    "Некорректный id для удаления в " + target + ": " + records.GetRecords()[i].Get(primaryKey)
                );
                ERROR_LOG << GetId() << ": incorrect primary key " << idLocal << Endl;
                return;
            }
            idsAll.emplace_back(idLocal);
            maxId = idLocal;
        }

        NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()),
            "Планируется удалить из " + target + " : " + ::ToString(records.size()) + " записей пачками по " + ::ToString(Config->GetCleanerDiscretizationRecordsCount()) + " штук. "
            + " Максимальный ID: " + maxId);

        ui32 removedRecords = 0;
        const ui32 maxAtt = 5;
        TString maxIdRemoved;
        for (ui32 i = 0; i < idsAll.size();) {
            const ui32 next = ::Min<ui32>(i + Config->GetCleanerDiscretizationRecordsCount(), idsAll.size());
            TVector<TString> ids;
            TString maxIdRemovedNext;
            for (ui32 j = i; j < next; ++j) {
                ids.emplace_back(idsAll[j]);
                maxIdRemovedNext = idsAll[j];
            }
            INFO_LOG << "REMOVE HISTORY: " << JoinSeq(", ", ids) << Endl;
            for (ui32 att = 1; att <= maxAtt; ++att) {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!session->Exec("DELETE FROM " + target + " WHERE " + primaryKey + " IN (" + session.GetTransaction()->Quote(ids) + ")")->IsSucceed() || !session.Commit()) {
                    if (att == maxAtt) {
                        NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()),
                            "Не получилось удалить пакет старых записей истории " + target + " целиком: удалено "
                            + ::ToString(removedRecords) + " записей до id: " + maxIdRemoved + " из " + ::ToString(idsAll.size()) + ": "
                            + session.GetStringReport());
                        return;
                    } else {
                        ERROR_LOG << session.GetStringReport() << Endl;
                        continue;
                    }
                } else {
                    removedRecords += ids.size();
                    NDrive::TEventLog::Log("DbServiceDiscoverRemoveRecords", NJson::TMapBuilder
                        ("table", target)
                        ("count", ids.size())
                    );
                    break;
                }
            }
            maxIdRemoved = maxIdRemovedNext;
            Sleep(Config->GetPauseBetweenRemoves());
            i = next;
        }
        NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()),
            "Удалены старые записи истории из " + target + " до " + instantBorder.ToString() + " (" + ::ToString(removedRecords) + " записей)");

        if (records.size() < recordsCountLimit) {
            break;
        }
    }
}

void TDBServiceProcessor::CleanTagsHistory(const NDrive::IServer* server) const {
    TSet<TString> aggressiveTags;
    TSet<TString> tagsForKill = Config->GetDeadlyTags();
    ITagsMeta::TTagDescriptionsByName tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    for (auto&& [name, description] : tags) {
        auto cleanupPolicy = description->OptionalCleanupPolicy();
        if (!cleanupPolicy) {
            tagsForKill.insert(name);
            continue;
        }
        switch (*cleanupPolicy) {
            case ECleanupPolicy::Aggressive:
                aggressiveTags.insert(name);
                break;
            case ECleanupPolicy::Ignore:
                tagsForKill.erase(name);
                break;
        }
    }

    aggressiveTags.insert({
        TChargableTag::Reservation,
        TChargableTag::Acceptance,
        TChargableTag::Riding,
        TChargableTag::Parking,
        TTelematicsCommandTag::Type(),
        TTransformationTag::TypeName,
        "full_tank_marker",
        "inexpensive_parking",
        "yaauto_deferred_warming",
        "ugc_unspecified",
        "ugc_dirty_exterior",
        "ugc_dirty_interior",
    });

    CleanTagsHistory("car_tags", "car_tags_history", tagsForKill, Config->GetTagHistoryLiveTime(), server);
    CleanTagsHistory("car_tags", "car_tags_history", aggressiveTags, Config->GetAggressiveCarTagHistoryLiveTime(), server);
    CleanTagsHistory("user_tags", "user_tags_history", tagsForKill, Config->GetUserTagHistoryLiveTime(), server);
    CleanTagsHistory("user_tags", "user_tags_history", aggressiveTags, Config->GetAggressiveUserTagHistoryLiveTime(), server);
    CleanTagsHistory("trace_tags", "trace_tags_history", tagsForKill, Config->GetTraceTagHistoryLiveTime(), server);
    CleanTagsHistory("trace_tags", "trace_tags_history", aggressiveTags, Config->GetAggressiveTraceTagHistoryLiveTime(), server);
}

void TDBServiceProcessor::CleanTagsHistory(const TString& tagTable, const TString& tagHistoryTable, const TSet<TString>& tags, TDuration lifetime, const NDrive::IServer* server) const {
    auto until = Now() - lifetime;
    auto condition = TStringBuilder() << "True";
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        {
            condition << " AND history_timestamp < " << until.Seconds();
        }
        if (!tags.empty()) {
            condition << " AND tag IN (" << session->Quote(tags) << ")";
        }
        auto optionalPerformedTagIds = GetPerformedTagIds(tagTable, session);
        if (!optionalPerformedTagIds) {
            NDrive::INotifier::Notify(
                server->GetNotifier(Config->GetNotifier()),
                "cannot get performed car_tags: " + session.GetStringReport()
            );
            return;
        }
        if (!optionalPerformedTagIds->empty()) {
            NDrive::INotifier::Notify(
                server->GetNotifier(Config->GetNotifier()),
                tagTable + ": " + ToString(optionalPerformedTagIds->size()) + " performed tags"
            );
            condition << " AND tag_id NOT IN (" << session->Quote(*optionalPerformedTagIds) << ")";
        }
    }
    RemoveCommon(condition, tagHistoryTable, until, server);
}

void TDBServiceProcessor::CleanCarPropositionsHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetCarTagPropositionsHistoryLiveTime();
    RemoveCommon("proposition_id IN (select distinct(proposition_id) from car_tag_propositions where history_action IN ('approved', 'rejected') AND history_timestamp < " + ToString(instantBorder.Seconds()) + ")"
        , "car_tag_propositions", instantBorder, server);
}

void TDBServiceProcessor::CleanDeprecatedTagsHistory(const NDrive::IServer* server) const {
    TSet<TString> tagsForKill = Config->GetTagsToRemove();
    ITagsMeta::TTagDescriptionsByName tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    for (auto&& i : tags) {
        if (!i.second->GetDeprecated()) {
            tagsForKill.erase(i.first);
        }
    }

    if (tagsForKill.empty()) {
        return;
    }

    const TInstant instantBorder = Now() - TDuration::Days(1);
    RemoveCommon("tag IN ('" + JoinSeq("','", tagsForKill) + "') "
        + "AND history_timestamp < " + ::ToString(instantBorder.Seconds()), "car_tags_history", instantBorder, server);

    RemoveCommon("tag IN ('" + JoinSeq("','", tagsForKill) + "') "
        + "AND history_timestamp < " + ::ToString(instantBorder.Seconds()), "user_tags_history", instantBorder, server);

    RemoveCommon("tag IN ('" + JoinSeq("','", tagsForKill) + "') "
        + "AND history_timestamp < " + ::ToString(instantBorder.Seconds()), "trace_tags_history", instantBorder, server);
}

void TDBServiceProcessor::CleanHeadSessionsHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetHeadSessionsHistoryLiveTime();

    RemoveCommon("history_action = 'update_data' OR history_timestamp < "
        + ::ToString(instantBorder.Seconds()), "head_app_sessions_history", instantBorder, server);
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "short_sessions_history", instantBorder, server);
}

void TDBServiceProcessor::CleanExpiredUserTemporaryActionTags(const NDrive::IServer* server) const {
    const auto& robotUserId = GetRobotUserId(server);
    const auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
    const auto& tagNames = tagsManager.GetTagsMeta().GetRegisteredTagNames({ TTemporaryActionTag::TypeName });
    const auto& userTagManager = tagsManager.GetUserTags();
    auto tx = userTagManager.BuildSession(true);
    auto optionalTags = userTagManager.RestoreTags(TVector<TString>{}, MakeVector(tagNames), tx);

    TVector<TDBTag> tagsForRemove;
    if (optionalTags) {
        for (auto&& i : *optionalTags) {
            if (i.GetTagAs<TTemporaryActionTag>() && i.GetTagAs<TTemporaryActionTag>()->IsExpired()) {
                tagsForRemove.emplace_back(i);
            }
        }
    } else {
        ERROR_LOG << GetId() << ": cannot RestoreTags: " << tx.GetStringReport() << Endl;
        return;
    }
    NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()), "Начинается очистка старых пользовательских тегов: " + ::ToString(tagsForRemove.size()));
    ui32 deleted = 0;
    ui32 failed = 0;
    for (ui32 i = 0; i < tagsForRemove.size(); ) {
        const ui32 iNext = Min<ui32>(i + Config->GetCleanerDiscretizationRecordsCount(), tagsForRemove.size());
        TVector<TDBTag> tagsForRemoveLocal(tagsForRemove.begin() + i, tagsForRemove.begin() + iNext);
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsForRemoveLocal, robotUserId, server, session) || !session.Commit()) {
            failed += iNext - i;
            ERROR_LOG << "CleanExpiredUserTemporaryActionTags: " << session.GetStringReport() << Endl;
        } else {
            deleted += iNext - i;
            NOTICE_LOG << "CleanExpiredUserTemporaryActionTags: removed " << iNext - i << " records" << Endl;
        }
        i = iNext;
        Sleep(TDuration::Seconds(1));
    }
    NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()), "Очищен пакет старых пользовательских тегов: удалено "
        + ::ToString(deleted) + " из " + ::ToString(failed + deleted));
}

void TDBServiceProcessor::CleanCompiledRides(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetCompiledRidesLiveTime();
    RemoveCommon(R"(
        tag_id IN (
            SELECT
                tag_id
            FROM
                trace_tags
            INNER JOIN
                compiled_rides
            ON
                trace_tags.object_id = compiled_rides.session_id
            WHERE
                compiled_rides.history_timestamp < )" + ::ToString(instantBorder.Seconds()) +
        ")", "trace_tags", instantBorder, server, "tag_id");
    RemoveCommon(R"(
        tag_id IN (
            SELECT
                tag_id
            FROM
                trace_tags
            INNER JOIN
                compiled_rides
            ON
                trace_tags.object_id = compiled_rides.session_id
            WHERE
                compiled_rides.history_timestamp < )" + ::ToString(instantBorder.Seconds()) +
        ")", "trace_tags_history", instantBorder, server, "tag_id");
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "compiled_rides", instantBorder, server);
}


void TDBServiceProcessor::CleanCompiledBills(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetCompiledBillsLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "compiled_bills", instantBorder, server);
}

void TDBServiceProcessor::CleanImagesHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetImagesHistoryLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "drive_images_history", instantBorder, server);
}

void TDBServiceProcessor::CleanInsuranceTasksHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetInsuranceTasksLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "insurance_tasks_history", instantBorder, server);
}

void TDBServiceProcessor::CleanDistributingBlockEventStatsHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetDistributingBlockEventStatsHistoryLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "distributing_block_event_stats_history", instantBorder, server);
}

void TDBServiceProcessor::CleanBillingTasksHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetBillingTasksLiveTime();
    TSet<TString> tasksForKill = Config->GetBillingTaskTypes();
    if (tasksForKill.empty()) {
        return;
    }
    RemoveCommon("billing_type IN ('" + JoinSeq("','", tasksForKill) + "') "
        + "AND history_timestamp < " + ::ToString(instantBorder.Seconds()), "billing_tasks_history", instantBorder, server);
}

void TDBServiceProcessor::CleanPayments(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetPaymentsLiveTime();
    auto query = TStringBuilder()
        << " last_update_ts < " << ::ToString(instantBorder.Seconds())
        << " AND drive_payments.status IN ('cleared', 'canceled', 'refunded', 'not_authorized')"
        << " AND session_id NOT IN ( SELECT session_id FROM billing_tasks )";
    RemoveCommon(query, "drive_payments", instantBorder, server, "id");
}


void TDBServiceProcessor::CleanBillingAccountHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetBillingAccountsLiveTime();
    TSet<TString> actionsForKill = Config->GetAccountActions();
    if (actionsForKill.empty()) {
        return;
    }
    RemoveCommon("history_action IN ('" + JoinSeq("','", actionsForKill) + "') "
        + "AND history_timestamp < " + ::ToString(instantBorder.Seconds()) + " AND history_comment IS NULL", "billing_account_history", instantBorder, server);

    TSet<ui64> fastCleanAccounts = Config->GetFastCleanAccountTypes();
    if (fastCleanAccounts.empty()) {
        return;
    }
    const TInstant fastBorder = Now() - Config->GetFastAccountsLiveTime();
    RemoveCommon("type_id IN (" + JoinSeq(",", fastCleanAccounts) + ") "
        + "AND history_timestamp < " + ::ToString(fastBorder.Seconds()), "billing_account_history", fastBorder, server);
}

void TDBServiceProcessor::CleanUserDataHistory(const NDrive::IServer* server) const {
    const TInstant instantBorder = Now() - Config->GetUserDataHistoryLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(instantBorder.Seconds()), "drive_user_data_history", instantBorder, server);

    const TInstant userDevicesHistoryBorder = Now() - Config->GetUserDevicesHistoryLiveTime();
    RemoveCommon("history_timestamp < " + ::ToString(userDevicesHistoryBorder.Seconds()), "user_devices_history", userDevicesHistoryBorder, server);
}

bool TDBServiceProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    CleanTagsHistory(server);
    CleanHeadSessionsHistory(server);
    CleanUserDataHistory(server);
    CleanCompiledRides(server);
    CleanCompiledBills(server);
    CleanImagesHistory(server);
    CleanInsuranceTasksHistory(server);
    CleanBillingTasksHistory(server);
    CleanPayments(server);
    CleanBillingAccountHistory(server);
    CleanExpiredUserTemporaryActionTags(server);
    CleanDeprecatedTagsHistory(server);
    CleanCarPropositionsHistory(server);
    CleanDistributingBlockEventStatsHistory(server);
    return true;
}

TDBServiceProcessor::TDBServiceProcessor(const TDBServiceConfig* config)
    : TBase(*config)
    , Config(config)
{}

bool TDBRegularServiceProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    auto robotUserId = GetRobotUserId(server);
    TRegularDBServiceMessage message(robotUserId);
    SendGlobalMessage(message);
    INFO_LOG << "TDBRegularServiceProcessor: " << message.GetStringReport() << Endl;
    if (!message.IsEmpty()) {
        NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifier()), "Регулярные сервисные процессы:\n" + message.GetStringReport());
    }
    return true;
}

TDBRegularServiceProcessor::TDBRegularServiceProcessor(const TDBRegularServiceConfig* config)
    : TBase(*config)
    , Config(config)
{
}
