#include "processor.h"

#include "config.h"

#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/markers.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/major/client.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/disk/client.h>
#include <drive/library/cpp/tex_builder/tex_builder.h>

#include <util/charset/wide.h>

bool TUpdateOSAGOBackgroundProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server->GetDriveAPI()->HasMajorClient()) {
        ERROR_LOG << "Major client not defined" << Endl;
        return false;
    }

    if (!server->GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "Mds client not defined" << Endl;
        return false;
    }
    const TS3Client::TBucket* mdsBucket = server->GetDriveAPI()->GetMDSClient().GetBucket(Config->GetMdsBucket());
    if (!mdsBucket) {
        ERROR_LOG << "Mds bucket " << Config->GetMdsBucket() << " not exists" << Endl;
        return false;
    }

    TSet<TString> filteredCars;
    if (!Config->GetCarsFilter().GetAllowedCarIds(filteredCars, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        return true;
    }

    auto robotUserId = GetRobotUserId(server);
    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(filteredCars);

    TVector<TString> carsWithoutOSAGO;
    TVector<TString> carsChangeOSAGO;
    TVector<TString> carsNeedOSAGO;

    TInstant startProcess(TInstant::Now());
    for (const auto& car : gCars.GetResult()) {
        if (!car.second.GetNumber()) {
            continue;
        }

        TCarGenericAttachment currentRegistryDocument;
        server->GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(car.first, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess);
        auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());

        if (!!baseDocuments && (baseDocuments->GetOsagoDateTo() > startProcess + Config->GetUpdateTime()) && baseDocuments->GetOsagoMDSKey()) {
            continue;
        }

        THolder<TCarRegistryDocument> carRegistryDocument;
        if (!!baseDocuments) {
            carRegistryDocument.Reset(new TCarRegistryDocument(*baseDocuments));
        } else {
            carRegistryDocument.Reset(new TCarRegistryDocument());
            carRegistryDocument->SetNumber(car.second.GetNumber()).SetRegistrationId(car.second.GetRegistrationID());
        }
        if (!carRegistryDocument->GetVin()) {
            carRegistryDocument->SetVin(car.second.GetVin());
        }

        bool isUpdate = false;
        TMessagesCollector errors;
        if (!carRegistryDocument->UpdateOSAGO(isUpdate, *mdsBucket, *server, errors)) {
            if (!!baseDocuments) {
                carsNeedOSAGO.emplace_back(car.second.GetHRReport() + " " + errors.GetStringReport());
            } else {
                carsWithoutOSAGO.emplace_back(car.second.GetHRReport() + " " + errors.GetStringReport());
            }
            continue;
        }

        if (!isUpdate) {
            carsNeedOSAGO.emplace_back(car.second.GetHRReport() + " only old version is in Major");
            continue;
        }

        {
            TCarGenericAttachment attachment(carRegistryDocument.Release());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetCarAttachmentAssignments().Attach(attachment, car.second.GetId(), robotUserId, session, server) || !session.Commit()) {
                carsNeedOSAGO.emplace_back(car.second.GetHRReport() + " " + session.GetMessages().GetStringReport());
                continue;
            }
        }

        if (Config->GetDeliveryTagName()) {
            auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Config->GetDeliveryTagName());
            ITag::TPtr tag = NDrive::ITag::TFactory::Construct(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagTypeByName(Config->GetDeliveryTagName()));
            if (!tag) {
                carsNeedOSAGO.emplace_back(car.second.GetHRReport() + " incorrect tag name");
                continue;
            }
            tag->SetName(Config->GetDeliveryTagName());
            if (!!description) {
                tag->SetTagPriority(description->GetDefaultPriority());
            }
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, robotUserId, car.first, server, session, EUniquePolicy::SkipIfExists) || !session.Commit()) {
                carsNeedOSAGO.emplace_back(car.second.GetHRReport() + " " + session.GetMessages().GetStringReport());
                continue;
            }
        }
        carsChangeOSAGO.emplace_back(car.second.GetHRReport());
    }
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Осаго отсутствует для (" + ToString(carsWithoutOSAGO.size()) + " объектов):", carsWithoutOSAGO);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Требуется замена осаго для (" + ToString(carsNeedOSAGO.size()) + " объектов):", carsNeedOSAGO);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Изменилось осаго для (" + ToString(carsChangeOSAGO.size()) + " объектов):", carsChangeOSAGO);
    return true;
}

TUpdateOSAGOBackgroundProcessor::TUpdateOSAGOBackgroundProcessor(const TUpdateOSAGOBackgroundProcessorConfig* config)
    : TBase(*config)
    , Config(config)
{}

bool TUpdateSTSBackgroundProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasMajorClient()) {
        ERROR_LOG << "TMajorRequester: Major api not found" << Endl;
        return false;
    }

    if (!server->GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "Mds client not defined" << Endl;
        return false;
    }
    const TS3Client::TBucket* mdsBucket = server->GetDriveAPI()->GetMDSClient().GetBucket(Config->GetMdsBucket());
    if (!mdsBucket) {
        ERROR_LOG << "Mds bucket " << Config->GetMdsBucket() << " not exists" << Endl;
        return false;
    }

    auto absentSTSDescription = dynamic_cast<const TRepairTagRecord::TDescription*>(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Config->GetAbsentTag()).Get());
    if (!absentSTSDescription || absentSTSDescription->GetWorkType() != NMajorClient::EWorkType::UpdateSTS) {
        ERROR_LOG << "incorrect update_sts tag " << Config->GetAbsentTag() << Endl;
        return false;
    }

    auto deliveryDescription = dynamic_cast<const TRepairTagRecord::TDescription*>(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Config->GetDeliveryTag()).Get());
    if (!deliveryDescription) {
        ERROR_LOG << "incorrect delivery_sts tag " << Config->GetDeliveryTag() << Endl;
        return false;
    }

    TSet<TString> filteredCars;
    if (!Config->GetCarsFilter().GetAllowedCarIds(filteredCars, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        return true;
    }

    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCached(filteredCars);

    TVector<NMajorClient::TGetQueriesRequest::TQuery> allOpenRequests;
    {
        TMessagesCollector majorErrors;
        if (!server->GetDriveAPI()->GetMajorClient().GetQueryList(false, allOpenRequests, majorErrors)) {
            NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), "Cannot get open major queries: " + majorErrors.GetStringReport());
            return true;
        }
    }
    TMap<TString, TString> openSTSQueries;
    for (const auto& query : allOpenRequests) {
        if (query.GetTypeName() != "Восстановление СТС") {
            continue;
        }
        if (!openSTSQueries.emplace(query.GetVIN(), query.GetId()).second) {
            NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), "Двойная заявка на стс для авто: " + query.GetVIN());
            return true;
        }
    }

    TVector<TDBTag> tags;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(filteredCars, { Config->GetAbsentTag() }, tags, session)) {
            ERROR_LOG << "Cannot restore tags" << Endl;
            return true;
        }
    }

    TVector<TString> errorNotification;
    TInstant startProcess = Now();

    TSet<TString> allowedNewOsagoIds;
    if (!Config->GetOsagoCarsFilter().GetAllowedCarIds(allowedNewOsagoIds, server, TInstant::Zero())) {
        ERROR_LOG << "Cannot restore allowed osago" << Endl;
        return true;
    }

    auto robotUserId = GetRobotUserId(server);
    auto robotPermissions = server->GetDriveAPI()->GetUserPermissions(robotUserId);
    if (!robotPermissions) {
        ERROR_LOG << GetId() << ": cannot create permissions for " << robotUserId << Endl;
        return true;
    }
    for (auto&& dbTag : tags) {
        TMessagesCollector errors;
        auto car = gCars.GetResultPtr(dbTag.GetObjectId());
        if (!car) {
            continue;
        }
        TRepairTagRecord* repairTag = dbTag.MutableTagAs<TRepairTagRecord>();
        if (!repairTag) {
            errorNotification.emplace_back(car->GetHRReport() + " incorrect tag type for " + dbTag->GetName());
            continue;
        }

        if (repairTag->GetQueryId().empty()) {
            TString queryId;
            auto queryIter = openSTSQueries.find(car->GetVin());
            if (queryIter != openSTSQueries.end()) {
                queryId = queryIter->second;
            } else {
                NMajorClient::TSTSRequest::TResponse answer;
                if (!server->GetDriveAPI()->GetMajorClient().CreateSTSRequest(car->GetVin(), answer, errors)) {
                    errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                    continue;
                }
                queryId = answer.GetId();
            }

            repairTag->SetQueryId(queryId);
            repairTag->SetDate(Now());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(dbTag, robotUserId, session) || !session.Commit()) {
                errorNotification.emplace_back(car->GetHRReport() + " TSTSUpdater: Query was created but id not saved " + session.GetStringReport());
            }
        } else {
            TString status;
            bool isClosed = false;
            if (!server->GetDriveAPI()->GetMajorClient().GetStatus(repairTag->GetQueryId(), status, isClosed, errors)) {
                errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                continue;
            }

            if (status != Config->GetDeliveryStatus() && !isClosed) {
                continue;
            }

            ui64 registrationId = 0;
            TInstant registrationDate = TInstant::Zero();
            TString newNumber;
            NMajorClient::TQueryInfoRequest::TFullQueryInfo info;
            if (!server->GetDriveAPI()->GetMajorClient().GetQueryInfo(repairTag->GetQueryId(), info, errors)
                || !info.TryGetValue(Config->GetNumberFieldName(), registrationId) || !registrationId
                || !info.TryGetInstant(Config->GetDateFieldName(), registrationDate) || registrationDate == TInstant::Zero()
                || !info.TryGetValue(Config->GetNewRegNoFieldName(), newNumber) || !newNumber)
            {
                errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport() + " " + info.GetJsonReport().GetStringRobust());
                continue;
            }

            TCarGenericAttachment currentRegistryDocument;
            server->GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(car->GetId(), EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess);
            auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            THolder<TCarRegistryDocument> carRegistryDocument;
            if (!!baseDocuments) {
                carRegistryDocument.Reset(new TCarRegistryDocument(*baseDocuments));
            } else {
                carRegistryDocument.Reset(new TCarRegistryDocument());
                carRegistryDocument->SetNumber(car->GetNumber());
            }
            if (!carRegistryDocument->GetVin()) {
                carRegistryDocument->SetVin(car->GetVin());
            }
            carRegistryDocument->SetRegistrationId(registrationId);
            carRegistryDocument->SetRegistrationDate(registrationDate);

            TVector<NMajorClient::TGetFileRequest::TMajorFile> files;
            if (!server->GetDriveAPI()->GetMajorClient().GetFiles(repairTag->GetQueryId(), { ".JPG" }, files, errors)) {
                errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                continue;
            }

            if (files.empty()) {
                errorNotification.emplace_back(car->GetHRReport() + " absent sts photos");
                continue;
            }

            {
                NTexBuilder::TTexBuilder builder(NTexBuilder::TTexBuilderConfig::ParseFromString(""));
                if (!builder.Init()) {
                    errorNotification.emplace_back(car->GetHRReport() + " cannot init tex_builder");
                    continue;
                }
                for (const auto& file : files) {
                    if (!builder.AddImage(file.GetFile(), errors)) {
                        errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                        continue;
                    }
                }
                auto pdfDocument = builder.BuildFinalDocument(errors);
                if (pdfDocument.Empty()) {
                    errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                    continue;
                }
                TStringBuf resultPDF(pdfDocument.AsCharPtr(), pdfDocument.Size());
                TString fileName = "STS/" + car->GetVin() + "/majorAPI/" + ToString(startProcess.Seconds()) + ".pdf";
                if (mdsBucket->PutKey(fileName, TString(resultPDF), errors) != HTTP_OK) {
                    ERROR_LOG << "failed to load file" << Endl;
                    errorNotification.emplace_back(car->GetHRReport() + " mds error " + errors.GetStringReport());
                    continue;
                } else {
                    carRegistryDocument->SetRegistrationMDSKey(fileName);
                }
            }

            newNumber = ToLowerUTF8(newNumber);
            if (newNumber != car->GetNumber()) {
                carRegistryDocument->SetNumber(newNumber);
                if (allowedNewOsagoIds.contains(car->GetId())) {
                    bool isUpdate = false;
                    if (!carRegistryDocument->UpdateOSAGO(isUpdate, *mdsBucket, *server, errors, true)) {
                        errorNotification.emplace_back(car->GetHRReport() + " osago updating failed: " + errors.GetStringReport());
                        continue;
                    }
                    if (!isUpdate && carRegistryDocument->GetOsagoDate() > (carRegistryDocument->GetContractDate() + TDuration::Days(180))) {
                        errorNotification.emplace_back(car->GetHRReport() + " old osago");
                        continue;
                    }
                }
            }

            TCarGenericAttachment attachment(carRegistryDocument.Release());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetCarAttachmentAssignments().Attach(attachment, car->GetId(), robotUserId, session, server)) {
                errorNotification.emplace_back(car->GetHRReport() + " " + session.GetStringReport());
                continue;
            }

            ITag::TPtr newTag;
            {
                newTag = TRepairTagRecord::BuildFromJson(server->GetDriveAPI()->GetTagsManager(), dbTag->SerializeToJson(), &errors);
                if (!newTag) {
                    errorNotification.emplace_back(car->GetHRReport() + " " + errors.GetStringReport());
                    continue;
                }
                newTag->SetName(Config->GetDeliveryTag());
            }
            if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().EvolveTag(dbTag, newTag, *robotPermissions, server, session) || !session.Commit()) {
                errorNotification.emplace_back(car->GetHRReport() + " " + session.GetStringReport());
            } else {
                NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), "Требуется разложить новое стс для " + car->GetHRReport());
            }
        }
    }
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Ошибка при обновлении стс (" + ToString(errorNotification.size()) + " объектов):", errorNotification);
    return true;
}

TUpdateSTSBackgroundProcessor::TUpdateSTSBackgroundProcessor(const TUpdateSTSBackgroundProcessorConfig* config)
    : TBase(*config)
    , Config(config)
{}

bool ILoadFromDiskBackgroundProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server->GetDriveAPI()->HasYandexDiskClient()) {
        ERROR_LOG << "disk client not defined" << Endl;
        return false;
    }

    if (!server->GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "Mds client not defined" << Endl;
        return false;
    }
    const TS3Client::TBucket* mdsBucket = server->GetDriveAPI()->GetMDSClient().GetBucket(Config->GetMdsBucket());
    if (!mdsBucket) {
        ERROR_LOG << "Mds bucket " << Config->GetMdsBucket() << " not exists" << Endl;
        return false;
    }

    TSet<TString> filteredCars;
    if (!Config->GetCarsFilter().GetAllowedCarIds(filteredCars, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        return true;
    }

    auto robotUserId = GetRobotUserId(server);
    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCached(filteredCars);

    TVector<TString> carsWithoutBaseDocument;
    TVector<TString> carsWithout;
    TVector<TString> carsError;
    TVector<TString> carsOK;

    TInstant startProcess(TInstant::Now());
    for (const auto& car : gCars.GetResult()) {
        if (car.second.GetNumber().empty()) {
            continue;
        }

        TCarGenericAttachment currentRegistryDocument;
        if (!server->GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(car.first, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess)) {
            carsWithoutBaseDocument.emplace_back(car.second.GetHRReport());
            continue;
        }

        auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
        if (!baseDocuments) {
            carsWithoutBaseDocument.emplace_back(car.second.GetHRReport());
            continue;
        }

        if (!Config->GetIgnoreExisting() && CheckDocumentExistence(*baseDocuments)) {
            continue;
        }

        TMessagesCollector errors;

        TUtf16String wideNumber = TUtf16String::FromUtf8(car.second.GetNumber());
        wideNumber.to_upper();
        TVector<IDocumentStorage::IDocumentStorageFile::TPtr> fileList;
        if (!server->GetDriveAPI()->GetYandexDiskClient().GetFiles(Config->GetDiskPath() + "/" + WideToUTF8(wideNumber), fileList, errors)) {
            carsWithout.emplace_back(car.second.GetHRReport() + " " + errors.GetStringReport());
            continue;
        } else {
            IDocumentStorage::IDocumentStorageFile::TPtr storageFile = nullptr;
            for (const auto& file : fileList) {
                TUtf16String wideName = TUtf16String::FromUtf8(file->GetName());
                wideName.to_upper();
                if (WideToUTF8(wideName).Contains(Config->GetDocumentType()) && ToLowerUTF8(file->GetName()).Contains(".pdf")) {
                    if (!!storageFile) {
                        carsWithout.emplace_back(car.second.GetHRReport() + " дубль");
                        break;
                    }
                    storageFile = file;
                }
            }
            if (!storageFile) {
                carsWithout.emplace_back(car.second.GetHRReport() + " файл не найден");
                continue;
            }

            THolder<TCarRegistryDocument> carRegistryDocument(new TCarRegistryDocument(*baseDocuments));
            TString fileName = Config->GetDocumentType() + "/" + car.second.GetVin() + "/" + ToString(startProcess.Seconds()) + ".pdf";
            TString fileData;
            if (!storageFile->GetData(fileData, errors) || !fileData) {
                carsError.emplace_back(car.second.GetHRReport() + " disk " + errors.GetStringReport());
                continue;
            }
            if (mdsBucket->PutKey(fileName, fileData, errors) != HTTP_OK) {
                ERROR_LOG << "failed to load file" << Endl;
                carsError.emplace_back(car.second.GetHRReport() + " mds " + errors.GetStringReport());
                continue;
            } else {
                SetDocumentPath(*carRegistryDocument, fileName);
            }

            TCarGenericAttachment attachment(carRegistryDocument.Release());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetCarAttachmentAssignments().Attach(attachment, car.second.GetId(), robotUserId, session, server) ||
                !session.Commit()) {
                carsError.emplace_back(car.second.GetHRReport() + " attach " + session.GetMessages().GetStringReport());
            } else {
                carsOK.emplace_back(car.second.GetHRReport());
            }
        }
    }
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Добавлена ссылка на " + Config->GetDocumentType() + " для (" + ToString(carsOK.size()) + " объектов):", carsOK);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Ошибка при добавлении " + Config->GetDocumentType() + " для (" + ToString(carsError.size()) + " объектов):", carsError);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Отсутствует " + Config->GetDocumentType() + " на диске для (" + ToString(carsWithout.size()) + " объектов):", carsWithout);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Отсутствует основной документ для (" + ToString(carsWithoutBaseDocument.size()) + " объектов):", carsWithoutBaseDocument);
    return true;
}

ILoadFromDiskBackgroundProcessor::ILoadFromDiskBackgroundProcessor(const ILoadFromDiskBackgroundProcessorConfig* config)
    : TBase(*config)
    , Config(config)
{}
