#include "processor.h"

#include "config.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/data/service_session.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/fueling_manager/fueling_manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/toloka/fueling/toloka.h>
#include <drive/library/cpp/yt/common/writer.h>

#include <library/cpp/yson/node/node_io.h>

#include <rtline/library/unistat/cache.h>


bool IsStatusChanged(const TUserFuelingTag& tag, const EFuelingStatus status, const TMaybe<TFuelingManager::TStatusInfo>& statusInfo, const TString& comment) {
    static const TSet<EFuelingStatus> incorrectStatuses = {
          EFuelingStatus::ReadyForStart
        , EFuelingStatus::ServerProblems
        , EFuelingStatus::NotInStation
        , EFuelingStatus::Unknown
        , EFuelingStatus::Unavailable
    };
    if (incorrectStatuses.contains(status)) {
        return false;
    }
    if (tag.GetCurrentState() == EFuelingStatus::WaitCancel && status != EFuelingStatus::Free) {
        return false;
    }
    if (tag.GetCurrentState() != status) {
        return true;
    }
    if (statusInfo && status != EFuelingStatus::Free) {
        if (!tag.HasActualFuelType() || tag.GetActualFuelTypeRef() != statusInfo->FuelId) {
            return true;
        }
        if (!tag.HasActualLiters() || tag.GetActualLitersRef() != statusInfo->Liters) {
            return true;
        }
        return false;
    }
    return !comment.empty() && tag.GetComment() != comment;
}

bool CheckNecessityToRemove(const TUserFuelingTag& tag, const EFuelingStatus status) {
    return tag.GetCurrentState() == EFuelingStatus::WaitCancel && status == EFuelingStatus::Free;
}

void SetTagData(TUserFuelingTag& tag, const EFuelingStatus status, const TMaybe<TFuelingManager::TStatusInfo>& statusInfo, const TString& comment, const TFuelGroups& fuelGroups) {
    tag.SetCurrentState(status);
    if (statusInfo) {
        if (status == EFuelingStatus::Free) {
            return;
        }
        tag.SetActualLiters(statusInfo->Liters);
        tag.SetActualFuelType(statusInfo->FuelId);
        if (tag.HasFuelType() &&  !fuelGroups.FuelTypeCmp(tag.GetFuelTypeUnsafe(), statusInfo->FuelId)) {
            tag.SetCurrentState(EFuelingStatus::WrongFuelType);
        }
    } else {
        tag.SetComment(comment);
    }
}

bool TUserFuelingProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    TVector<TDBTag> tagsFullList;
    if (!server->GetFuelingManager()) {
        return false;
    }
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {TUserFuelingTag::GetTypeName()}, tagsFullList, session)) {
            ERROR_LOG << session.GetStringReport() << Endl;
            return true;
        }
    }
    static const TSet<EFuelingStatus> skipCheckStatuses = {
          EFuelingStatus::FuelingCompleted
        , EFuelingStatus::WrongFuelLiters
        , EFuelingStatus::WrongFuelType
        , EFuelingStatus::TriedOrderCreate
    };

    TFuelGroups fuelGroups;
    fuelGroups.InitFromSettings(server->GetSettings());

    TVector<TDBTag> tags;
    TVector<TDBTag> tagsToRemove;
    for (auto&& i : tagsFullList) {
        TUserFuelingTag* ufTag = i.MutableTagAs<TUserFuelingTag>();
        if (!ufTag || skipCheckStatuses.contains(ufTag->GetCurrentState())) {
            continue;
        }
        const EFuelClientType clientType = ufTag->GetFuelClientTypeDef(EFuelClientType::Default);
        if (!server->GetFuelingManager()->HasFuelClientKey(clientType)) {
            WARNING_LOG << "Can't update tag data: " << i.GetTagId() << Endl;
            continue;
        }
        TString comment;
        TMaybe<TFuelingManager::TStatusInfo> statusInfo;
        TMessagesCollector errors;
        const EFuelingStatus fStatus = ufTag->GetOrderId().empty()
            ? server->GetFuelingManager()->GetPostStatus(ufTag->GetStationId(), ufTag->GetColumnId(), statusInfo, clientType, errors)
            : server->GetFuelingManager()->GetStatus(ufTag->GetOrderId(), comment, clientType, errors);
        TUnistatSignalsCache::SignalAdd("fueling", "status-" + ::ToString(fStatus), 1);
        if (fStatus == EFuelingStatus::ServerProblems) {
            NDrive::TEventLog::Log("FuelingStatusProcessError", NJson::TMapBuilder
                ("error", errors.GetStringReport())
                ("order_id", ufTag->GetOrderId())
                ("station_id", ufTag->GetStationId())
                ("column_id", ufTag->GetColumnId())
                ("client_type", ToString(clientType))
            );
        }
        if (IsStatusChanged(*ufTag, fStatus, statusInfo, comment)) {
            if (CheckNecessityToRemove(*ufTag, fStatus)) {
                tagsToRemove.emplace_back(i);
            } else {
                SetTagData(*ufTag, fStatus, statusInfo, comment, fuelGroups);
                tags.emplace_back(i);
            }
        }
    }
    auto robotUserId = GetRobotUserId(server);
    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagsData(tags, robotUserId, session)) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return true;
    }
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsToRemove, robotUserId, server, session)) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return true;
    }
    if (!session.Commit()) {
        ERROR_LOG << session.GetStringReport() << Endl;
    }
    return true;
}

TUserFuelingProcessor::TUserFuelingProcessor(const TUserFuelingConfig* config)
    : TBase(*config)
    , Config(config)
{
    Y_UNUSED(Config);
}

bool TRegularFuelingReportProcessor::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    auto carsInfo = server->GetDriveAPI()->GetCarsData()->GetCached();

    const auto& historyManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager();
    auto builder = historyManager.GetSessionsBuilder("service", Now());
    CHECK_WITH_LOG(builder);
    ui64 currentDay = TInstant::Now().Days();
    auto userSessions = builder->GetSessionsActual(TInstant::Days(currentDay - 1), TInstant::Days(currentDay));
    auto userIds = MakeSet(NContainer::Keys(userSessions));
    auto usersInfo = server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds);

    TStringStream csvReport;
    csvReport << "fueling time;full name;car number;duration;fuel level before;fuel level after;tag name" << Endl;
    for (const auto& user : userSessions) {
        auto itUser = usersInfo.find(user.first);
        if (itUser == usersInfo.end()) {
            ERROR_LOG << "Undefined user " << user.first << Endl;
            continue;
        }

        for (const auto& session : user.second) {
            TMaybe<TServiceSession::TCompilation> compilation = session->GetCompilationAs<TServiceSession::TCompilation>();
            if (!compilation) {
                ERROR_LOG << "Cannot use service session withno compilation for object " << session->GetObjectId() << " and user: " << session->GetUserId() << Endl;
                continue;
            }

            auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(compilation->GetTagName());
            if (!description) {
                continue;
            }

            if (description->GetType() != "simple_fueling_tag" && description->GetType() != "service_fueling_tag") {
                continue;
            }

            auto itCar = carsInfo.GetResult().find(session->GetObjectId());
            if (itCar == carsInfo.GetResult().end()) {
                ERROR_LOG << "Undefined car " << session->GetObjectId() << Endl;
                continue;
            }
            csvReport << ToString(session->GetLastTS())
                      << ";" << itUser->second.GetFullName()
                      << ";" << itCar->second.GetNumber()
                      << ";" << (session->GetLastTS() - session->GetStartTS()).Minutes()
                      << ";" << compilation->GetFuelLevelBefore()
                      << ";" << compilation->GetFuelLevelAfter()
                      << ";" << compilation->GetTagName() << Endl;
        }
    }
    TString stringDay = ToString(TInstant::Days(currentDay - 1)).substr(0, 10);
    NDrive::INotifier::TMessage message(stringDay, csvReport.Str());
    message.SetAdditionalInfo(Config->GetDescription());
    message.SetTitle(stringDay + ".csv");

    if (!NDrive::INotifier::SendDocument(server->GetNotifier(Config->GetNotifierName()), message, "text/csv", NDrive::INotifier::TContext().SetServer(server))) {
        ERROR_LOG << "Notifier internal error" << Endl;
    }
    return true;
}

TRegularFuelingReportProcessor::TRegularFuelingReportProcessor(const TRegularFuelingReportConfig * config)
    : TBase(*config)
    , Config(config)
{}


TFuelingSessionsExport::TFuelingSessionsExport(const TServiceSessionsExportConfig& config)
    : TCommonHistoryExport(config)
    , LocalConfig(config)
{}

bool TFuelingSessionsExport::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const auto& historyManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager();

    auto sessionBuilder = historyManager.GetSessionsBuilder("service");
    if (!sessionBuilder) {
        ERROR_LOG << "Incorrect session builder 'service'" << Endl;
        return true;
    }

    const ui64 historyIdCursor = historyManager.GetLockedMaxEventId();
    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = sessionBuilder->GetSessionsActualSinceId(LastEventId);
    TCarsDB::TFetchResult allCars = server->GetDriveAPI()->GetCarsData()->GetCached();

    ui64 nextReal = LastEventId;
    TInstant minSessionTs = TInstant::Max();

    try {
        auto locale = DefaultLocale;
        auto ytClient = NYT::CreateClient(Config.GetYtCluster());
        TYTWritersSet<TTableSelectorWork> writers(ytClient, Config.GetYtDataPath() + "/session");
        for (auto&& serviceSession : sessions) {
            if (serviceSession->GetLastEventId() > historyIdCursor) {
                continue;
            }
            nextReal = Max(nextReal, serviceSession->GetLastEventId());

            TServiceSession::TCompilation compilation(server->GetDriveAPI()->GetTagsManager().GetTagsMeta());
            if (!serviceSession->GetClosed() || serviceSession->GetLastEventId() <= LastEventId || !serviceSession->FillCompilation(compilation)) {
                continue;
            }

            auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(compilation.GetTagName());
            if (!tagDescription || !LocalConfig.GetTagsTypes().contains(tagDescription->GetType())) {
                continue;
            }

            auto carInfo = allCars.GetResultPtr(serviceSession->GetObjectId());
            if (!carInfo) {
                continue;
            }

            NYT::TNode recordNode;
            recordNode["session_id"] = compilation.GetSessionId();
            recordNode["start_ts"] = serviceSession->GetStartTS().Seconds();
            recordNode["last_ts"] = serviceSession->GetLastTS().Seconds();
            recordNode["user_id"] = serviceSession->GetUserId();
            recordNode["object_id"] = serviceSession->GetObjectId();
            recordNode["tag_id"] = compilation.GetTagId();
            recordNode["car_number"] = carInfo->GetNumber();
            recordNode["details"] = NYT::NodeFromJsonValue(compilation.GetReport(locale, server, nullptr));
            recordNode["work"] = TTableSelectorWork::GetTable(serviceSession->GetStartTS());
            writers.GetWriter(serviceSession->GetStartTS())->AddRow(recordNode);

            minSessionTs = Min(serviceSession->GetStartTS(), minSessionTs);
        }
        writers.Finish();
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return true;
    }

    try {
        if (LocalConfig.NeedInTolokaPools()) {
            INFO_LOG << "Check Toloka pools for " << minSessionTs.Seconds() << Endl;
            if (minSessionTs != TInstant::Max()) {
                auto ytClient = NYT::CreateClient(Config.GetYtCluster());
                TFuelingPoolsWrapper tolokaWrapper(LocalConfig.GetTolokaConfig(), ytClient, LocalConfig.GetYtDataPath());
                tolokaWrapper.CreatePools(minSessionTs, 1, server->GetNotifier(LocalConfig.GetNotifierName()));
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "TolokaError " << FormatExc(e) << Endl;
        return true;
    }
    LastEventId = nextReal;
    return true;
}
