#include "processor.h"

#include "config.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/accessor.h>

bool THeadTagWatcher::GetHeadTags(const NDrive::IServer* server, TMap<TString, TDBTag>& result) const {
    const auto& manager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
    auto tx = manager.BuildTx<NSQL::ReadOnly>();
    TVector<TDBTag> dbTags;
    if (!manager.RestoreTags({}, {Config->GetTagName()}, dbTags, tx)) {
        ERROR_LOG << "Cannot restore " << Config->GetTagName()  << "  tags" << Endl;
        return false;
    }
    for (const auto& tag : dbTags) {
        result.insert(std::make_pair(tag.GetObjectId(), std::move(tag)));
    }
    return true;
}

void AddCarModelSignal(TMap<TString, ui64>& signalCount, const TString& carModel) {
    auto itCount = signalCount.find(carModel);
    if (itCount != signalCount.end()) {
        ++itCount->second;
    } else {
        signalCount[carModel] = 1;
    }
}

bool THeadTagWatcher::CheckHeads(const NDrive::IServer* server) const {
    auto carsInfo = server->GetDriveAPI()->GetCarsData()->GetCached();
    if (carsInfo.empty()) {
        return true;
    }

    TMap<TString, TString> currentHeadIds;
    if (!server->GetDriveAPI()->GetHeadAccountManager().GetHeadIds({}, currentHeadIds)) {
        return false;
    }

    TMap<TString, TDBTag> currentTags;
    if (!GetHeadTags(server, currentTags)) {
        return false;
    }

    TMap<TString, ui64> EmptySignalCount;
    TVector<TDBTag> tagsForRemove;
    TMap<TString, ITag::TPtr> tagsForAdd;
    for (const auto& carInfo : carsInfo.GetResult()) {
        const TString& carId = carInfo.first;
        const TString& carModel = carInfo.second.GetModel();
        auto iterHead = currentHeadIds.find(carId);
        if (iterHead != currentHeadIds.end()) {
            auto iterTag = currentTags.find(carId);
            if (iterTag == currentTags.end()) {
                auto tag = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTagAs<TDeviceAdditionalFeature>(Config->GetTagName(), "Auto mark heads");
                if (!tag) {
                    ERROR_LOG << GetId() << ": cannot create DeviceAdditionalFeature tag " << Config->GetTagName() << Endl;
                    return false;
                }
                AddSignal.Signal(carModel, 1);
                tagsForAdd.emplace(carId, std::move(tag));
            }
        } else {
            AddCarModelSignal(EmptySignalCount, carModel);
            auto iterTag = currentTags.find(carId);
            if (iterTag != currentTags.end()) {
                RemoveSignal.Signal(carModel, 1);;
                tagsForRemove.emplace_back(iterTag->second);
            }
        }
    }

    for (const auto& model : EmptySignalCount) {
        EmptySignal.Signal(model.first, model.second);
    }

    if (!Config->GetDryRunMode()) {
        const TDeviceTagsManager& manager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
        const TString& robotUserId = GetRobotUserId(server);
        auto tx = manager.BuildTx<NSQL::Writable>();
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTags(tagsForRemove, robotUserId, server, tx, true)) {
            ERROR_LOG << "Cannot remove tags: " << tx.GetStringReport() << Endl;
            return false;
        }
        for (const auto& car : tagsForAdd) {
            if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTags({ car.second }, robotUserId, car.first, server, tx, EUniquePolicy::Rewrite)) {
                ERROR_LOG << "Cannot add tag: " << tx.GetStringReport() << Endl;
                return false;
            }
        }
        if (!tx.Commit()) {
            ERROR_LOG << "Session commit error: " << tx.GetStringReport() << Endl;
            return false;
        }
    }
    return true;
}

void THeadTagWatcher::UpdateSignalMap(const NDrive::IServer* server) const {
    auto models = server->GetDriveAPI()->GetModelsData()->GetCached();
    for (const auto& model : models) {
        if (!EmptySignal.HasKey(model.first)) {
            EmptySignal.RegisterSignal(model.first, new TUnistatSignal<double>({ "frontend-car-without-head-" + model.first }, EAggregationType::LastValue, "attm"));
            AddSignal.RegisterSignal(model.first, new TUnistatSignal<double>({ "frontend-add-head-tag-" + model.first }, EAggregationType::Sum, "dmmm"));
            RemoveSignal.RegisterSignal(model.first, new TUnistatSignal<double>({ "frontend-remove-head-tag-" + model.first }, EAggregationType::Sum, "dmmm"));
        }
    }
}

bool THeadTagWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!ITag::TFactory::Has(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagTypeByName(Config->GetTagName()))) {
        ERROR_LOG << "tag " << Config->GetTagName() << " doesn't exist" << Endl;
        return true;
    }
    UpdateSignalMap(server);
    if (!CheckHeads(server)) {
        ERROR_LOG << "Cannot execute CheckHeads" << Endl;
    }
    return true;
}

THeadTagWatcher::THeadTagWatcher(const THeadTagWatcherConfig* config)
    : TBase(*config)
    , Config(config)
{}
