#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/major/client.h>
#include <drive/backend/tags/tags_manager.h>

void TMajorRequester::SerializeToProto(NDrive::NProto::TMajorRequester& proto) const {
    INFO_LOG << "TMajorRequester::SerializeToProto" << Endl;
    for (const auto& i : CurrentFailedCars) {
        NDrive::NProto::TFailedCar protoCar;
        protoCar.SetCarId(i.first);
        protoCar.SetCount(i.second);
        *proto.AddFailedCars() = protoCar;
    }
}

bool TMajorRequester::DeserializeFromProto(const NDrive::NProto::TMajorRequester& proto) {
    PreviousFailedCars.clear();
    INFO_LOG << "TMajorRequester::DeserializeFromProto" << Endl;
    for (const auto& i : proto.GetFailedCars()) {
        PreviousFailedCars[i.GetCarId()] = i.GetCount();
    }
    return true;
}

void TMajorRequester::AddFailedCar(const TString& carId) const {
    auto carIt = CurrentFailedCars.find(carId);
    if (carIt == CurrentFailedCars.end()) {
        ui32 prevCount = 0;
        auto prevIt = PreviousFailedCars.find(carId);
        if (prevIt != PreviousFailedCars.end()) {
            prevCount = prevIt->second;
        }
        carIt = CurrentFailedCars.emplace(carId, prevCount).first;
    }
    carIt->second++;
}

bool TMajorRequester::IsFailedCar(const TString& carId) const {
    auto carIt = PreviousFailedCars.find(carId);
    return carIt != PreviousFailedCars.end();
}

bool TMajorRequester::HasCriticalSendLimit(const TString& carId) const {
    auto carIt = PreviousFailedCars.find(carId);
    return carIt != PreviousFailedCars.end() && carIt->second > Config->GetFailedCriticalLimit();
}

void TMajorRequester::SendFailedCarNotifications(const NDrive::IServer* server) const {
    TVector<TString> failedCarReports;
    for (const auto& car : PreviousFailedCars) {
        if (car.second > Config->GetFailedCriticalLimit()) {
            auto result = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(car.first);
            auto carInfo = result.GetResultPtr(car.first);
            if (!!carInfo) {
                failedCarReports.emplace_back(carInfo->GetHRReport() + " " + ToString(car.second));
            } else {
                failedCarReports.emplace_back(car.first + " " + ToString(car.second));
            }
        }
    }
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Cлишком много попыток отправки заявки в MajorAuto:", failedCarReports);
}

bool TMajorRequester::SendRequests(const NDrive::IServer* server) const {
    auto robotUserId = GetRobotUserId(server);
    TVector<TDBTag> tags;
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasMajorClient()) {
        ERROR_LOG << "TMajorRequester: Major api not found" << Endl;
        CurrentFailedCars = PreviousFailedCars;
        return false;
    }
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, Config->GetTagNames(), tags, session)) {
            ERROR_LOG << "TMajorRequester: cannot restore repair tags " << session.GetStringReport() << Endl;
            CurrentFailedCars = PreviousFailedCars;
            return true;
        }
    }
    TVector<TString> errors;
    TVector<TString> cancelInMajor;
    TMap<TString, TVector<TDBTag>> carTags;
    TSet<TString> carInMajor;
    if (!Config->GetCarsFilter().GetAllowedCarIds(carInMajor, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        CurrentFailedCars = PreviousFailedCars;
        return true;
    }
    for (auto&& tag : tags) {
        TRepairTagRecord* repairTag = dynamic_cast<TRepairTagRecord*>(tag.GetData().Get());
        if (!repairTag) {
            errors.emplace_back(tag.GetObjectId() + " " + tag->GetName() + " is not repair tag");
            continue;
        }
        NMajorClient::EWorkType workType = NMajorClient::EWorkType::None;
        auto description = dynamic_cast<const TRepairTagRecord::TDescription*>(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName()).Get());
        if (description) {
            workType = description->GetWorkType();
        }
        if (workType == NMajorClient::EWorkType::None || workType == NMajorClient::UpdateSTS) {
            IncorrectWorkTypeSignal.Signal(1);
            ERROR_LOG << "Unknown work type for tag " << tag->GetName() << Endl;
            AddFailedCar(tag.GetObjectId());
            continue;
        }

        if (repairTag->GetQueryId().empty()) {
            WorkTypeSignal.Signal(workType, 1);

            TString vin;
            {
                TCarsDB::TFetchResult result = server->GetDriveAPI()->GetCarsData()->GetCached(tag.GetObjectId());
                auto carPtr = result.GetResultPtr(tag.GetObjectId());
                if (carPtr && carPtr->GetVin()) {
                    vin = carPtr->GetVin();
                } else {
                    ERROR_LOG << "TMajorRequester: " << tag.GetObjectId() << " Vin not found" << Endl;
                    AddFailedCar(tag.GetObjectId());
                    continue;
                }
            }

            NMajorClient::TCreateQueryRequest::TQuery query(vin, workType);
            query.SetComment(repairTag->GetComment());
            query.SetLocalID(tag.GetTagId());
            double mileage = 0;
            {
                TRTDeviceSnapshot ds = server->GetSnapshotsManager().GetSnapshot(tag.GetObjectId());

                NDrive::TLocation location;
                if (!ds.GetLocation(location, Config->GetMaxAge())) {
                    ERROR_LOG << "TMajorRequester: " << tag.GetObjectId() << " Location not defined " << Endl;
                    if (workType == NMajorClient::Maintenance || !HasCriticalSendLimit(tag.GetObjectId())) {
                        AddFailedCar(tag.GetObjectId());
                        continue;
                    }
                } else {
                    query.SetCoord(location.Longitude, location.Latitude);
                }

                if (workType == NMajorClient::Maintenance) {
                    if (mileage = repairTag->GetMileage(); repairTag->GetMileage() == 0 && !ds.GetMileage(mileage)) {
                        ERROR_LOG << "TMajorRequester: " << tag.GetObjectId() << " mileage not defined " << Endl;
                        AddFailedCar(tag.GetObjectId());
                        continue;
                    }
                    query.SetCommonCarRun((ui64)mileage);
                }
            }
            if (workType == NMajorClient::BodyRepair) {
                query.SetInsuranceNumber(repairTag->GetInsuranceNumber());
            }

            TMessagesCollector errors;
            NMajorClient::TCreateQueryRequest::TResponse answer;
            if (server->GetDriveAPI()->GetMajorClient().CreateQuery(query, answer, errors)) {
                CreateSignal.Signal(EMajorRequestStatus::Accept, 1);
            } else {
                CreateSignal.Signal(EMajorRequestStatus::Reject, 1);
                ERROR_LOG << "TMajorRequester: " << errors.GetStringReport() << Endl;
                AddFailedCar(tag.GetObjectId());
                continue;
            }

            bool wasUpdate = false;
            TString sessionReport;
            {
                repairTag->SetQueryId(answer.GetId());
                repairTag->SetMileage(mileage);
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                wasUpdate = (server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(tag, robotUserId, session) && session.Commit());
                sessionReport = session.GetStringReport();
            }

            if (!wasUpdate) {
                ERROR_LOG << "TMajorRequester: " << tag.GetTagId() << " cannot update repair tag " << sessionReport << Endl;
                if (server->GetDriveAPI()->GetMajorClient().CancelQuery(answer.GetId(), errors)) {
                    CancelSignal.Signal(EMajorRequestStatus::Accept, 1);
                    ERROR_LOG << "TMajorRequester: " << tag.GetTagId() << " cannot update repair tag and major query was rejected" << Endl;
                } else {
                    CancelSignal.Signal(EMajorRequestStatus::Reject, 1);
                    ERROR_LOG << "TMajorRequester: Query was created but id not saved" << errors.GetStringReport() << Endl;
                }
            }
        } else {
            carTags[tag.GetObjectId()].emplace_back(tag);
        }
    }
    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCached(MakeSet(NContainer::Keys(carTags)));

    TSet<TString> majorQueryIds;
    {
        TMessagesCollector majorErrors;
        TVector<NMajorClient::TGetQueriesRequest::TQuery> openedQueries;
        if (!server->GetDriveAPI()->GetMajorClient().GetQueryList(false, openedQueries, majorErrors)) {
            ERROR_LOG << "cannot fetch major opened queries " << majorErrors.GetStringReport() << Endl;
            CurrentFailedCars = PreviousFailedCars;
            return true;
        }
        for (const auto& query : openedQueries) {
            majorQueryIds.emplace(query.GetId());
        }
    }

    TVector<TString> removedCars;
    for (auto&& car : carTags) {
        auto carInfo = gCars.GetResultPtr(car.first);
        if (!carInfo) {
            continue;
        }
        for (auto&& tag : car.second) {
            auto repairTag = tag.MutableTagAs<TRepairTagRecord>();
            if (!repairTag) {
                errors.emplace_back(tag.GetObjectId() + " " + tag->GetName() + " is not repair tag");
                continue;
            }

            if (!majorQueryIds.contains(repairTag->GetQueryId())) {
                AddFailedCar(car.first);

                TString status;
                bool isClosed = false;
                TMessagesCollector majorErrors;
                if (!server->GetDriveAPI()->GetMajorClient().GetStatus(repairTag->GetQueryId(), status, isClosed, majorErrors)) {
                    errors.emplace_back(carInfo->GetHRReport() + " " + majorErrors.GetStringReport());
                    continue;
                }

                NMajorClient::TQueryInfoRequest::TFullQueryInfo info;
                if (!server->GetDriveAPI()->GetMajorClient().GetQueryInfo(repairTag->GetQueryId(), info, majorErrors)) {
                    errors.emplace_back(carInfo->GetHRReport() + " " + majorErrors.GetStringReport());
                    continue;
                }
                TString cancelComment = "";
                info.TryGetValue(Config->GetCancelCommentName(), cancelComment);
                repairTag->SetComment(status + " " + cancelComment);
                if (!IsFailedCar(car.first)) {
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(tag, robotUserId, session) || !session.Commit()) {
                        errors.emplace_back(carInfo->GetHRReport() + " не удалось обновить причину отказа " + session.GetStringReport());
                        continue;
                    }
                }
                if (tag->GetPerformer().empty() || Config->GetRemoveWithPerformer()) {
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(tag, robotUserId, server, session, Config->GetRemoveWithPerformer()) || !session.Commit()) {
                        errors.emplace_back(carInfo->GetHRReport() + " не удалось удалить тег " + session.GetStringReport());
                    } else {
                        removedCars.emplace_back(carInfo->GetHRReport() + " " + tag->GetName() + "(" + cancelComment + ")");
                    }
                } else {
                    cancelInMajor.emplace_back(carInfo->GetHRReport() + " заявка закрыта: необходимо отказаться от исполнения " + tag->GetName() + " " + cancelComment);
                }
            }
        }
    }
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Ошибки при выполнении процесса MajorRequester:", errors);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Требуется действие от Major:", cancelInMajor);
    NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Удалены теги на машинах:", removedCars);
    return true;
}

bool TMajorRequester::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    INFO_LOG << "TMajorRequester: start" << Endl;
    CurrentFailedCars.clear();
    SendFailedCarNotifications(server);
    return SendRequests(server);
}

TMajorRequester::TMajorRequester(const TMajorRequesterConfig* config)
    : TBase(*config)
    , Config(config)
    , CreateSignal({ "frontend-major-create-request-count" }, false)
    , CancelSignal({ "frontend-major-cancel-request-count" }, false)
    , WorkTypeSignal({ "frontend-major-task-count" }, false)
    , IncorrectWorkTypeSignal({ "frontend-major-incorrect-type" }, false)
{}
