#pragma once

#include "process.h"

#include <library/cpp/messagebus/scheduler/scheduler.h>

#include <util/thread/pool.h>

class TBackgroundProcessesManagerConfig {
private:
    TVector<IBackgroundProcessConfig::TPtr> Generators;
    ui32 ThreadsCount = 8;

public:
    ui32 GetThreadsCount() const {
        return ThreadsCount;
    }

    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    const TVector<IBackgroundProcessConfig::TPtr>& GetGenerators() const {
        return Generators;
    }
};

class TBackgroundProcessesManager {
private:
    TMap<TString, IBackgroundProcess::TPtr> Processes;
    THolder<NBus::NPrivate::TScheduler> Scheduler;
    TThreadPool Queue;

    const TBackgroundProcessesManagerConfig Config;
    const IServerBase* Server = nullptr;

private:
    class TExecuteItem: public IObjectInQueue {
    private:
        IBackgroundProcess::TPtr ProcessHandle;
        TBackgroundProcessesManager* Owner;

    public:
        TExecuteItem(IBackgroundProcess::TPtr process, TBackgroundProcessesManager* owner)
            : ProcessHandle(process)
            , Owner(owner)
        {
        }

        virtual void Process(void* /*threadSpecificResource*/) override {
            ProcessHandle->Execute(Owner, ProcessHandle, Owner->Server);
        }
    };

    class TScheduledItem: public NBus::NPrivate::IScheduleItem {
    private:
        IBackgroundProcess::TPtr Process;
        TBackgroundProcessesManager* Owner;

    public:
        TScheduledItem(IBackgroundProcess::TPtr process, const TInstant startInstant, TBackgroundProcessesManager* owner)
            : NBus::NPrivate::IScheduleItem(startInstant)
            , Process(process)
            , Owner(owner)
        {
        }

        virtual void Do() {
            Owner->AddForExecute(Process);
        }
    };

    void AddForExecute(IBackgroundProcess::TPtr process) {
        if (IsActive()) {
            Queue.SafeAddAndOwn(THolder(new TExecuteItem(process, this)));
        }
    }

public:
    bool IsActive() const {
        return Scheduler != nullptr;
    }

    void RescheduleProcess(IBackgroundProcess::TPtr process, const TInstant nextInstant) {
        if (Scheduler) {
            Scheduler->Schedule(new TScheduledItem(process, nextInstant, this));
        }
    }

    void RegisterProcess(IBackgroundProcessConfig* processGenerator);

    TBackgroundProcessesManager(const TBackgroundProcessesManagerConfig& config, const IServerBase* server)
        : Queue(IThreadPool::TParams()
            .SetThreadName("background_process_manager")
        )
        , Config(config)
        , Server(server)
    {
    }

    void Start() {
        Queue.Start(Config.GetThreadsCount());
        Scheduler = MakeHolder<NBus::NPrivate::TScheduler>();
        for (auto&& i : Config.GetGenerators()) {
            RegisterProcess(i.Get());
        }
    }

    void Stop() {
        if (Scheduler) {
            Scheduler->Stop();
            Scheduler.Destroy();
        }
        Queue.Stop();
        for (auto&& i : Processes) {
            i.second->Stop();
        }
    }
};
