#include "process.h"

#include "manager.h"

#include <drive/backend/background/manager/proto/regular.pb.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/logging/events.h>

#include <kernel/daemon/common/time_guard.h>

#include <rtline/util/blob_with_header.h>

TSignalByKey<TString, double> IBackgroundProcess::ProcessExecuteSignal;
TSignalByKey<TString, double> IBackgroundProcess::ProcessErrorSignal;

class TProcessExecute: public IContextSignal<double> {
private:
    TUnistatSignal<double> SignalCounter;

public:
    TProcessExecute(const TString& processName)
        : SignalCounter({ "frontend-" + processName + "-exec-count" }, false)
    {
    }

    void Signal(const double& value) const override {
        SignalCounter.Signal(value);
    }
};

class TProcessError: public IContextSignal<double> {
private:
    TUnistatSignal<double> SignalCounter;

public:
    TProcessError(const TString& processName)
        : SignalCounter({ "frontend-" + processName + "-error-count" }, false)
    {
    }

    void Signal(const double& value) const override {
        SignalCounter.Signal(value);
    }
};

IBackgroundProcess::ERestoreResult IBackgroundProcess::RestoreState() {
    CHECK_WITH_LOG(Storage);
    TString stateData;
    if (!Storage->ExistsNode(GetStorageNodeName())) {
        return ERestoreResult::OK;
    }
    if (!Storage->GetValue(GetStorageNodeName(), stateData, -1, false)) {
        ERROR_LOG << "No data for process '" << Id << "'" << Endl;
        return ERestoreResult::IncorrectData;
    }
    TBlobWithHeader<NDrive::NProto::TStateMeta> bwh;
    if (!bwh.Load(TBlob::FromString(stateData))) {
        Storage->RemoveNode(GetStorageNodeName());
        ERROR_LOG << "Cannot restore process '" << Id << "' composite state" << Endl;
        return ERestoreResult::IncorrectData;
    }

    LastCallInstant = TInstant::Seconds(bwh.GetHeader().GetStoreTimestamp());
    DEBUG_LOG << LastCallInstant.Seconds() << " / " << GetNextCallInstant(LastCallInstant).Seconds() << Endl;
    if (GetNextCallInstant(LastCallInstant) > Now() + TDuration::Seconds(1)) {
        return ERestoreResult::TooFreshState;
    }

    if (!Deserialize(bwh.GetData())) {
        ERROR_LOG << "Cannot restore process '" << Id << "' state" << Endl;
        return ERestoreResult::IncorrectData;
    }
    return ERestoreResult::OK;
}

void IBackgroundProcess::StoreState(const TInstant callInstant) const {
    NDrive::NProto::TStateMeta header;
    header.SetStoreTimestamp(callInstant.Seconds());
    TBlobWithHeader<NDrive::NProto::TStateMeta> bwh(header, Serialize());
    const TBlob data = bwh.Save();
    Storage->SetValue(GetStorageNodeName(), TString(data.AsCharPtr(), data.Size()), false, false);
}

IBackgroundProcess::IBackgroundProcess(const IBackgroundProcessConfig& config)
    : Id(config.GetId())
    , SyncStorageName(config.GetSyncStorageName())
{
    ProcessExecuteSignal.RegisterSignal(Id, new TProcessExecute(Id));
    ProcessErrorSignal.RegisterSignal(Id, new TProcessError(Id));
}

IBackgroundProcess::~IBackgroundProcess() {
}

TString IBackgroundProcess::GetRobotUserId(const IServerBase* server) const {
    auto optionalResult = server ? server->GetRtBackgroundRobotUserId(Id) : nullptr;
    return optionalResult.GetOrElse("robot-frontend");
}

void IBackgroundProcess::Execute(TBackgroundProcessesManager* manager, TPtr self, const IServerBase* server) {
    NDrive::TEventLog::TSourceGuard sourceGuard(Id);
    Storage = server->GetVersionedStorage(SyncStorageName);
    CHECK_WITH_LOG(!!Storage);
    auto lock = Storage->WriteLockNode("regular_" + Id, TDuration::Zero());
    TTimeGuard tg("Background process '" + Id + "'");
    if (!lock) {
        INFO_LOG << "Cannot lock background process '" << Id << "'" << Endl;
        manager->RescheduleProcess(self, GetNextCallInstant(Now()));
        return;
    } else {
        INFO_LOG << "Background process '" << Id << "' locked" << Endl;
    }
    if (NeedState()) {
        switch (RestoreState()) {
            case ERestoreResult::IncorrectData:
                ERROR_LOG << "Cannot restore process '" << Id << "' state" << Endl;
                manager->RescheduleProcess(self, GetNextCallInstant(Now()));
                return;
            case ERestoreResult::TooFreshState:
                DEBUG_LOG << "Too fresh state for '" << Id << "' process" << Endl;
                manager->RescheduleProcess(self, GetNextCallInstant(LastCallInstant));
                return;
            case ERestoreResult::OK:
                break;
        }
    }

    TInstant instantCall;
    try {
        if (server->GetSettings().GetBackgroundValueDef(Id, "enabled", true) && !DoExecute(manager, self, server)) {
            ProcessErrorSignal.Signal(GetId(), 1);
            return;
        }

        instantCall = Now();
        manager->RescheduleProcess(self, GetNextCallInstant(instantCall));
        ProcessExecuteSignal.Signal(GetId(), 1);
    } catch (const std::exception& e) {
        INFO_LOG << "Cannot execute regular process: " << FormatExc(e) << Endl;
        return;
    }

    if (NeedState()) {
        StoreState(instantCall);
    }
}
