#pragma once
#include <util/generic/ptr.h>
#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/yconf/conf.h>
#include <util/stream/output.h>
#include <util/datetime/base.h>
#include <rtline/library/storage/abstract.h>
#include <util/memory/blob.h>
#include <library/cpp/logger/global/global.h>
#include <rtline/util/blob_with_header.h>
#include <rtline/library/unistat/signals.h>

class TBackgroundProcessesManager;
class IServerBase;

class IBackgroundProcessConfig;

class IBackgroundProcess {
public:
    using TPtr = TAtomicSharedPtr<IBackgroundProcess>;

protected:
    TInstant LastCallInstant = Now();
    NRTLine::IVersionedStorage::TPtr Storage;
    TString Id;
    TString SyncStorageName;

    static TSignalByKey<TString, double> ProcessExecuteSignal;
    static TSignalByKey<TString, double> ProcessErrorSignal;

    TString GetStorageNodeName() const {
        const TString salt = GetSaltForStatusKey();
        return "regular_" + Id + "_state" + (!!salt ? ("_" + salt) : "");
    }

    virtual TString GetSaltForStatusKey() const {
        return "";
    }

    virtual TInstant GetNextCallInstant(const TInstant lastCallInstant) const = 0;

    enum class ERestoreResult {
        OK = 0,
        IncorrectData = 1,
        TooFreshState = 2
    };

    virtual bool Deserialize(const TBlob& /*data*/) {
        return true;
    }

    virtual TBlob Serialize() const {
        return TBlob();
    }

    virtual bool DoExecute(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const IServerBase* server) = 0;

    virtual ERestoreResult RestoreState() final;
    virtual void StoreState(const TInstant callInstant) const final;
    virtual bool NeedState() const {
        return true;
    }

public:
    IBackgroundProcess(const IBackgroundProcessConfig& config);
    virtual ~IBackgroundProcess();

    const TString& GetId() const {
        return Id;
    }

    TString GetRobotUserId(const IServerBase* server) const;

    virtual void Execute(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const IServerBase* server);

    virtual TInstant GetStartInstant() const {
        return Now();
    }

    virtual void Start() = 0;
    virtual void Stop() = 0;
};

template <class TProto, class TBaseClass>
using ISerializableProtoBackgroundProcess = IProtoSerializable<TProto, TBaseClass>;

class IBackgroundProcessConfig {
private:
    TString Type;
    TString Id;
    TString SyncStorageName;

public:
    using TPtr = TAtomicSharedPtr<IBackgroundProcessConfig>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IBackgroundProcessConfig, TString, TString, TString>;

public:
    virtual ~IBackgroundProcessConfig() = default;

    IBackgroundProcessConfig(const TString& type, const TString& id)
        : Type(type)
        , Id(id)
    {
    }

    const TString& GetSyncStorageName() const {
        return SyncStorageName;
    }

    const TString& GetId() const {
        return Id;
    }

    const TString& GetType() const {
        return Type;
    }

    virtual void Init(const TYandexConfig::Section* section) {
        SyncStorageName = section->GetDirectives().Value("SyncStorageName", SyncStorageName);
        AssertCorrectConfig(!!SyncStorageName, "Incorrect SyncStorageName for regular process " + Id);
    }

    virtual void ToString(IOutputStream& os) const {
        os << "SyncStorageName: " << SyncStorageName << Endl;
    }

    virtual IBackgroundProcess* Construct() const = 0;
};
