#pragma once
#include "process.h"

#include <drive/backend/abstract/base.h>
#include <rtline/library/unistat/cache.h>

#include <util/generic/cast.h>

class IBackgroundRegularProcessConfig: public IBackgroundProcessConfig {
private:
    TDuration Period = TDuration::Seconds(100);
    bool DryRunMode = false;
    TSet<ui16> Timetable;

public:
    using IBackgroundProcessConfig::IBackgroundProcessConfig;

    TDuration GetPeriod() const {
        return Period;
    }

    virtual void Init(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;

    TInstant GetNextCallInstant(const TInstant lastCallInstant) const;

    bool GetDryRunMode() const {
        return DryRunMode;
    }
};

template <class TServer>
class IBackgroundRegularProcessImpl: public IBackgroundProcess {
private:
    const IBackgroundRegularProcessConfig* Config;

protected:
    virtual bool DoExecuteImpl(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const TServer* server) const = 0;

    TInstant GetNextCallInstant(const TInstant lastCallInstant) const override {
        return Config->GetNextCallInstant(lastCallInstant);
    }

public:
    IBackgroundRegularProcessImpl(const IBackgroundProcessConfig& config)
        : IBackgroundProcess(config)
        , Config(VerifyDynamicCast<const IBackgroundRegularProcessConfig*>(&config))
    {
    }

    virtual bool DoExecute(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const IServerBase* server) override try {
        bool result = DoExecuteImpl(manager, self, server->GetAsPtrSafe<TServer>());
        INFO_LOG << "Background process " << GetId() << " iteration finished" << Endl;
        TUnistatSignalsCache::SignalAdd(GetId(), "background-success-exec", 1);
        return result;
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": an exception occurred: " << FormatExc(e) << Endl;
        TUnistatSignalsCache::SignalAdd(GetId(), "background-exception", 1);
        return true;
    }
};

using IBaseBackgroundRegularProcess = IBackgroundRegularProcessImpl<IServerBase>;
