#include "processor.h"

#include "config.h"

#include <drive/backend/background/common/common.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/notifier.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

void TNoTagsWatcher::SerializeToProto(NDrive::NProto::TDeviceProblemsState& /*proto*/) const {
    INFO_LOG << "TNoTagsWatcher::SerializeToProto" << Endl;
}

bool TNoTagsWatcher::DeserializeFromProto(const NDrive::NProto::TDeviceProblemsState& /*proto*/) {
    INFO_LOG << "TNoTagsWatcher::DeserializeFromProto" << Endl;
    return true;
}

bool TNoTagsWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    auto robotUserId = GetRobotUserId(server);
    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

    TRecordsSet records;
    auto result = session->Exec("SELECT * FROM \"car\" WHERE NOT id IN (SELECT DISTINCT object_id FROM car_tags)", &records);

    if (!result || !result->IsSucceed()) {
        ERROR_LOG << "cannot take cars withno tags: " << session.GetStringReport() << Endl;
        return true;
    }

    NDrive::INotifier::TPtr notifier = server->GetNotifier(Config->GetNotifierName());
    if (records.empty()) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Все машины протегированы. Коррекция не требуется."));
        }
        return true;
    }

    session.SetComment("no tags detection");
    for (auto&& i : records) {
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord(Config->GetTagName()), robotUserId, i.Get("id"), server, session)) {
            ERROR_LOG << "cannot add definition tag (" << Config->GetTagName() << ") for  car " << i.Get("id") << ": " << session.GetStringReport() << Endl;
            return true;
        }
    }

    if (!session.Commit()) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Не получается инициировать идентифицирующие теги " + Config->GetTagName() + " для " + ToString(records.size()) + " машин: " + session.GetStringReport()));
        }
        ERROR_LOG << "cannot init identifier tags for cars: " << session.GetStringReport() << Endl;
        return true;
    }
    if (notifier) {
        TSet<TString> objects;
        for (auto&& i : records) {
            objects.emplace(i.Get("id"));
        }

        auto gCars = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(objects);

        TStringStream ss = "Добавили теги типа " + Config->GetTagName() + " для идентификации машин:\n";
        TStringStream ssList;
        ui32 idx = 0;
        for (auto&& i : objects) {
            if (++idx == 50) {
                notifier->Notify(NDrive::INotifier::TMessage(ss.Str() + ssList.Str()));
                ssList.clear();
                idx = 0;
            }
            auto* carInfo = gCars.GetResultPtr(i);
            if (carInfo) {
                ssList << carInfo->GetHRReport() << Endl;
            } else {
                ssList << i << Endl;
            }
        }
        if (idx) {
            notifier->Notify(NDrive::INotifier::TMessage(ss.Str() + ssList.Str()));
        }

    }
    return true;
}

TNoTagsWatcher::TNoTagsWatcher(const TNoTagsWatcherConfig* config)
    : TBase(*config)
    , Config(config)
{

}
