#include "config.h"

#include "processor.h"

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <util/stream/file.h>
#include <util/string/strip.h>
#include <util/string/vector.h>

IBackgroundProcess* TParkingPaymentWatcherConfig::Construct() const {
    return new TParkingPaymentWatcher(*this);
}

void TParkingPaymentWatcherConfig::Init(const TYandexConfig::Section* section) {
    IBackgroundRegularProcessConfig::Init(section);
    const auto& directives = section->GetDirectives();
    ApplicationId = directives.Value("ApplicationId", ApplicationId);
    GeoObjectsHost = directives.Value("GeoObjectsHost", GeoObjectsHost);
    GeoObjectsPort = directives.Value("GeoObjectsPort", GeoObjectsPort);
    GeoObjectsService = directives.Value("GeoObjectsService", GeoObjectsService);
    KznClientToken = directives.Value("KznClientToken", KznClientToken);
    MskClientToken = directives.Value("MskClientToken", MskClientToken);
    SpbClientToken = directives.Value("SpbClientToken", SpbClientToken);
    TagName = directives.Value("TagName", TagName);
    TracksApiName = directives.Value("TracksApiName", TracksApiName);
    TracksApiTimeout = TDuration::Parse(directives.Value("TracksApiTimeout", TracksApiTimeout.ToString()));
    DryRun = directives.Value("DryRun", DryRun);
    const auto nonParkableStatuses = directives.Value("NonParkableStatuses", JoinStrings(NonParkableStatuses.begin(), NonParkableStatuses.end(), ","));
    StringSplitter(nonParkableStatuses).Split(',').SkipEmpty().Collect(&NonParkableStatuses);
    const auto includedModels = directives.Value("IncludedModels", JoinStrings(IncludedModels.begin(), IncludedModels.end(), ","));
    StringSplitter(includedModels).Split(',').SkipEmpty().Collect(&IncludedModels);
    const auto excludedModels = directives.Value("ExcludedModels", JoinStrings(ExcludedModels.begin(), ExcludedModels.end(), ","));
    StringSplitter(excludedModels).Split(',').SkipEmpty().Collect(&ExcludedModels);
    Notifier = directives.Value("Notifier", Notifier);
    NotifySuccess = directives.Value("NotifySuccess", NotifySuccess);
    TokenGroupsFile = directives.Value("TokenGroupsFile", TokenGroupsFile);
    const auto tariffTimetableString = directives.Value("TariffTimetable", TariffTimetable.SerializeToJson().GetStringRobust());
    NJson::TJsonValue tariffTimetable;
    AssertCorrectConfig(NJson::ReadJsonFastTree(tariffTimetableString, &tariffTimetable), "cannot read TariffTimetable as Json");
    AssertCorrectConfig(TariffTimetable.DeserializeFromJson(tariffTimetable), "cannot parse TariffTimetable");
    Quantum = TDuration::Parse(directives.Value("Quantum", Quantum.ToString()));
    SwitchMskAggregatorId = directives.Value("SwitchMskAggregatorId", SwitchMskAggregatorId);

    if (TokenGroupsFile) {
        TIFStream input(TokenGroupsFile);
        TString line;
        while (input.ReadLine(line)) {
            auto tokens = StringSplitter(line).Split('\t').ToList<TString>();
            AssertCorrectConfig(tokens.size() == 2, "incorrect tokens count for line: %s", line.c_str());
            TokenGroups[tokens[0]] = tokens[1];
        }
    }
    {
        const TYandexConfig::TSectionsMap children = section->GetAllChildren();
        auto it = children.find("MskParkingConfig");
        if (it != children.end()) {
            MskParkingConfig = MakeHolder<NDrive::TMskParkingPaymentConfig>();
            MskParkingConfig->Init(it->second);
        }
    }
}

void TParkingPaymentWatcherConfig::ToString(IOutputStream& os) const {
    IBackgroundRegularProcessConfig::ToString(os);
    os << "ApplicationId: " << ApplicationId << Endl;
    os << "GeoObjectsHost: " << GeoObjectsHost << Endl;
    os << "GeoObjectsPort: " << GeoObjectsPort << Endl;
    os << "GeoObjectsService: " << GeoObjectsService << Endl;
    os << "KznClientToken: " << KznClientToken << Endl;
    os << "MskClientToken: " << MskClientToken << Endl;
    os << "SpbClientToken: " << SpbClientToken << Endl;
    os << "TagName: " << TagName << Endl;
    os << "TracksApiName: " << TracksApiName << Endl;
    os << "TracksApiTimeout: " << TracksApiTimeout << Endl;
    os << "IncludedModels: " << JoinStrings(IncludedModels.begin(), IncludedModels.end(), ",") << Endl;
    os << "ExcludedModels: " << JoinStrings(ExcludedModels.begin(), ExcludedModels.end(), ",") << Endl;
    os << "DryRun: " << DryRun << Endl;
    os << "NonParkableStatuses: " << JoinStrings(NonParkableStatuses.begin(), NonParkableStatuses.end(), ",");
    os << "Notifier: " << Notifier << Endl;
    os << "NotifySuccess: " << NotifySuccess << Endl;
    os << "TokenGroupsFile: " << TokenGroupsFile << Endl;
    os << "TariffTimetable: " << TariffTimetable.SerializeToJson().GetStringRobust() << Endl;
    os << "Quantum: " << Quantum << Endl;
    if (MskParkingConfig) {
        os << "<MskParkingConfig>" << Endl;
        MskParkingConfig->ToString(os);
        os << "</MskParkingConfig>" << Quantum << Endl;
    }
    os << "SwitchMskAggregatorId: " << SwitchMskAggregatorId << Endl;
}

IBackgroundProcess* TParkingBalanceWatcherConfig::Construct() const {
    return new TParkingBalanceWatcher(*this);
}

void TParkingBalanceWatcherConfig::Init(const TYandexConfig::Section* section) {
    IBackgroundRegularProcessConfig::Init(section);
    const auto& directives = section->GetDirectives();

    CertificateFile = directives.Value("CertificateFile", CertificateFile);
    PrivateKeyFile = directives.Value("PrivateKeyFile", PrivateKeyFile);
    PrivateKeyPassword = directives.Value("PrivateKeyPassword", PrivateKeyPassword);

    if (CertificateFile || PrivateKeyFile) {
        Y_ENSURE(CertificateFile, "certificate file is not set");
        Y_ENSURE(PrivateKeyFile, "private key file is not set");
    }

    WalletsFile = directives.Value("WalletsFile", WalletsFile);
    if (WalletsFile) {
        TIFStream input(WalletsFile);
        TString line;
        while (input.ReadLine(line)) {
            TString stripped = StripInPlace(line);
            if (stripped) {
                Wallets.insert(stripped);
            }
        }
        Y_ENSURE(Wallets.size(), "no wallets registered");
    }

    AgentId = directives.Value("AgentId", AgentId);
    ApplicationId = directives.Value("ApplicationId", ApplicationId);
    ShopId = directives.Value("ShopId", ShopId);
    ShopArticleId = directives.Value("ShopArticleId", ShopArticleId);

    if (AgentId || ApplicationId || ShopId || ShopArticleId) {
        Y_ENSURE(AgentId, "agentId is not set");
        Y_ENSURE(ApplicationId, "applicationId is not set");
        Y_ENSURE(ShopId, "shopId is not set");
        Y_ENSURE(ShopArticleId, "shopArticleId is not set");
    }

    KznClientToken = directives.Value("KznClientToken", KznClientToken);
    SpbClientToken = directives.Value("SpbClientToken", SpbClientToken);

    Notifier = directives.Value("Notifier", Notifier);
    Threshold = directives.Value("Threshold", Threshold);
    PaymentAmount = directives.Value("PaymentAmount", PaymentAmount);
    WalletThreshold = directives.Value("WalletThreshold", WalletThreshold);
    Y_ENSURE(Threshold > 0);
    Y_ENSURE(PaymentAmount > 0);
    Y_ENSURE(WalletThreshold > 0);
}

void TParkingBalanceWatcherConfig::ToString(IOutputStream& os) const {
    IBackgroundRegularProcessConfig::ToString(os);

    os << "CertificateFile: " << CertificateFile << Endl;
    os << "PrivateKeyFile: " << PrivateKeyFile << Endl;
    os << "PrivateKeyPassword: " << PrivateKeyPassword << Endl;
    os << "WalletsFile: " << WalletsFile << Endl;

    os << "AgentId: " << AgentId << Endl;
    os << "ApplicationId: " << ApplicationId << Endl;
    os << "ShopId: " << ShopId << Endl;
    os << "ShopArticleId: " << ShopArticleId << Endl;

    os << "KznClientToken: " << KznClientToken << Endl;
    os << "SpbClientToken: " << SpbClientToken << Endl;
    os << "Notifier: " << Notifier << Endl;
    os << "Threshold: " << Threshold << Endl;
    os << "PaymentAmount: " << PaymentAmount << Endl;
    os << "WalletThreshold: " << WalletThreshold << Endl;
}

IBackgroundRegularProcessConfig::TFactory::TRegistrator<TParkingPaymentWatcherConfig> ParkingPaymentWatcherRegistrator("parking_payment");
IBackgroundRegularProcessConfig::TFactory::TRegistrator<TParkingBalanceWatcherConfig> ParkingBalanceWatcherRegistrator("parking_balance");
