#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/parking_payment.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags.h>

#include <drive/library/cpp/parking/api/client.h>
#include <drive/library/cpp/parking/api/money.h>
#include <drive/library/cpp/parking/inhouse/client.h>
#include <drive/library/cpp/tracks/client.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/random/random.h>

NUnistat::TIntervals PriceIntervals = {
    40,
    60,
    80,
    100,
    200,
};

NUnistat::TIntervals RefundIntervals = {
    0,
    10,
    20,
    30,
    40,
    50,
    100,
    200,
};

TUnistatSignal<> SignalParkingPrice({ "parking-price" }, PriceIntervals);
TUnistatSignal<> SignalParkingStart({ "parking-start" }, false);
TUnistatSignal<> SignalParkingRefund({ "parking-refund" }, RefundIntervals);
TUnistatSignal<> SignalParkingStop({ "parking-stop" }, false);
TUnistatSignal<> SignalParkingStartError({ "parking-start-error" }, false);
TUnistatSignal<> SignalParkingStopError({ "parking-stop-error" }, false);

TUnistatSignal<> SignalRemainingWalletsCount({ "yamoney-wallets-remaining" }, EAggregationType::LastValue, "annn");

TParkingPaymentWatcher::TParkingPaymentWatcher(const TParkingPaymentWatcherConfig& config)
    : IBackgroundRegularProcessImpl<NDrive::IServer>(config)
    , Config(config)
    , KznClient(config.GetKznClientToken())
    , MskClient(config.GetMskClientToken())
    , SpbClient(config.GetSpbClientToken())
{
    if (config.GetMskParkingConfig()) {
        MskParkingClient = MakeHolder<NDrive::TMskParkingPaymentClient>(*config.GetMskParkingConfig());
    }
}

template <>
THolder<NDrive::TParkingListClient> TParkingPaymentWatcher::CreateLocator<NDrive::TParkingListClient>(const TString& /*token*/) const {
    return MakeHolder<NDrive::TParkingListClient>(Config.GetGeoObjectsHost(), Config.GetGeoObjectsPost(), Config.GetGeoObjectsService());
}

bool TParkingPaymentWatcher::DoExecuteImpl(TBackgroundProcessesManager* backgroundProcessesManager, IBackgroundProcess::TPtr self, const NDrive::IServer* server) const {
    Y_UNUSED(backgroundProcessesManager);
    Y_UNUSED(self);
    const TDriveAPI* api = server->GetDriveAPI();
    const IDriveTagsManager& manager = api->GetTagsManager();
    const TDeviceTagsManager& deviceTagManager = manager.GetDeviceTags();
    const TString& robotUserId = GetRobotUserId(server);
    const ISettings& settings = server->GetSettings();
    NDrive::INotifier::TPtr notifier = Config.GetNotifier() ? server->GetNotifier(Config.GetNotifier()) : nullptr;
    auto notifyParkingExceptions = settings.GetValue<bool>("parking_payment.notify_parking_exceptions").GetOrElse(false);
    auto useParkingCostStub = settings.GetValue<bool>("parking_payment.use_stub_parking_cost").GetOrElse(false);

    INFO_LOG << GetId() << ": " << "wake up" << Endl;
    TString runId = ToString(Now().MicroSeconds()) + "-" + ToString(RandomNumber<ui32>());
    TString tagName = Config.GetTagName();
    if (!tagName) {
        tagName = Config.IsDryRun() ? TParkingInfoTag::DryRunTagName() : TParkingInfoTag::TagName();
    }
    TVector<TTaggedDevice> objects;
    {
        auto tags = { tagName };
        if (!manager.GetDeviceTags().GetObjectsFromCache(tags, objects, tags, Now())) {
            ERROR_LOG << GetId() << ": " << "Cannot get tagged objects " << Endl;
            if (notifier) {
                notifier->Notify(TString("Cannot get tagged objects"));
            }
        }
    }
    INFO_LOG << GetId() << ": " << "Got " << objects.size() << " tagged devices" << Endl;

    TSet<TString> objectIds;
    for (auto&& object : objects) {
        objectIds.emplace(object.GetId());
    }
    TCarsDB::TFetchResult fetchResult = api->GetCarsData()->FetchInfo(objectIds);

    ui32 startedParkingObjects = 0;
    ui32 finishedParkingObjects = 0;
    ui32 totalParkedObjects = 0;
    ui32 totalNonParkedObjects = 0;
    ui32 totalNoLocation = 0;
    ui32 totalChanges = 0;
    ui32 errors = 0;

    TString blockingObjectTagName = settings.GetValue<TString>("parking_payment.blocking_tag").GetOrElse("parking_payment_blocker");
    TSet<TString> blockingPaymentMethods = StringSplitter(settings.GetValue<TString>("parking_payment.banned_payment_methods").GetOrElse({})).Split(',').SkipEmpty();
    TMap<TString, TDBTag> ParkableObjects;
    TMap<TString, TDBTag> NonParkableObjects;

    auto statuses = server->GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
    auto sessionBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    auto itStatus = statuses.begin();
    const TString unknownStatus = "unknown";
    for (auto&& i : fetchResult.GetResult()) {
        const TString& id = i.first;
        if (!Config.GetIncludedModels().empty()) {
            if (!Config.GetIncludedModels().contains(i.second.GetModel())) {
                continue;
            }
        }
        if (!Config.GetExcludedModels().empty()) {
            if (Config.GetExcludedModels().contains(i.second.GetModel())) {
                continue;
            }
        }

        while (itStatus != statuses.end() && itStatus->first < id) {
            ++itStatus;
        }
        const TString& status = (itStatus != statuses.end() && itStatus->first == id) ? itStatus->second : unknownStatus;
        TString paymentMethod;
        TString disabledByTagId;
        bool disabledByPaymentMethod = false;
        bool idle = false;
        auto idleStatus = EIdleStatus::Unknown;
        bool parkable = !Config.GetNonParkableStatuses().contains(status);
        if (!parkable) {
            idleStatus = GetIdleStatus(id, server);
            idle = idleStatus == EIdleStatus::IsIdle;
            parkable = idle;
        }
        if (parkable && sessionBuilder && !blockingPaymentMethods.empty()) {
            auto session = sessionBuilder->GetLastObjectSession(id);
            auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
            auto billingCompilation = billingSession ? billingSession->GetCompilationAs<TBillingSession::TBillingCompilation>() : Nothing();
            if (billingCompilation && billingCompilation->IsAccepted() && !billingCompilation->GetIsFinished()) {
                auto offer = billingCompilation->GetCurrentOffer();
                paymentMethod = offer ? offer->GetSelectedCharge() : TString();
                if (blockingPaymentMethods.contains(paymentMethod)) {
                    disabledByPaymentMethod = true;
                    parkable = false;
                }
            }
        }
        if (parkable && blockingObjectTagName) {
            auto bannedTag = manager.GetDeviceTags().GetTagFromCache(id, blockingObjectTagName);
            if (bannedTag && *bannedTag) {
                disabledByTagId = bannedTag->GetTagId();
                parkable = false;
            }
        }
        if (!parkable) {
            NonParkableObjects[id] = {};
        } else {
            ParkableObjects[id] = {};
        }
        NDrive::TEventLog::Log("SetParkableStatus", NJson::TMapBuilder
            ("disabled_by_payment_method", disabledByPaymentMethod)
            ("disabled_by_tag_id", NJson::ToJson(NJson::Nullable(disabledByTagId)))
            ("id", id)
            ("idle", idle)
            ("idle_status", ToString(idleStatus))
            ("run", runId)
            ("parkable", parkable)
            ("payment_method", NJson::ToJson(NJson::Nullable(paymentMethod)))
            ("status", status)
        );
    }
    for (auto&& object : objects) {
        const TString& id = object.GetId();
        TDBTag tagInfo;
        for (auto&& tag : object.GetTags()) {
            if (tag->GetName() == tagName) {
                tagInfo = tag;
                continue;
            }
        }
        if (!tagInfo.GetTagId()) {
            WARNING_LOG << GetId() << ": " << "ParkingInfo tag for " << id << " is not found" << Endl;
            ParkableObjects.erase(id);
            NonParkableObjects.erase(id);
            continue;
        }

        auto parkable = ParkableObjects.find(id);
        if (parkable != ParkableObjects.end()) {
            parkable->second = tagInfo;
        }

        auto nonparkable = NonParkableObjects.find(id);
        if (nonparkable != NonParkableObjects.end()) {
            nonparkable->second = tagInfo;
        }
    }

    for (auto&& [id, tag] : NonParkableObjects) {
        auto parkingInfo = tag.MutableTagAs<TParkingInfoTag>();
        if (!parkingInfo) {
            ERROR_LOG << GetId() << ": cannot cast tag " << tag.GetTagId() << Endl;
            continue;
        }

        try {
            INFO_LOG << GetId() << ": " << "Started processing parking info for " << id << Endl;
            bool changed = false;
            auto now = Now();
            for (auto&& slot : parkingInfo->GetSlots()) {
                if (!slot.IsActive(now)) {
                    continue;
                }
                if (Config.IsDryRun()) {
                    NDrive::TEventLog::Log("DryRunStoppingParking", NJson::TMapBuilder("id", id)("run", runId)("slot", slot.ToJson()));
                    INFO_LOG << GetId() << ": " << "DryRun Stopping parking session " << slot.ParkingSessionId << " for " << id << Endl;
                    NJson::TJsonValue refund;
                    refund["refund"]["amount"] = 0;
                    refund["refund"]["currency"] = "RUB";
                    slot.Refund = refund;
                    INFO_LOG << GetId() << ": " << "DryRun Stopped parking session " << slot.ParkingSessionId << " for " << id << ": " << slot.Refund.GetStringRobust() << Endl;
                    NDrive::TEventLog::Log("DryRunStoppedParking", NJson::TMapBuilder("id", id)("run", runId)("slot", slot.ToJson()));
                } else {
                    NDrive::TEventLog::Log("StoppingParking", NJson::TMapBuilder("id", id)("run", runId)("slot", slot.ToJson()));
                    INFO_LOG << GetId() << ": " << "Stopping parking session " << slot.ParkingSessionId << " for " << id << Endl;
                    switch (slot.SessionAggregatorId) {
                    case NDrive::TParkingListClient::KznAggregatorId:
                        slot.Refund = KznClient.StopParkingRobust(slot.ParkingSessionId).ToJson();
                        break;
                    case NDrive::TParkingListClient::SpbAggregatorId:
                        if (Config.GetSpbClientToken()) {
                            slot.Refund = SpbClient.StopParkingRobust(slot.ParkingSessionId).ToJson();
                        }
                        break;
                    case NDrive::TParkingListClient::FitDevMskParkingId:
                        if (Config.GetMskClientToken()) {
                            slot.Refund = MskClient.StopParkingRobust(slot.ParkingSessionId).ToJson();
                        }
                        break;
                    case NDrive::TParkingListClient::AmppAggregatorId:
                        if (MskParkingClient) {
                            auto car = fetchResult.GetResultPtr(id);
                            if (!car) {
                                throw yexception() << "unknown car" << id;
                            }
                            auto futureSession = MskParkingClient->StopParking(car->GetNumber());
                            if (!futureSession.Wait(MskParkingClient->GetConfig().GetRequestTimeout())) {
                                throw yexception() << "stop parking failed by timeout";
                            }
                            auto session = futureSession.ExtractValue();
                            slot.Session = session.ToJson();
                            slot.SessionFinish = session.EndTime;
                            slot.Refund = NJson::TJsonValue(NJson::JSON_MAP);
                        }
                        break;
                    default:
                    {
                        NDrive::TParkingPaymentClient3 client(slot.Token, Config.GetApplicationId());
                        slot.Refund = client.StopParkingRobust(slot.ParkingSessionId).ToJson();
                        break;
                    }
                    }
                    INFO_LOG << GetId() << ": " << "Stopped parking session " << slot.ParkingSessionId << " for " << id << ": " << slot.Refund.GetStringRobust() << Endl;
                    NDrive::TEventLog::Log("StoppedParking", NJson::TMapBuilder("id", id)("run", runId)("slot", slot.ToJson()));
                    SignalParkingStop.Signal(1);
                    SignalParkingRefund.Signal(slot.Refund["refund"]["amount"].GetDouble());
                }
                changed = true;
            }
            if (changed) {
                INFO_LOG << GetId() << ": " << "Committing changes for " << id << Endl;
                auto session = deviceTagManager.BuildTx<NSQL::Writable>();
                if (deviceTagManager.UpdateTagData(tag, robotUserId, session) && session.Commit()) {
                    INFO_LOG << GetId() << ": " << "Committed changes for " << id << Endl;
                    NDrive::TEventLog::Log("CommitParking", tag->SerializeToJson());
                    if (notifier && Config.ShouldNotifySuccess()) {
                        notifier->Notify("successfully stopped parking for " + fetchResult.GetResult().at(id).GetHRReport());
                    }
                    finishedParkingObjects += 1;
                    totalChanges += 1;
                } else {
                    ERROR_LOG << GetId() << ": " << "Could not commit changes for " << id << ": " << session.GetStringReport() << Endl;
                    NDrive::TEventLog::Log("CommitParkingFailure", session.GetReport());
                    if (notifier) {
                        notifier->Notify("cannot commit stopped parking for " + fetchResult.GetResult().at(id).GetHRReport() + ": " + session.GetStringReport());
                    }
                    errors += 1;
                }
            }
            INFO_LOG << GetId() << ": " << "Finished processing parking info for " << id << Endl;
        } catch (const std::exception& e) {
            TString error = FormatExc(e);
            TString message = "Cannot process parking info for " + fetchResult.GetResult().at(id).GetHRReport() + ": " + error;
            ERROR_LOG << GetId() << ": " << message << Endl;
            NDrive::TEventLog::Log("ParkingException", NJson::TMapBuilder("id", id)("error", error));
            SignalParkingStopError.Signal(1);

            if (notifier && notifyParkingExceptions) {
                notifier->Notify(message);
            }
            errors += 1;
        }
    }

    TDeviceLocationOptions locationOptions = server->GetSnapshotsManager().GetLocationOptions();
    THolder<NDrive::TParkingListClient> locator;
    auto listParkings = [&](const TDBTag& tag) -> NThreading::TFuture<typename NDrive::TParkingListClient::TParkings> {
        const TString& id = tag.GetObjectId();
        const auto parkingInfo = tag.GetTagAs<TParkingInfoTag>();
        Y_ENSURE(parkingInfo, "cannot cast tag " << tag.GetTagId());
        if (!parkingInfo->IsActive(Now())) {
            auto snapshot = server->GetSnapshotsManager().GetSnapshot(id);
            NDrive::TLocation location;
            if (snapshot.GetLocation(location, TDuration::Max(), locationOptions)) {
                INFO_LOG << GetId() << ": " << "Location for " << id << ": " << location.ToJson().GetStringRobust() << Endl;
                const NDrive::TLocationTags locationTags = api->GetTagsInPoint(location.GetCoord());
                if (locationTags.contains("no_parking_payment")) {
                    INFO_LOG << GetId() << ": " << "Skip listing parking due to no_parking_payment tag: " << id << Endl;
                    NDrive::TEventLog::Log("SkipListingParking", NJson::TMapBuilder
                        ("id", id)
                        ("run", runId)
                        ("latitude", location.Latitude)
                        ("longitude", location.Longitude)
                        ("location_tags", NJson::ToJson(locationTags))
                    );
                    return NThreading::MakeFuture(typename NDrive::TParkingListClient::TParkings());
                }

                if (!locator) {
                    locator = CreateLocator<NDrive::TParkingListClient>(parkingInfo->GetSlots().at(0).Token);
                }

                NDrive::TEventLog::Log("ListingParking", NJson::TMapBuilder
                    ("id", id)
                    ("run", runId)
                    ("latitude", location.Latitude)
                    ("longitude", location.Longitude)
                );
                return locator->GetParkings(location.Latitude, location.Longitude);
            } else {
                WARNING_LOG << GetId() << ": " << "No location data for " << id << Endl;
                totalNoLocation += 1;
            }
        } else {
            totalParkedObjects += 1;
        }
        return NThreading::MakeFuture(typename NDrive::TParkingListClient::TParkings());
    };

    TMap<TString, NThreading::TFuture<typename NDrive::TParkingListClient::TParkings>> idToParkings;
    /*
    for (auto&& [id, tag] : ParkableObjects) {
        try {
            idToParkings[id] = listParkings(tag);
        } catch (const std::exception& e) {
            ERROR_LOG << GetId() << ": " << "cannot prefetch parkings: " << FormatExc(e) << Endl;
        }
    }
    */

    TMap<TString, ui32> groupParkingCount;
    groupParkingCount[Default<TString>()] = Max<ui16>();
    for (auto&& [id, tag] : ParkableObjects) {
        const auto parkingInfo = dynamic_cast<TParkingInfoTag*>(tag.GetData().Get());
        if (!parkingInfo) {
            continue;
        }
        for (auto&& slot : parkingInfo->GetSlots()) {
            const TString& group = GetTokenGroup(slot.Token);
            if (!group) {
                continue;
            }
            if (slot.IsActive(Now())) {
                groupParkingCount[group]++;
            }
        }
    }

    auto startAmppParking = [this, &fetchResult](const typename NDrive::TParkingPaymentClient3& /* client */, const NDrive::TParkingPaymentClient3::TOffer& offer, const TDuration duration, const TString& carId) {
        if (!MskParkingClient) {
            throw yexception() << "msk client is not set";
        }
        auto car = fetchResult.GetResultPtr(carId);
        if (!car) {
            throw yexception() << "unknown car to start " << carId;
        }
        auto futureSession = MskParkingClient->StartParking(car->GetNumber(), offer.Parking.ParkingId, duration);
        if (!futureSession.Wait(MskParkingClient->GetConfig().GetRequestTimeout())) {
            throw yexception() << "start parking failed by timeout";
        }
        return futureSession.ExtractValue();
    };

    auto startSimpleParking = [](const typename NDrive::TParkingPaymentClient3& client, const NDrive::TParkingPaymentClient3::TOffer& offer, const TDuration duration, const TString& /* carId */) {
        return client.StartParkingRobust(offer, duration);
    };

    TMap<TString, ui32> costsMap;
    if (MskParkingClient) {
        try {
            auto futureMap = MskParkingClient->GetCosts();
            if (!futureMap.Wait(MskParkingClient->GetConfig().GetRequestTimeout())) {
                throw yexception() << "parking zones failed by timeout";
            }
            costsMap = futureMap.ExtractValue();
            NDrive::TEventLog::Log("ParkingZones", NJson::ToJson(costsMap));
        } catch (const std::exception& e) {
            TString error = FormatExc(e);
            TString message = "Cannot process parking zones map";
            ERROR_LOG << GetId() << ": " << message << Endl;
            NDrive::TEventLog::Log("ParkingException", NJson::TMapBuilder("run", runId)("error", error));
            SignalParkingStartError.Signal(1);
            if (notifier && notifyParkingExceptions) {
                notifier->Notify(message);
            }
            auto costsString = settings.GetValue<TString>("parking_payment.default_parking_costs");
            if (costsString) {
                NJson::TJsonValue costsJson;
                if (!NJson::ReadJsonFastTree(*costsString, &costsJson) || !NJson::TryFromJson(costsJson, costsMap)) {
                    ERROR_LOG << GetId() << ": cannot parse default_parking_costs from " << costsString << Endl;
                }
            }
        }
    }

    auto getParkingCost = [&](const NDrive::TParkingPaymentClient3::TParking& parking) {
        TMaybe<ui32> result;
        if (!result) {
            auto cost = costsMap.FindPtr(parking.ParkingId);
            if (cost) {
                result = *cost;
            }
        }
        if (!result) {
            if (useParkingCostStub) {
                result = 0;
            }
        }
        return result;
    };

    for (auto&& [objectId, tag] : ParkableObjects) {
        const auto& id = objectId;
        auto parkingInfo = tag.MutableTagAs<TParkingInfoTag>();
        if (!parkingInfo) {
            ERROR_LOG << GetId() << ": cannot cast tag " << tag.GetTagId() << Endl;
            continue;
        }

        TVector<TParkingInfoTag::TSlot*> slots;
        for (auto&& slot : parkingInfo->GetSlots()) {
            slots.push_back(&slot);
        }
        std::sort(slots.begin(), slots.end(), [&](TParkingInfoTag::TSlot* left, TParkingInfoTag::TSlot* right) {
            if (!left) {
                return true;
            }
            if (!right) {
                return false;
            }
            return groupParkingCount[GetTokenGroup(left->Token)] < groupParkingCount[GetTokenGroup(right->Token)];
        });

        try {
            INFO_LOG << GetId() << ": " << "Started processing parking info for " << id << Endl;
            ui32 changed = 0;
            {
                {
                    auto p = idToParkings.find(id);
                    auto parkingsF = p != idToParkings.end() ? p->second : listParkings(tag);
                    auto parkings = parkingsF.GetValue(TDuration::Seconds(10));

                    TMaybe<ui32> maxPrice;
                    NJson::TJsonValue parks(NJson::JSON_ARRAY);
                    for (auto&& parking : parkings) {
                        parks.AppendValue(parking.ToJson());
                        auto price = getParkingCost(parking);
                        maxPrice = Max<ui32>(price.GetOrElse(0), maxPrice.GetOrElse(0));
                    }
                    NDrive::TEventLog::Log("ListedParking", NJson::TMapBuilder("id", id)("run", runId)("parkings", std::move(parks)));

                    TSet<i64> prices;
                    auto startParking3 = [&] (const typename NDrive::TParkingPaymentClient3::TParking& parking, const auto& startMskParking) {
                        if (MskParkingClient) {
                            if (!prices.empty()) {
                                INFO_LOG << GetId() << ": " << "Skipping parking " << parking.ToJson().GetStringRobust() << Endl;
                                NDrive::TEventLog::Log("SkipOffer", NJson::TMapBuilder("id", id)("run", runId)("parking", parking.ToJson())("reason", "paid_max_price"));
                                return 0;
                            }
                        }
                        TVector<TString> errors;
                        TInstant now = Now();
                        for (auto&& pslot : slots) {
                            if (!pslot) {
                                continue;
                            }

                            auto& slot = *pslot;
                            try {
                                if (slot.IsActive(now)) {
                                    continue;
                                }

                                NDrive::TParkingPaymentClient3 client(slot.Token, Config.GetApplicationId());
                                auto requiredDuration = std::min(Config.GetQuantum(), Config.GetTariffTimetable().GetNextCorrectionTime(now) - now);
                                auto duration = TDuration::Minutes(std::max<ui64>(requiredDuration.Minutes() / 30, 1) * 30);
                                NDrive::TParkingPaymentClient3::TOffer offer;
                                if (MskParkingClient) {
                                    auto parkingCost = getParkingCost(parking);
                                    if (auto it = parkingCost) {
                                        offer.Cost.Amount = *it;
                                        offer.Parking = parking;
                                    } else if (maxPrice.GetOrElse(0)) {
                                        INFO_LOG << GetId() << ": " << "Skipping parking " << parking.ToJson().GetStringRobust() << Endl;
                                        NDrive::TEventLog::Log("SkipOffer", NJson::TMapBuilder("id", id)("run", runId)("parking", parking.ToJson())("reason", "unknown_price"));
                                        return 0;
                                    } else {
                                        throw yexception() << "fail to find zone" << parking.ToJson().GetStringRobust();
                                    }
                                } else {
                                    auto vehicle = client.GetVehicleByReference(slot.ParkingVehicleId);
                                    auto pc3Parking = parking;
                                    pc3Parking.AggregatorId = NDrive::TParkingListClient::MskAggregatorId;
                                    offer = client.GetOffer(pc3Parking, vehicle, duration);
                                }

                                auto price = static_cast<i64>(offer.Cost.Amount);
                                if (maxPrice && price < *maxPrice) {
                                    INFO_LOG << GetId() << ": " << "Skipping offer " << offer.ToJson().GetStringRobust() << Endl;
                                    NDrive::TEventLog::Log("SkipOffer", NJson::TMapBuilder("id", id)("run", runId)("offer", offer.ToJson())("reason", "low_price"));
                                    return 0;
                                }
                                if (prices.contains(price)) {
                                    INFO_LOG << GetId() << ": " << "Skipping offer " << offer.ToJson().GetStringRobust() << Endl;
                                    NDrive::TEventLog::Log("SkipOffer", NJson::TMapBuilder("id", id)("run", runId)("offer", offer.ToJson())("reason", "paid_price"));
                                    return 0;
                                }

                                typename NDrive::TParkingPaymentClient3::TSession session;
                                if (Config.IsDryRun()) {
                                    NDrive::TEventLog::Log("DryRunStartingParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("offer", offer.ToJson()));
                                    TInstant start = now;
                                    TInstant finish = start + duration;
                                    session.Id = offer.Id;
                                    session.ParkingStartTime = start.ToStringLocal();
                                    session.ParkingEndTime = finish.ToStringLocal();
                                    INFO_LOG << GetId() << ": " << "DryRun Started parking session " << session.ToJson().GetStringRobust() << " for " << id << Endl;
                                    NDrive::TEventLog::Log("DryRunStartedParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("session", session.ToJson()));
                                } else {
                                    NDrive::TEventLog::Log("StartingParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("offer", offer.ToJson()));
                                    session = startMskParking(client, offer, duration, id);
                                    INFO_LOG << GetId() << ": " << "Started parking session " << session.ToJson().GetStringRobust() << " for " << id << Endl;
                                    NDrive::TEventLog::Log("StartedParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("session", session.ToJson()));
                                    SignalParkingStart.Signal(1);
                                    SignalParkingPrice.Signal(price);
                                }
                                if (!session.Parking) {
                                    session.Parking = parking;
                                }

                                if (notifier && Config.ShouldNotifySuccess()) {
                                    TStringStream ss;
                                    ss << "parked " << fetchResult.GetResult().at(id).GetHRReport() << " at " << offer.Parking.GetParkingName() << Endl;
                                    ss << "cost: " << offer.Cost.Amount << " " << offer.Cost.Currency << Endl;
                                    ss << "from: " << session.ParkingStartTime << Endl;
                                    ss << "to: " << session.ParkingEndTime << Endl;
                                    notifier->Notify(ss.Str());
                                }

                                slot.Cost = offer.Cost.Amount;
                                slot.ParkingSessionId = session.Id;
                                slot.Session = session.ToJson();
                                slot.SessionAggregatorId = parking.AggregatorId;
                                slot.SessionFinish = session.EndTime;
                                slot.Refund = NJson::JSON_NULL;
                                prices.insert(price);

                                const TString& group = GetTokenGroup(slot.Token);
                                if (group) {
                                    groupParkingCount[group]++;
                                }

                                return 1;
                            } catch (const yexception& e) {
                                WARNING_LOG << GetId() << ": " << "Cannot start parking for " << id << " using token " << slot.Token << ": " << e.what() << Endl;
                                errors.push_back(e.what());
                            }
                        }
                        TStringStream totalError;
                        for (auto&& error : errors) {
                            totalError << error << Endl;
                        }
                        throw yexception() << "cannot park " << id << " on " << parking.ToJson().GetStringRobust() << ":\n" << totalError.Str();
                    };

                    auto startParkingFitDev = [&](const NDrive::TFitDevParkingPaymentClient& client, const NDrive::TParkingPaymentClient3::TParking& parking) {
                        if (!prices.empty()) {
                            return 0;
                        }

                        auto objectInfo = fetchResult.GetResultPtr(id);
                        Y_ENSURE(objectInfo, "cannot find ObjectInfo for " << id);
                        NDrive::TFitDevParkingPaymentClient::TVehicle vehicle;
                        vehicle.LicensePlate = objectInfo->GetNumber();
                        vehicle.Synonym = id;

                        TVector<TString> errors;
                        TInstant now = Now();
                        for (auto&& pslot : slots) {
                            if (!pslot) {
                                continue;
                            }

                            auto& slot = *pslot;
                            try {
                                if (slot.IsActive(now)) {
                                    continue;
                                }

                                auto duration = Config.GetQuantum();
                                auto offer = client.GetOffer(parking, vehicle, duration);
                                auto price = static_cast<i64>(offer.Cost.Amount);

                                typename NDrive::TParkingPaymentClient3::TSession session;
                                if (Config.IsDryRun()) {
                                    NDrive::TEventLog::Log("DryRunStartingParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("offer", offer.ToJson()));
                                    TInstant start = now;
                                    TInstant finish = start + duration;
                                    session.Id = offer.Id;
                                    session.ParkingStartTime = start.ToStringLocal();
                                    session.ParkingEndTime = finish.ToStringLocal();
                                    INFO_LOG << GetId() << ": " << "DryRun Started parking session " << session.ToJson().GetStringRobust() << " for " << id << Endl;
                                    NDrive::TEventLog::Log("DryRunStartedParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("session", session.ToJson()));
                                } else {
                                    NDrive::TEventLog::Log("StartingParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("offer", offer.ToJson()));
                                    session = client.StartParkingRobust(offer, duration);
                                    INFO_LOG << GetId() << ": " << "Started parking session " << session.ToJson().GetStringRobust() << " for " << id << Endl;
                                    NDrive::TEventLog::Log("StartedParking", NJson::TMapBuilder("id", id)("run", runId)("token", slot.Token)("session", session.ToJson()));
                                    SignalParkingStart.Signal(1);
                                    SignalParkingPrice.Signal(price);
                                }

                                slot.Cost = offer.Cost.Amount;
                                slot.ParkingSessionId = session.Id;
                                slot.Session = session.ToJson();
                                slot.SessionAggregatorId = parking.AggregatorId;
                                slot.SessionFinish = session.EndTime;
                                slot.Refund = NJson::JSON_NULL;
                                prices.insert(price);

                                return 1;
                            } catch (const yexception& e) {
                                WARNING_LOG << GetId() << ": " << "Cannot start parking for " << id << " using token " << slot.Token << ": " << e.what() << Endl;
                                errors.push_back(e.what());
                            }
                        }
                        TStringStream totalError;
                        for (auto&& error : errors) {
                            totalError << error << Endl;
                        }
                        throw yexception() << "cannot park " << id << " on " << parking.ToJson().GetStringRobust() << ":\n" << totalError.Str();
                    };

                    for (size_t i = 0; i < parkings.size(); ++i) {
                        auto& parking = parkings.at(i);
                        if ((parking.AggregatorId == NDrive::TParkingListClient::MskAggregatorId || parking.AggregatorId == NDrive::TParkingListClient::AmppAggregatorId)
                            && (ui64)parking.AggregatorId != Config.GetSwitchMskAggregatorId())
                        {
                            NDrive::TEventLog::Log("SwitchMskAggregatorId", NJson::TMapBuilder
                                ("id", id)
                                ("run", runId)
                                ("origin_id", parking.AggregatorId)
                                ("new_id", Config.GetSwitchMskAggregatorId())
                            );
                            parking.AggregatorId = Config.GetSwitchMskAggregatorId();
                        }
                        switch (parking.AggregatorId) {
                        case NDrive::TParkingListClient::KznAggregatorId:
                            changed += startParkingFitDev(KznClient, parking);
                            break;
                        case NDrive::TParkingListClient::SpbAggregatorId:
                            if (Config.GetSpbClientToken()) {
                                changed += startParkingFitDev(SpbClient, parking);
                                break;
                            }
                            NDrive::TEventLog::Log("SkipParking", NJson::TMapBuilder
                                ("id", id)
                                ("run", runId)
                                ("parking", parking.ToJson())
                            );
                            break;
                        case NDrive::TParkingListClient::FitDevMskParkingId:
                            if (Config.GetMskClientToken()) {
                                changed += startParkingFitDev(MskClient, parking);
                                break;
                            }
                            NDrive::TEventLog::Log("SkipParking", NJson::TMapBuilder
                                ("id", id)
                                ("run", runId)
                                ("parking", parking.ToJson())
                            );
                            break;
                        case NDrive::TParkingListClient::AmppAggregatorId:
                            changed += startParking3(parking, startAmppParking);
                            break;
                        case NDrive::TParkingListClient::MskAggregatorId:
                            changed += startParking3(parking, startSimpleParking);
                            break;
                        }
                    }
                    if (parkings.empty()) {
                        totalNonParkedObjects += 1;
                    }
                }
            }
            if (changed) {
                INFO_LOG << GetId() << ": " << "Committing changes for " << id << Endl;
                auto session = deviceTagManager.BuildTx<NSQL::Writable>();
                if (deviceTagManager.UpdateTagData(tag, robotUserId, session) && session.Commit()) {
                    INFO_LOG << GetId() << ": " << "Committed changes for " << id << Endl;
                    NDrive::TEventLog::Log("CommitParking", tag->SerializeToJson());
                    if (notifier && Config.ShouldNotifySuccess()) {
                        notifier->Notify("successfully started " + ToString(changed) + " parkings for " + fetchResult.GetResult().at(id).GetHRReport());
                    }
                    startedParkingObjects += 1;
                    totalParkedObjects += 1;
                    totalChanges += 1;
                } else {
                    ERROR_LOG << GetId() << ": " << "Could not commit changes for " << id << ": " << session.GetStringReport() << Endl;
                    NDrive::TEventLog::Log("CommitParkingFailure", session.GetReport());
                    if (notifier) {
                        notifier->Notify("cannot commit stopped parking for " + fetchResult.GetResult().at(id).GetHRReport() + ": " + session.GetStringReport());
                    }
                    errors += 1;
                }
            }
            INFO_LOG << GetId() << ": " << "Finished processing parking info for " << id << Endl;
        } catch (const std::exception& e) {
            TString error = FormatExc(e);
            TString message = "Cannot process parking info for " + fetchResult.GetResult().at(id).GetHRReport() + ": " + error;
            ERROR_LOG << GetId() << ": " << message << Endl;
            NDrive::TEventLog::Log("ParkingException", NJson::TMapBuilder("id", id)("run", runId)("error", error));
            SignalParkingStartError.Signal(1);

            if (notifier && notifyParkingExceptions) {
                notifier->Notify(message);
            }
            errors += 1;
        }
    }
    if (notifier && (totalChanges || errors)) {
        TStringStream ss;
        ss << GetId() << Endl;
        ss << "Parkable status: " << ParkableObjects.size() << Endl;
        ss << "Non-parkable status: " << NonParkableObjects.size() << Endl;
        ss << "Started parkings: " << startedParkingObjects << Endl;
        ss << "Finished parkings: " << finishedParkingObjects << Endl;
        ss << "Total parked: " << totalParkedObjects << Endl;
        ss << "Total not required: " << totalNonParkedObjects << Endl;
        ss << "Total no location: " << totalNoLocation << Endl;
        ss << "Total errors: " << errors << Endl;
        notifier->Notify(ss.Str());
    }
    INFO_LOG << GetId() << ": " << "ParkingPaymentWatcher sleep" << Endl;
    return true;
}

const TString& TParkingPaymentWatcher::GetTokenGroup(const TString& token) const {
    auto p = Config.GetTokenGroups().find(token);
    if (p != Config.GetTokenGroups().end()) {
        return p->second;
    } else {
        return Default<TString>();
    }
}

TParkingPaymentWatcher::EIdleStatus TParkingPaymentWatcher::GetIdleStatus(const TString& id, const NDrive::IServer* server) const {
    const TString& tracksApiName = Config.GetTracksApiName();
    if (!tracksApiName) {
        return EIdleStatus::ConfigError;
    }
    const auto tracksApi = server->GetRTLineAPI(tracksApiName);
    if (!tracksApi) {
        ERROR_LOG << GetId() << ": " << "cannot find Tracks API " << tracksApiName << Endl;
        return EIdleStatus::ConfigError;
    }
    NDrive::TTracksClient tracksClient(tracksApi->GetSearchClient());

    auto now = Now();
    auto idleDuration = Config.GetIdleDuration();
    auto idleThreshold = now - idleDuration;
    auto snapshot = server->GetSnapshotsManager().GetSnapshot(id);
    auto speed = snapshot.GetSensor(VEGA_SPEED, idleDuration);
    if (speed && std::abs(speed->ConvertTo<double>()) > 1) {
        return EIdleStatus::SensorSpeed;
    }

    NDrive::TTrackQuery trackQuery;
    trackQuery.DeviceId = id;
    trackQuery.Since = idleThreshold;
    auto asyncTracks = tracksClient.GetTracks(trackQuery, Config.GetTracksApiTimeout());
    if (!asyncTracks.Wait(Config.GetTracksApiTimeout())) {
        ERROR_LOG << GetId() << ": " << "wait timeout for " << id << Endl;
        return EIdleStatus::Timeout;
    }
    if (!asyncTracks.HasValue()) {
        const TString exception = NThreading::GetExceptionMessage(asyncTracks);
        NDrive::TEventLog::Log("SetParkableStatusError", NJson::TMapBuilder
            ("id", id)
            ("error", exception)
        );
        ERROR_LOG << GetId() << ": " << "exception for " << id << ": " << exception << Endl;
        return EIdleStatus::Exception;
    }

    auto tracks = asyncTracks.ExtractValue();
    auto tail = NDrive::GetTail(tracks, idleThreshold);
    if (!tail || tail->Coordinates.empty()) {
        ERROR_LOG << GetId() << ": " << "empty tail for " << id << Endl;
        return EIdleStatus::EmptyTail;
    }
    for (auto&& coordinate : tail->Coordinates) {
        if (coordinate.Speed > 1) {
            INFO_LOG << GetId() << ": " << id << " is not idle " << coordinate.Timestamp << Endl;
            return EIdleStatus::TailSpeed;
        }
    }
    NOTICE_LOG << GetId() << ": " << id << " is idle" << Endl;
    return EIdleStatus::IsIdle;
}

TParkingBalanceWatcher::TParkingBalanceWatcher(const TParkingBalanceWatcherConfig& config)
    : TBase(config)
    , Config(config)
    , ActiveSpent(0)
{
}

bool TParkingBalanceWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const TDriveAPI* api = server->GetDriveAPI();
    const IDriveTagsManager& manager = api->GetTagsManager();
    NDrive::INotifier::TPtr notifier = Config.GetNotifier() ? server->GetNotifier(Config.GetNotifier()) : nullptr;

    NDrive::TYandexMoneyClient::TOptions ymco;
    ymco.AgentId = Config.GetAgentId();
    ymco.ShopId = Config.GetShopId();
    ymco.ShopArticleId = Config.GetShopArticleId();

    ymco.CertificateFile = Config.GetCertificateFile();
    ymco.PrivateKeyFile = Config.GetPrivateKeyFile();
    ymco.PrivateKeyPassword = Config.GetPrivateKeyPassword();

    INFO_LOG << GetId() << ": " << "ParkingBalanceWatcher wake up" << Endl;
    if (Config.PaymentsEnabled()) try {
        NDrive::TYandexMoneyClient ymc(ymco);
        auto balance = ymc.BalanceSafe(Seconds());
        Y_ENSURE(balance.Code == NDrive::TYandexMoneyClient::ECode::Success, "cannot request Balance " << int(balance.Code) << ' ' << balance.Error);
        INFO_LOG << GetId() << ": " << "Account balance " << balance.Value << Endl;
        if (notifier) {
            notifier->Notify("Acquired account balance " + ToString(balance.Value));
        }
        MakeBalanceNotification(balance.Value, "msk", *server);
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": " << "Cannot check account balance: " << FormatExc(e) << Endl;
    }

    if (Config.GetKznClientToken()) try {
        NDrive::TFitDevParkingPaymentClient client(Config.GetKznClientToken());
        auto balance = client.GetBalance();
        INFO_LOG << GetId() << ": " << "Kzn balance " << balance.Amount << Endl;
        if (notifier) {
            notifier->Notify("Acquired kzn balance " + ToString(balance.Amount));
        }
        MakeBalanceNotification(balance.Amount, "kzn", *server);
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": " << "Cannot check kzn balance: " << FormatExc(e) << Endl;
    }

    if (Config.GetSpbClientToken()) try {
        NDrive::TSpbParkingPaymentClient client(Config.GetSpbClientToken());
        auto balance = client.GetBalance();
        INFO_LOG << GetId() << ": " << "Spb balance " << balance.Amount << Endl;
        if (notifier) {
            notifier->Notify("Acquired spb balance " + ToString(balance.Amount));
        }
        MakeBalanceNotification(balance.Amount, "spb", *server);
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": " << "Cannot check spb balance: " << FormatExc(e) << Endl;
    }

    TVector<TTaggedDevice> objects;
    {
        auto tags = { TParkingInfoTag::TagName() };
        if (!manager.GetDeviceTags().GetObjectsFromCache(tags, objects, tags, Now())) {
            ERROR_LOG << GetId() << ": " << "Cannot receive objects from cache" << Endl;
            if (notifier) {
                notifier->Notify(TString("Cannot receive objects from cache"));
            }
            return true;
        }
    }
    INFO_LOG << GetId() << ": " << "Got " << objects.size() << " tagged devices" << Endl;

    NJson::TJsonValue parkingJson;
    parkingJson["name"] = "fake";
    parkingJson["aggregatorId"] = 2;
    parkingJson["id"] = 4007;
    parkingJson["coordinates"]["latitude"] = 55.77878425;
    parkingJson["coordinates"]["longitude"] = 37.68308326;
    NDrive::TParkingPaymentClient3::TParking parking;
    parking.FromJson(parkingJson);

    try {
        GetWallet();
    } catch (const std::exception& e) {
        DEBUG_LOG << GetId() << ": " << FormatExc(e) << Endl;
    }

    TSet<TString> tokens;
    TMap<i32, TString> lows;
    for (auto&& object : objects) {
        const TString& id = object.GetId();
        TDBTag tagInfo;
        for (auto&& tag : object.GetTags()) {
            if (tag->GetName() == TParkingInfoTag::TagName()) {
                tagInfo = tag;
                break;
            }
        }
        if (!tagInfo) {
            WARNING_LOG << GetId() << ": " << "ParkingInfo tag for " << id << " is not found" << Endl;
            continue;
        }
        const auto parkingInfo = dynamic_cast<TParkingInfoTag*>(tagInfo.GetData().Get());
        CHECK_WITH_LOG(parkingInfo);
        for (auto&& slot : parkingInfo->GetSlots()) {
            if (!tokens.insert(slot.Token).second) {
                INFO_LOG << GetId() << ": " << "Skip " << slot.Token << Endl;
                continue;
            }
            try {
                NDrive::TParkingPaymentClient3 client(slot.Token, Config.GetApplicationId());
                NDrive::TEventLog::Log("CheckingParkingBalance", NJson::TMapBuilder
                    ("token", slot.Token)
                    ("parking", parking.ToJson())
                );
                auto balance = client.GetBalance(parking);
                NDrive::TEventLog::Log("CheckedParkingBalance", NJson::TMapBuilder
                    ("token", slot.Token)
                    ("balance", balance.ToJson())
                );
                if (balance.Amount < Config.GetThreshold()) {
                    auto value = static_cast<i32>(balance.Amount);
                    lows[value] = slot.Token;
                }
                INFO_LOG << GetId() << ": " << slot.Token << " balance " << balance.ToJson().GetStringRobust() << Endl;
                if (Config.PaymentsEnabled() && balance.Amount < Config.GetPaymentAmount()) {
                    NDrive::TParkingPaymentClient3 ppc(slot.Token, Config.GetApplicationId());
                    NDrive::TYandexMoneyClient ymc(ymco);

                    auto amount = Config.GetPaymentAmount();
                    auto vehicles = ppc.GetVehicles();
                    auto vehicle = vehicles.at(0);
                    auto offer = ppc.GetOffer(parking, vehicle, TDuration::Minutes(30));
                    auto contractId = ymc.PaymentReservationSafe(offer.Id, amount).ContractId;
                    INFO_LOG << GetId() << ": " << slot.Token << ": contractId " << contractId << Endl;

                    auto wallet = GetWallet();
                    NDrive::TEventLog::Log("MakeDepositionRequest", NJson::TMapBuilder
                        ("token", slot.Token)
                        ("offerId", offer.Id)
                        ("contractId", contractId)
                        ("amount", amount)
                        ("wallet", wallet)
                    );
                    NDrive::TYandexMoneyClient::TDepositionOptions dopt(ymco.AgentId, Seconds());
                    auto deposit = ymc.MakeDepositionSafe(contractId, wallet, dopt, amount);
                    NDrive::TEventLog::Log("MakeDepositionResponse", NJson::TMapBuilder
                        ("token", slot.Token)
                        ("offerId", offer.Id)
                        ("code", static_cast<ui32>(deposit.Code))
                        ("error", deposit.Error)
                    );
                    INFO_LOG << GetId() << ": " << slot.Token << ": deposit " << int(deposit.Code) << ' ' << deposit.Error << " " << deposit.Message << Endl;
                    if (deposit.Error == 44 || deposit.Error == 48) {
                        WARNING_LOG << GetId() << ": " << "Skipping wallet " << wallet;
                        NextWallet();
                    }
                    Y_ENSURE(deposit.Code == NDrive::TYandexMoneyClient::ECode::Success, "cannot MakeDeposition " << int(deposit.Code) << ' ' << deposit.Error << ' ' << deposit.Message);
                    UseWallet(amount);
                    StoreState(Now());

                    if (notifier) {
                        notifier->Notify("MakeDeposition " + slot.Token + " " + ToString(amount));
                    }

                    auto refund = ppc.StopParkingRobust(offer.Id);
                    NDrive::TEventLog::Log("MakeDepositionRefund", NJson::TMapBuilder
                        ("offerId", offer.Id)
                        ("token", slot.Token)
                        ("refund", refund.ToJson())
                    );

                    auto balance2 = client.GetBalance(parking);
                    NDrive::TEventLog::Log("CheckedParkingBalance2", NJson::TMapBuilder
                        ("offerId", offer.Id)
                        ("token", slot.Token)
                        ("balance", balance2.ToJson())
                    );
                }
            } catch (const std::exception& e) {
                ERROR_LOG << GetId() << ": " << "Cannot check balance for " << slot.Token << ": " << FormatExc(e) << Endl;
                NDrive::TEventLog::Log("CheckParkingBalanceException", NJson::TMapBuilder
                    ("token", slot.Token)
                    ("error", FormatExc(e))
                );
            }
        }
    }
    if (notifier && !lows.empty()) {
        TStringStream ss;
        ss << "Low balance:" << Endl;
        for (auto&& i : lows) {
            ss << i.second << " " << i.first << Endl;
        }
        notifier->Notify(ss.Str());
    }
    INFO_LOG << GetId() << ": " << "ParkingBalanceWatcher sleep" << Endl;
    return true;
}

void TParkingBalanceWatcher::SerializeToProto(NDrive::NProto::TParkingBalanceData& proto) const {
    proto.SetActiveWallet(ActiveWallet);
    proto.SetActiveSpent(ActiveSpent);
}

bool TParkingBalanceWatcher::DeserializeFromProto(const NDrive::NProto::TParkingBalanceData& proto) {
    ActiveWallet = proto.GetActiveWallet();
    ActiveSpent = proto.GetActiveSpent();
    return true;
}

TString TParkingBalanceWatcher::GetWallet() const {
    Y_ENSURE(!Config.GetWallets().empty(), "no wallets registered");
    if (!ActiveWallet) {
        ActiveWallet = *Config.GetWallets().begin();
    } else if (ActiveSpent >= Config.GetWalletThreshold()) {
        NextWallet();
    }
    auto i = Config.GetWallets().find(ActiveWallet);
    if (i != Config.GetWallets().end()) {
        auto remaining = std::distance(i, Config.GetWallets().end());
        SignalRemainingWalletsCount.Signal(remaining);
    }
    return ActiveWallet;
}

void TParkingBalanceWatcher::NextWallet() const {
    auto i = Config.GetWallets().lower_bound(ActiveWallet);
    if (i == Config.GetWallets().end()) {
        i = Config.GetWallets().begin();
    }
    while (i != Config.GetWallets().end() && *i == ActiveWallet) {
        i++;
    }
    if (i == Config.GetWallets().end()) {
        ythrow yexception() << "no wallets left after " << ActiveWallet;
    }
    ActiveWallet = *i;
    ActiveSpent = 0;
}

void TParkingBalanceWatcher::UseWallet(double amount) const {
    ActiveSpent += amount;
}

void TParkingBalanceWatcher::MakeBalanceNotification(double balance, const TString& segment, const NDrive::IServer& server) const {
    const ISettings& settings = server.GetSettings();

    auto thresholdNotifierName = settings.GetValue<TString>("parking_payment." + segment + ".balance.notification.notifier").GetOrElse({});
    auto thresholdNotifier = thresholdNotifierName ? server.GetNotifier(thresholdNotifierName) : nullptr;
    auto threshold = settings.GetValue<double>("parking_payment." + segment + ".balance.notification.threshold").GetOrElse(0);
    auto templateId = settings.GetValue<TString>("parking_payment." + segment + ".balance.notification.template").GetOrElse({});
    auto rrString = settings.GetValue<TString>("parking_payment." + segment + ".balance.notification.recipients").GetOrElse({});
    auto rr = StringSplitter(rrString).SplitBySet(", ").ToList<TString>();
    if (threshold > balance && thresholdNotifier && templateId && !rr.empty()) {
        NJson::TJsonValue body;
        body["amount"] = ToString(balance);
        NDrive::INotifier::TMessage message(body.GetStringRobust());
        message.SetHeader(templateId);
        NDrive::INotifier::TContext context;
        for (auto&& r : rr) {
            TUserContacts recipient;
            recipient.SetEmail(r);
            context.MutableRecipients().push_back(std::move(recipient));
        }
        auto result = thresholdNotifier->Notify(message, context);
        if (result && !result->HasErrors()) {
            INFO_LOG << GetId() << ": " << segment << " successful notification" << Endl;
        } else if (result) {
            ERROR_LOG << GetId() << ": " << segment << " notification error: " << result->SerializeToJson().GetStringRobust() << Endl;
        } else {
            ERROR_LOG << GetId() << ": " << segment << " null notification result" << Endl;
        }
    }
}
