#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/service_session.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

bool TNewAutoRegularReport::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    TSet<TString> filteredCars;
    if (!Config->GetCarsFilter().GetAllowedCarIds(filteredCars, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        return true;
    }

    auto gCars = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(filteredCars);

    TVector<TDBTag> tags;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(filteredCars, Config->GetAdditionalTags(), tags, session)) {
            ERROR_LOG << "cannot take current tags" << Endl;
            return true;
        }
    }

    TMap<TString, TVector<TString>> tagsByCar;
    for (const auto& tag : tags) {
        tagsByCar[tag.GetObjectId()].push_back(tag->GetName());
    }

    TStringStream csvReport;
    csvReport << "vin;model;imei;deptrans_documents_date;parking_permit_start_date;tags" << Endl;
    for (const auto& car : gCars.GetResult()) {
        const TCarRegistryDocument* baseDocuments = nullptr;
        TCarGenericAttachment attachment;
        if (server->GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(car.first, EDocumentAttachmentType::CarRegistryDocument, attachment)) {
            baseDocuments = dynamic_cast<const TCarRegistryDocument*>(attachment.Get());
        }

        csvReport << car.second.GetVin() << ";"
            << car.second.GetModel() << ";"
            << (!car.second.GetIMEI().empty() ? "смонтирована" : "не смонтирована") << ";";
        if (!baseDocuments) {
            csvReport << ";;";
        } else {
            csvReport << (baseDocuments->GetDeptransDocumentsDate() != TInstant::Zero() ? baseDocuments->GetDeptransDocumentsDate().FormatLocalTime("%d %b %Y %H:%M:%S") : "") << ";"
                << (baseDocuments->GetParkingPermitStartDate() != TInstant::Zero() ? baseDocuments->GetParkingPermitStartDate().FormatLocalTime("%d %b %Y %H:%M:%S") : "") << ";";
        }
        auto additionalTags = tagsByCar.find(car.first);
        if (additionalTags != tagsByCar.end()) {
            csvReport << JoinSeq(",", additionalTags->second);
        }
        csvReport << Endl;
    }

    TString stringDay = ToString(TInstant::Now()).substr(0, 10);
    NDrive::INotifier::TMessage message(stringDay, csvReport.Str());
    message.SetAdditionalInfo(Config->GetDescription());
    message.SetTitle(stringDay + ".csv");

    if (!NDrive::INotifier::SendDocument(server->GetNotifier(Config->GetNotifierName()), message, "text/csv", NDrive::INotifier::TContext().SetServer(server))) {
        ERROR_LOG << "Notifier internal error" << Endl;
    }
    return true;
}

TNewAutoRegularReport::TNewAutoRegularReport(const TNewAutoRegularReportConfig* config)
    : TBase(*config)
    , Config(config)
{
}

bool TCleaningRegularReport::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server->GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "Mds client not defined" << Endl;
        return false;
    }
    auto bucket = server->GetDriveAPI()->GetMDSClient().GetBucket(Config->GetBucketName());
    if (!bucket) {
        ERROR_LOG << "Mds bucket not found '" << Config->GetBucketName() << "'" << Endl;
        return false;
    }

    TSet<TString> filteredCars;
    if (!Config->GetCarsFilter().GetAllowedCarIds(filteredCars, server)) {
        ERROR_LOG << "cannot take filtered cars" << Endl;
        return true;
    }
    auto carsInfo = server->GetDriveAPI()->GetCarsData()->GetCached(filteredCars);

    TMaybe<TSet<TString>> optionalFilteredUsers;
    if (!Config->GetRoles().empty()) {
        TSet<TString> filteredUsers;
        if (!server->GetDriveAPI()->GetUsersData()->GetRoles().GetUsersWithAtLeastOneRoles(Config->GetRoles(), filteredUsers, true)) {
            ERROR_LOG << "cannot take users by roles" << Endl;
            return true;
        }
        optionalFilteredUsers = std::move(filteredUsers);
    }

    auto sessionBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("service", Now());
    if (!sessionBuilder) {
        ERROR_LOG << "Incorrect session builder 'service'" << Endl;
        return true;
    }
    const ui64 maxEventId = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetLockedMaxEventId();
    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = sessionBuilder->GetSessionsActualSinceId(LastEventId);
    std::sort(sessions.begin(), sessions.end(), [](IEventsSession<TCarTagHistoryEvent>::TPtr a, IEventsSession<TCarTagHistoryEvent>::TPtr b) -> bool {
        return a->GetLastEventId() < b->GetLastEventId();
    });
    TInstant currentTime = TInstant::Now();
    ui64 nextReal = LastEventId;
    size_t count = 1;
    for (const auto& serviceSession : sessions) {
        if (count > Config->GetMaxSessionsCount()) {
            break;
        }

        if (serviceSession->GetLastEventId() > maxEventId) {
            break;
        }

        if (serviceSession->GetLastTS() + Config->GetPhotoDelay() > currentTime) {
            continue;
        }

        nextReal = Max(nextReal, serviceSession->GetLastEventId());

        TServiceSession::TCompilation compilation(server->GetDriveAPI()->GetTagsManager().GetTagsMeta());
        if (!serviceSession->GetClosed() || serviceSession->GetLastEventId() <= LastEventId || !serviceSession->FillCompilation(compilation)) {
            continue;
        }

        if (!Config->GetTags().contains(compilation.GetTagName())) {
            continue;
        }

        auto carInfo = carsInfo.GetResultPtr(serviceSession->GetObjectId());
        if (!carInfo) {
            continue;
        }

        const auto& userId = serviceSession->GetUserId();
        if (optionalFilteredUsers && !optionalFilteredUsers->contains(userId)) {
            continue;
        }

        auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
        auto userInfo = userFetchResult.GetResultPtr(userId);
        if (!userInfo) {
            continue;
        }

        TString tagName = compilation.GetTagName();
        auto tagDescription = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(compilation.GetTagName());
        if (!!tagDescription) {
            tagName = tagDescription->GetDisplayName();
        }

        TString lastEventInfo = "";
        TMaybe<TCarTagHistoryEvent> lastEv = serviceSession->GetLastEvent();
        if (lastEv.Defined()) {
            if (lastEv->GetHistoryAction() == EObjectHistoryAction::Remove) {
                lastEventInfo = "ВЫПОЛНЕНО(" + ToString(++LastSuccessfulTaskId) + ")";
            } else {
                lastEventInfo = "ОТМЕНЕНО " + lastEv->GetHistoryComment();
            }
        }

        TStringStream caption;
        caption << tagName << " " << lastEventInfo << Endl;
        caption << carInfo->GetNumber() << Endl;
        caption << (userInfo->GetFullName() ? userInfo->GetFullName() : userInfo->GetLogin()) << Endl;
        caption << "Начало: " << serviceSession->GetStartTS().ToRfc822StringLocal() << Endl;
        caption << "Конец: " << serviceSession->GetLastTS().ToRfc822StringLocal() << Endl;

        auto images = compilation.GetImages();
        if (images.empty()) {
            NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), caption.Str() + "фото до/после не найдены");
            continue;
        } else {
            NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), caption.Str());
        }

        for (const auto& image : images) {
            TString binaryImage;
            bool loadingSuccess = false;
            TMessagesCollector errors;
            if (bucket->GetFile(image.second.Path, binaryImage, errors) / 100 == 2) {
                loadingSuccess = true;
                NDrive::INotifier::TMessage message(image.second.Marker, binaryImage);
                message.SetTitle("image.jpg");
                message.SetAdditionalInfo("");
                loadingSuccess = NDrive::INotifier::SendPhoto(server->GetNotifier(Config->GetNotifierName()), message, NDrive::INotifier::TContext().SetServer(server));
            }
            if (!loadingSuccess) {
                NDrive::INotifier::Notify(server->GetNotifier(Config->GetNotifierName()), image.second.Marker + "\nНе удалось загрузить фото " + errors.GetStringReport());
            }
        }
        count++;
    }
    LastEventId = nextReal;
    return true;
}

TCleaningRegularReport::TCleaningRegularReport(const TCleaningRegularReportConfig* config)
    : TBase(*config)
    , Config(config)
    , LastEventId(Config->GetStartEventId())
    , LastSuccessfulTaskId(Config->GetStartIndex())
{
}

void TCleaningRegularReport::SerializeToProto(NDrive::NProto::TCleaningReport& proto) const {
    proto.SetLastEventId(LastEventId);
    proto.SetLastSuccessfulTaskId(LastSuccessfulTaskId);
}

bool TCleaningRegularReport::DeserializeFromProto(const NDrive::NProto::TCleaningReport& proto) {
    LastEventId = proto.GetLastEventId();
    LastSuccessfulTaskId = proto.GetLastSuccessfulTaskId();
    return true;
}

