#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/notifier.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags.h>

#include <rtline/util/instant_model.h>

void TRentPricingWatcher::SerializeToProto(NRentPricingProto::TState& proto) const {
    INFO_LOG << "TRentPricingWatcher::SerializeToProto" << Endl;
    proto.SetTimestampLastPriceCalculated(LastPriceCalculated.Seconds());
}

bool TRentPricingWatcher::DeserializeFromProto(const NRentPricingProto::TState& proto) {
    INFO_LOG << "TRentPricingWatcher::DeserializeFromProto" << Endl;
    if (proto.HasTimestampLastPriceCalculated()) {
        LastPriceCalculated = TInstant::Seconds(proto.GetTimestampLastPriceCalculated());
    } else {
        LastPriceCalculated = ModelingNow();
    }
    return true;
}

bool TRentPricingWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    TInstant predCallInstant = LastPriceCalculated;
    NDrive::INotifier::TPtr notifier = server->GetNotifier(Config->GetNotifierName());

    ::TSessionsBuilder<TCarTagHistoryEvent>::TPtr sessionBuilder;
    LastPriceCalculated = ModelingNow();
    for (ui32 att = 0; att < 10; ++att) {
        sessionBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilderSafe("billing", Now());
        if (!!sessionBuilder) {
            break;
        }
    }
    if (!sessionBuilder) {
        LastPriceCalculated = predCallInstant;
        const TString report = "Cannot take billing session";
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage(report));
        } else {
            ERROR_LOG << report << Endl;
        }
        return true;
    }
    TMap<TString, TVector<IEventsSession<TCarTagHistoryEvent>::TPtr>> sessions = sessionBuilder->GetSessionsActual(predCallInstant, LastCallInstant);

    TStringStream ss;
    ss << predCallInstant.ToString() << " -> " << LastPriceCalculated.ToString() << "(" << (LastPriceCalculated - predCallInstant).Seconds() << ")" << Endl;
    TMap<TString, double> bills;
    TMap<TString, double> deposits;

    TMap<TString, TVector<TString>> signalsBySession;
    for (auto&& i : sessions) {
        for (auto&& s : i.second) {
            TBillingSession::TBillingCompilation compilation;
            compilation.SetUntil(Now() - server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetConfig().GetGuaranteeFinishedTransactionsDuration());
            if (s->FillCompilation(compilation)) {
                if (!compilation.GetSessionId()) {
                    ss << i.first << ": incorrect offer id" << Endl;
                    continue;
                }
                signalsBySession.emplace(compilation.GetSessionId(), compilation.GetBillingSessionSignals(*server));
                bills[compilation.GetSessionId()] = compilation.GetBillingSumPrice();
                deposits[compilation.GetSessionId()] = compilation.GetDepositSum();
                ss << i.first << ": " << compilation.GetBillingSumPrice() << "/" << compilation.GetReportSumPrice() << Endl;
            } else {
                ss << i.first << ": Cannot compile session for user" << Endl;
            }
        }
    }
    if (bills.empty()) {
        ss << "No billing actions" << Endl;
    }

    TMap<TString, double> billDeltas;
    TMap<TString, double> depositDeltas;
    {
        auto session = server->GetDriveAPI()->GetBillingManager().BuildSession(false);
        if (!server->GetDriveAPI()->GetBillingManager().SetBillingInfo(bills, session, &billDeltas) || !session.Commit()) {
            LastPriceCalculated = predCallInstant;
            ss << session.GetStringReport() << Endl;
        }
    }

    {
        auto session = server->GetDriveAPI()->GetBillingManager().BuildSession(false);
        if (!server->GetDriveAPI()->GetBillingManager().SetDepositInfo(deposits, session, &depositDeltas) || !session.Commit()) {
            LastPriceCalculated = predCallInstant;
            ss << session.GetStringReport() << Endl;
        }
    }

    if (notifier) {
        notifier->Notify(NDrive::INotifier::TMessage(ss.Str()));
    } else {
        WARNING_LOG << ss.Str() << Endl;
    }
    TMap<TString, double> signals;
    auto itSignals = signalsBySession.begin();
    for (auto&& i : billDeltas) {
        if (!Advance(itSignals, signalsBySession.end(), i.first)) {
            continue;
        }
        for (auto&& s : itSignals->second) {
            signals[s] += i.second;
        }
    }

    for (auto&& i : signals) {
        TUnistatSignalsCache::SignalAdd("billing-delta", i.first, i.second);
    }
    return true;
}

TRentPricingWatcher::TRentPricingWatcher(const TRentPricingWatcherConfig* config)
    : TBase(*config)
    , Config(config)
{
}
