#include "processor.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive/takeout.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/library/cpp/takeout/client.h>
#include <drive/library/cpp/tracks/tio.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TTakeoutRegularConfig::TFactory::TRegistrator<TTakeoutRegularConfig> TTakeoutRegularConfig::Registrator("takeout_regular");

bool TTakeoutRegular::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasDocumentPhotosManager()) {
        ERROR_LOG << "user photo manager undefined" << Endl;
        return false;
    }
    auto& takeoutClient = server->GetDriveAPI()->GetTakeoutClient();
    auto newTakeoutRequests = server->GetDriveAPI()->GetTakeoutRequests().GetNewTakeoutRequests(Config->GetRequestsLimit());
    TVector<TString> notifierMessages;
    TSet<NUserDocument::EType> photoTypes;
    StringSplitter(server->GetSettings().GetBackgroundValueDef<TString>(GetId(), "photo_types", "")).SplitBySet(",").SkipEmpty().ParseInto(&photoTypes);
    for (auto&& takeoutRequest : newTakeoutRequests) {
        auto jobId = CGIUnescapeRet(takeoutRequest.GetJobId());
        auto userId = takeoutRequest.GetUserId();
        takeoutRequest.SetProcessingStartedAt(Now());

        if (userId == "user-absent") {
            DequeueRequest(server, takeoutRequest, jobId, {}, notifierMessages);
            continue;
        }

        bool isUploadFailed = false;
        TVector<TString> filenames;

        // Upload document photos
        auto photoDBObjects = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetAllForUser(userId);
        for (auto&& photo : photoDBObjects) {
            if (photoTypes && !photoTypes.contains(photo.GetType())) {
                continue;
            }
            TString content;
            if (!server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(photo.GetId(), content, *server)) {
                ERROR_LOG << "TAKEOUT UPLOAD FAILED: On getting photo " << photo.GetId() << " for user " << userId << Endl;
                notifierMessages.push_back("Не удалось получить фотографию " + photo.GetId() + " для пользователя " + userId);
            } else {
                auto filename = photo.BuildTakeoutFilename();
                if (!takeoutClient.SendFileToTakeout(jobId, filename, content)) {
                    isUploadFailed = true;
                    ERROR_LOG << "TAKEOUT UPLOAD FAILED: On submitting photo " << photo.GetId() << " for user " << userId << Endl;
                    notifierMessages.push_back("Не удалось отправить в Takeout фотографию " + photo.GetId() + " для пользователя " + userId);
                    break;
                }
                filenames.push_back(std::move(filename));
            }
        }
        if (isUploadFailed) {
            continue;
        }

        // Upload text data
        {
            auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            auto userPtr = userFetchResult.GetResultPtr(userId);
            CHECK_WITH_LOG(!!userPtr);
            auto userPersdataJson = userPtr->GetReport(NUserReport::ReportGDPR);

            userPersdataJson.InsertValue("passport_datasync_collection", "documents_unverified");
            userPersdataJson.InsertValue("passport_datasync_id", userPtr->GetPassportDatasyncRevision());
            userPersdataJson.InsertValue("driving_license_datasync_collection", "driving_license_unverified");
            userPersdataJson.InsertValue("driving_license_datasync_id", userPtr->GetDrivingLicenseDatasyncRevision());

            filenames.push_back("userinfo.json");
            if (!takeoutClient.SendFileToTakeout(jobId, "userinfo.json", userPersdataJson.GetStringRobust())) {
                ERROR_LOG << "TAKEOUT UPLOAD FAILED: On submitting userinfo.json for user " << userId << Endl;
                notifierMessages.push_back("Не удалось отправить в Takeout базовые данные для пользователя " + userId);
                isUploadFailed = true;
            }
        }
        if (isUploadFailed) {
            continue;
        }

        // Upload billing data
        {
            NJson::TJsonValue paymethods = NJson::JSON_ARRAY;
            auto account = server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetTrustAccount(userId, takeoutRequest.GetEnqueuedAt());
            if (account) {
                paymethods.AppendValue(account->GetUserReport());
            }

            filenames.push_back("paymethods.json");
            if (!takeoutClient.SendFileToTakeout(jobId, "paymethods.json", paymethods.GetStringRobust())) {
                ERROR_LOG << "TAKEOUT UPLOAD FAILED: On submitting paymethods.json for user " << userId << Endl;
                notifierMessages.push_back("Не удалось отправить в Takeout информацию по биллигу для пользователя " + userId);
                isUploadFailed = true;
            }
        }
        if (isUploadFailed) {
            continue;
        }

        // Upload rides data
        {
            NJson::TJsonValue rides = NJson::JSON_ARRAY;
            const auto& compiledRides = server->GetDriveAPI()->GetMinimalCompiledRides();
            auto since = takeoutRequest.GetEnqueuedAt() - TDuration::Days(90);
            auto tx = compiledRides.BuildTx<NSQL::ReadOnly>();
            auto ydbTx = server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("takeout_regular", server);
            auto optionalCompiledSessions = compiledRides.GetUser<TFullCompiledRiding>(userId, tx, ydbTx, since);
            if (!optionalCompiledSessions) {
                ERROR_LOG << "TAKEOUT UPLOAD FAILED: On getting rides list from HistoryManager" << Endl;
                notifierMessages.push_back("Не удалось получить из HistoryManager данные поездок для пользователя " + userId);
                isUploadFailed = true;
                continue;
            }
            {
                const auto& history = *optionalCompiledSessions;
                for (auto&& ridingRawInfo : history) {
                    auto locale = DefaultLocale;
                    auto report = ridingRawInfo.GetReport(locale, NDriveSession::ReportAll, *server);
                    auto deviceId = ridingRawInfo.TMinimalCompiledRiding::GetObjectId();
                    auto carFetchResult = server->GetDriveAPI()->GetCarsData()->GetCachedOrFetch(deviceId);
                    auto carPtr = carFetchResult.GetResultPtr(deviceId);
                    if (!!carPtr) {
                        NJson::TJsonValue carReport;
                        carReport["number"] = carPtr->GetNumber();
                        auto modelCode = carPtr->GetModel();
                        auto modelFetchResult = server->GetDriveAPI()->GetModelsData()->GetCached(modelCode);
                        auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
                        if (!!modelPtr) {
                            carReport["manufacturer"] = modelPtr->GetManufacturer();
                            carReport["model"] = modelPtr->GetName();
                        }
                        report.InsertValue("car", std::move(carReport));
                    }

                    rides.AppendValue(std::move(report));
                }
                filenames.push_back("rides.json");
                if (!takeoutClient.SendFileToTakeout(jobId, "rides.json", rides.GetStringRobust())) {
                    ERROR_LOG << "TAKEOUT UPLOAD FAILED: On submitting rides.json for user " << userId << Endl;
                    notifierMessages.push_back("Не удалось отправить в Takeout информацию по поездкам для пользователя " + userId);
                    isUploadFailed = true;
                }
            }
        }
        if (isUploadFailed) {
            continue;
        }

        if (Config->GetTracksAPIName()) {
            const auto tracksApi = server->GetRTLineAPI(Config->GetTracksAPIName());
            const auto& tracksClient = tracksApi->GetSearchClient();
            NRTLine::TQuery query = NDrive::MakeTrackQuery("", "", userId, "", takeoutRequest.GetEnqueuedAt() - TDuration::Days(90), takeoutRequest.GetEnqueuedAt());
            NRTLine::TSearchReply reply = tracksClient.SendQuery(query, TDuration::Seconds(10));
            if (reply.IsSucceeded()) {
                NDrive::TTrackVisitor visitor;
                TReportAccessor scanner(reply.GetReport());
                scanner.Visit(visitor);
                auto tracks = visitor.ExtractTracks();
                NJson::TJsonValue tracksJson = NJson::JSON_ARRAY;
                for (auto&& track : tracks) {
                    NJson::TJsonValue trackJson = NJson::JSON_ARRAY;
                    for (auto&& coord : track.Coordinates) {
                        NJson::TJsonValue trackCoord;
                        trackCoord.InsertValue("X", coord.X);
                        trackCoord.InsertValue("Y", coord.Y);
                        trackCoord.InsertValue("timestamp", coord.Timestamp.Seconds());
                        trackJson.AppendValue(std::move(trackCoord));
                    }
                    tracksJson.AppendValue(std::move(trackJson));
                }

                filenames.push_back("tracks.json");
                if (!takeoutClient.SendFileToTakeout(jobId, "tracks.json", tracksJson.GetStringRobust())) {
                    ERROR_LOG << "TAKEOUT UPLOAD FAILED: On submitting tracks.json for user " << userId << Endl;
                    notifierMessages.push_back("Не удалось отправить в Takeout информацию по трекам для пользователя " + userId);
                    isUploadFailed = true;
                }
            } else {
                ERROR_LOG << "TAKEOUT UPLOAD FAILED: on getting riding GPS coordinates" << Endl;
            }
        }

        if (!isUploadFailed) {
            DequeueRequest(server, takeoutRequest, jobId, filenames, notifierMessages);
        }
    }

    if (Config->GetNotifierName() && notifierMessages.size()) {
        NDrive::INotifier::MultiLinesNotify(server->GetNotifier(Config->GetNotifierName()), "Ошибки при выгрузке в Takeout", notifierMessages);
    }

    return true;
}

bool TTakeoutRegular::DequeueRequest(const NDrive::IServer* server, TTakeoutRequest& request, const TString& jobId, const TVector<TString>& filenames, TVector<TString>& errorMessages) const {
    if (!server->GetDriveAPI()->GetTakeoutClient().FinalizeReport(jobId, filenames)) {
        ERROR_LOG << "TAKEOUT UPLOAD FAILED: finalization " << request.GetUserId() << Endl;
        errorMessages.push_back("Не удалось отправить в Takeout запрос о финализации выгрузки для пользователя " + request.GetUserId());
        return false;
    }

    request.SetProcessingCompletedAt(Now());
    auto session = server->GetDriveAPI()->GetTakeoutRequests().BuildSession();
    if (!server->GetDriveAPI()->GetTakeoutRequests().Upsert(request, session) || !session.Commit()) {
        errorMessages.push_back("Не удалось записать в базу информацию о том, что выгрузка для " + request.GetUserId() + " готова");
        return false;
    }
    INFO_LOG << "TAKEOUT SUCCESSFUL: user" << request.GetUserId() << Endl;
    return true;
}

IBackgroundProcess* TTakeoutRegularConfig::Construct() const {
    return new TTakeoutRegular(this);
}
