#include "telegram.h"

#include <drive/backend/abstract/base.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/util/network/neh.h>
#include <rtline/util/network/neh_request.h>

#include <util/string/join.h>
#include <util/string/vector.h>


TActiveTelegramBotConfig::TFactory::TRegistrator<TActiveTelegramBotConfig> TActiveTelegramBotConfig::Registrator("telegram_bot");

TBotCommands ITelegramCommandsProcessor::ExtractBotCommands(const NJson::TJsonValue& message) {
    TBotCommands commands;
    const TString text = message["text"].GetStringRobust();
    const NJson::TJsonValue replyTo = message.Has("reply_to_message") ? message["reply_to_message"] : Default<NJson::TJsonValue>();

    if (message.Has("entities") && message["entities"].IsArray()) {

        const auto& entities = message["entities"].GetArray();
        for (ui32 i = 0; i < entities.size(); ++i) {
            if (!entities[i].Has("type") || entities[i]["type"] != "bot_command") {
                continue;
            }

            const NJson::TJsonValue* nextCommand = nullptr;
            for (ui32 j = i + 1; j < entities.size(); ++j) {
                if (!entities[j].Has("type") || entities[j]["type"] != "bot_command") {
                    continue;
                }
                nextCommand = &entities[j];
                break;
            }

            const ui64 offset = entities[i]["offset"].GetUIntegerRobust();
            const ui64 length = entities[i]["length"].GetUIntegerRobust();
            const TString command = text.substr(offset + 1, length - 1); // +1 to remove '/'

            const ui64 argsOffset = offset + length;
            const ui64 argsLength = nextCommand ? (*nextCommand)["offset"].GetUIntegerRobust() - argsOffset : text.npos;
            const TString args = argsOffset < text.size() ? text.substr(argsOffset, argsLength) : Default<TString>();
            const TVector<TString> argLines = SplitString(args, "\n");
            const TVector<TString> parts = argLines.size() ? SplitString(argLines[0], " ") : Default<TVector<TString>>();
            commands.push_back({ command, parts });
            commands.back().SetReplyToMessage(replyTo);
            DEBUG_LOG << "Execute: ExtractBotCommands: '" << command << "':'" << JoinSeq(",", parts) << "'" << Endl;
        }
    }

    if (commands.empty()) {
        TVector<TString> parts = SplitString(text, " ", 2);
        TVector<TString> argParts = parts.size() > 1 ? SplitString(parts[1], " ") : Default<TVector<TString>>();
        commands.push_back({ parts[0], argParts });
    }

    return commands;
}

bool ITelegramCommandsProcessor::GetDryRunMode() const {
    return Bot->GetDryRunMode();
}

bool TActiveTelegramBot::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const IServerBase* server) const {
    CHECK_WITH_LOG(!!Agent);
    NNeh::THttpRequest request;
    request.SetUri("/" + Config.GetBotId() + "/getUpdates?offset=" + ToString(CurrentUpdate));
    auto tgResult = Agent->SendMessageSync(request, Now() + TDuration::Seconds(10));
    if (tgResult.Code() != 200) {
        ERROR_LOG << tgResult.Code() << " / " << tgResult.Content() << " / " << tgResult.ErrorMessage() << Endl;
    } else {
        NJson::TJsonValue jsonInfo;
        CHECK_WITH_LOG(NJson::ReadJsonFastTree(tgResult.Content(), &jsonInfo)) << tgResult.Content() << Endl;
        NJson::TJsonValue::TArray messages;
        jsonInfo["result"].GetArray(&messages);
        for (auto&& i : messages) {
            if (CurrentUpdate >= i["update_id"].GetUInteger()) {
                continue;
            }
            CurrentPosition = i["message"]["message_id"].GetUInteger();
            CurrentUpdate = i["update_id"].GetUInteger();
            CommandsProcessor->Execute(i, server);
        }
    }
    return true;
}

TActiveTelegramBot::TActiveTelegramBot(const TActiveTelegramBotConfig& config)
    : TTelegramNotifierImpl(MakeAtomicShared<TActiveTelegramBotConfig>(config))
    , ISerializableProtoBackgroundProcess<NTelegramBotProto::TState, IBaseBackgroundRegularProcess>(config)
    , Config(config)
{
    CommandsProcessor = Config.GetProcessorConfig().Construct(this);
}

IBackgroundProcess* TActiveTelegramBotConfig::Construct() const {
    return new TActiveTelegramBot(*this);
}
