#pragma once
#include <drive/backend/background/telegram_bot/proto/telegram.pb.h>

#include <drive/backend/background/manager/regular.h>

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/notifications/telegram/telegram.h>

#include <library/cpp/yconf/conf.h>

#include <rtline/library/metasearch/simple/config.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/ptr.h>
#include <util/stream/output.h>

class ITelegramCommandsProcessor;
class TActiveTelegramBot;

class ITelegramCommandsProcessorConfig {
public:

    virtual ~ITelegramCommandsProcessorConfig() = default;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<ITelegramCommandsProcessorConfig, TString>;
    using TPtr = TAtomicSharedPtr<ITelegramCommandsProcessorConfig>;
    virtual void ToString(IOutputStream& os) const = 0;
    virtual void Init(TYandexConfig::Section* section) = 0;
    virtual ITelegramCommandsProcessor* Construct(TActiveTelegramBot* bot) const = 0;
};

class TActiveTelegramBotConfig: public TTelegramBotConfig, public IBackgroundRegularProcessConfig {
private:
    static TFactory::TRegistrator<TActiveTelegramBotConfig> Registrator;
    TString ProcessorType;
    ITelegramCommandsProcessorConfig::TPtr ProcessorConfig;
public:

    using IBackgroundRegularProcessConfig::IBackgroundRegularProcessConfig;

    virtual void Init(const TYandexConfig::Section* section) override {
        TTelegramBotConfig::Init(section);
        IBackgroundRegularProcessConfig::Init(section);
        CHECK_WITH_LOG(section->GetDirectives().GetValue("ProcessorType", ProcessorType));
        auto children = section->GetAllChildren();
        auto it = children.find("Processor");
        CHECK_WITH_LOG(it != children.end());
        ProcessorConfig = ITelegramCommandsProcessorConfig::TFactory::Construct(ProcessorType);
        CHECK_WITH_LOG(!!ProcessorConfig);
        ProcessorConfig->Init(it->second);
    }

    ITelegramCommandsProcessorConfig& GetProcessorConfig() const {
        return *ProcessorConfig;
    }

    virtual void ToString(IOutputStream& os) const override {
        TTelegramBotConfig::ToString(os);
        IBackgroundRegularProcessConfig::ToString(os);
        os << "ProcessorType: " << ProcessorType << Endl;
        ProcessorConfig->ToString(os);
    }

    virtual IBackgroundProcess* Construct() const override;
};


class TBotCommand {
    R_FIELD(TString, Command, Default<TString>());
    R_FIELD(TVector<TString>, Args, Default<TVector<TString>>());
    R_FIELD(NJson::TJsonValue, ReplyToMessage, Default<NJson::TJsonValue>());

public:
    TBotCommand()
    {
    }

    TBotCommand(const TString& command, const TVector<TString>& args)
        : Command(command)
        , Args(args)
    {
    }

    const TString& GetArg(ui32 index) const {
        if (index < Args.size()) {
            return Args[index];
        }
        return Default<TString>();
    }

    bool Is(const TString& command) const {
        return Command == command || Command.StartsWith(command + "@");
    }
};


using TBotCommands = TVector<TBotCommand>;

enum class EBotCommandStatus {
    Ok /* "ok" */,
    BadUser /* "bad_user" */,
    BadCommand /* "bad_command" */
};

class TTelegramBotCommandSignal: public TUnistatSignal<double> {
private:
    using TBase = TUnistatSignal<double>;
public:
    TTelegramBotCommandSignal(const TString& botName, EBotCommandStatus status)
        : TBase({
            "telegram_bot-" + botName + "-command-" + ToString<EBotCommandStatus>(status) + "-count"
          }, true)
    {
    }
};

class TActiveTelegramBot;
class IServerBase;
class ITelegramCommandsProcessor {
protected:
    TActiveTelegramBot* Bot = nullptr;
public:

    using TPtr = TAtomicSharedPtr<ITelegramCommandsProcessor>;

    virtual ~ITelegramCommandsProcessor() = default;

    ITelegramCommandsProcessor(TActiveTelegramBot* bot)
        : Bot(bot) {

    }

    virtual void Execute(const NJson::TJsonValue& message, const IServerBase* server) = 0;

    static TBotCommands ExtractBotCommands(const NJson::TJsonValue& message);

    bool GetDryRunMode() const;
};

class TActiveTelegramBot: public TTelegramNotifierImpl, public ISerializableProtoBackgroundProcess<NTelegramBotProto::TState, IBaseBackgroundRegularProcess> {
private:
    mutable ui32 CurrentPosition = 0;
    mutable ui32 CurrentUpdate = 0;
    const TActiveTelegramBotConfig Config;
    ITelegramCommandsProcessor::TPtr CommandsProcessor;

    virtual bool DoExecuteImpl(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const IServerBase* server) const override;

protected:
    virtual TString GetSaltForStatusKey() const override {
        return Config.GetBotId();
    }

    virtual void SerializeToProto(NTelegramBotProto::TState& proto) const override {
        proto.SetLastProcessedMessage(CurrentPosition);
        proto.SetLastUpdateMessage(CurrentUpdate);
    }

    virtual bool DeserializeFromProto(const NTelegramBotProto::TState& proto) override {
        CurrentPosition = proto.GetLastProcessedMessage();
        CurrentUpdate = proto.GetLastUpdateMessage();
        return true;
    }
public:

    bool GetDryRunMode() const {
        return Config.GetDryRunMode();
    }

    TActiveTelegramBot(const TActiveTelegramBotConfig& config);

    void Start() override {
        TTelegramNotifierImpl::Start();
    }

    void Stop() override {
        TTelegramNotifierImpl::Stop();
    }


};
