#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags.h>

#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/api/server/config.h>

#include <library/cpp/threading/future/async.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/digest/fnv.h>
#include <util/random/random.h>

#include <random>

namespace {
    template <class T>
    struct TSyncInfo {
        NDrive::TTelematicsClient::THandler Getter;
        NDrive::TTelematicsClient::THandler Setter;
        T Actual;
        T Expected;
        NDrive::TSensorId Parameter;
    };
}

TTelematicsConfigurationWatcher::TTelematicsConfigurationWatcher(const TTelematicsConfigurationWatcherConfig& config)
    : IBackgroundRegularProcessImpl<NDrive::IServer>(config)
    , Config(config)
{
}

void TTelematicsConfigurationWatcher::Start() {
}

void TTelematicsConfigurationWatcher::Stop() {
}

bool TTelematicsConfigurationWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const TDriveAPI* api = server->GetDriveAPI();
    const TString& requiredStatus = Config.GetFilterStatus();
    const TString& robotUserId = GetRobotUserId(server);
    const auto& carStatusBlacklist = Config.GetCarStatusBlacklist();
    auto objects = api->GetCarsData()->GetCached();
    auto statuses = api->GetStateFiltersDB()->GetObjectStates();

    auto shuffledIds = MakeVector(NContainer::Keys(objects.GetResult()));
    std::shuffle(shuffledIds.begin(), shuffledIds.end(), std::default_random_engine(Seconds()));

    auto start = Now();
    for (auto&& id : shuffledIds) {
        const auto p = objects.GetResultPtr(id);
        if (!p) {
            ERROR_LOG << GetId() << ": cannot get info for " << id << Endl;
            continue;
        }
        const auto& device = *p;
        const TString& imei = device.GetIMEI();
        if (!imei) {
            continue;
        }
        if (requiredStatus) {
            auto s = statuses.find(id);
            if (s == statuses.end() || s->second != requiredStatus) {
                DEBUG_LOG << GetId() << ": skip " << id << " due to non-matching status" << Endl;
                continue;
            }
        }
        auto duration = Now() - start;
        if (duration > Config.GetMaximumDuration()) {
            NOTICE_LOG << GetId() << ": breaking" << Endl;
            break;
        }
        if (!carStatusBlacklist.empty()) {
            auto s = statuses.find(id);
            if (s == statuses.end() || carStatusBlacklist.contains(s->second)) {
                DEBUG_LOG << GetId() << ": skip" << id << "due to matching statuses blacklist" << Endl;
                continue;
            }
        }

        Apply(device, server, robotUserId);
    }

    return true;
}

void TTelematicsConfigurationWatcher::Apply(const TDriveCarInfo& object, const NDrive::IServer* server, const TString& robotUserId) const {
    const TDriveAPI* api = server->GetDriveAPI();
    const IDriveTagsManager& tagsManager = api->GetTagsManager();

    const TString& id = object.GetId();
    const TString& imei = object.GetIMEI();
    try {
        const auto snapshot = server->GetSnapshotsManager().GetSnapshot(id);
        NDrive::THeartbeat heartbeat;
        if (!snapshot.GetHeartbeat(heartbeat, Config.GetPeriod())) {
            WARNING_LOG << GetId() << ": skipping offline " << id << Endl;
            return;
        }

        const NDrive::ISensorApi* sensors = server->GetSensorApi();
        if (!sensors) {
            ERROR_LOG << GetId() << ": SensorAPI is missing" << Endl;
            return;
        }

        auto firmware = sensors->GetSensor(imei, VEGA_MCU_FIRMWARE_VERSION).ExtractValueSync();
        if (!firmware) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to missing " << firmware->GetName() << Endl;
            return;
        }

        auto session = api->template BuildTx<NSQL::Writable>();

        TDBTag existing;
        {
            TVector<TDBTag> tags;
            if (!tagsManager.GetDeviceTags().RestoreEntityTags(id, { TTelematicsConfigurationTag::Type() }, tags, session)) {
                ERROR_LOG << GetId() << ": cannot restore " << TTelematicsConfigurationTag::Type() << " for " << id << ": " << session.GetStringReport() << Endl;
                return;
            }
            if (!tags.empty()) {
                existing = std::move(tags[0]);
            }
        }

        auto tag = existing.GetData();
        auto configuration = std::dynamic_pointer_cast<TTelematicsConfigurationTag>(tag);
        if (!configuration) {
            configuration = MakeAtomicShared<TTelematicsConfigurationTag>();
        }

        auto description = std::dynamic_pointer_cast<const TTelematicsConfigurationTag::TDescription>(
            tagsManager.GetTagsMeta().GetDescriptionByName(configuration->GetName())
        );

        auto taggedObject = server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(id);
        if (!taggedObject) {
            ERROR_LOG << GetId() << ": cannot get tagged object for " << id << Endl;
            return;
        }

        if (
            configuration->GetDeadline() > Now() &&
            configuration->GetConfigurationHash() == configuration->GetExpectedHash(imei, object.GetModel(), *taggedObject, description.Get())
        ) {
            INFO_LOG << GetId() << ": " << id << " is up to date" << Endl;
            return;
        }

        auto object = tagsManager.GetDeviceTags().RestoreObject(id, session);
        if (!object) {
            ERROR_LOG << GetId() << ": cannot restore " << id << ": " << session.GetStringReport() << Endl;
            return;
        }
        if (object->HasTag(Config.GetDisableTag())) {
            INFO_LOG << GetId() << ": disabled for " << id << Endl;
            return;
        }

        configuration->SetApplyOnAdd(true);
        if (Config.GetDefaultPasswordTag() && object->HasTag(Config.GetDefaultPasswordTag())) {
            configuration->SetPinOverride(Config.GetDefaultPasswordTag());
        }
        if (!tagsManager.GetDeviceTags().AddTag(configuration, robotUserId, id, server, session)) {
            ERROR_LOG << GetId() << ": cannot add tag to " << id << ": " << session.GetStringReport() << Endl;
            return;
        }
        if (!session.Commit()) {
            ERROR_LOG << GetId() << ": cannot commit transaction for " << id << ": " << session.GetStringReport() << Endl;
            return;
        }
        INFO_LOG << GetId() << ": " << id << " scheduled to configure" << Endl;
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": an exception occurred for " << id << " " << FormatExc(e) << Endl;
    }
}

void TTelematicsFirmwareWatcher::TUpdater::Init(const TVector<TTaggedDevice> &devices) {
    const TDriveAPI* api = Server.GetDriveAPI();
    if (!api) {
        return;
    }

    for (auto&& device : devices) try {
        TString id = device.GetId();
        auto tag= device.GetTag(TTelematicsFirmwareTag::Type());
        if (!tag) {
            continue;
        }

        auto firmwareInfo = GetCurrent(*tag);
        if (!firmwareInfo.FullName) {
            ERROR_LOG << GetId() << ": cannot get current firmware info for " << device.GetId() << Endl;
            continue;
        }

        const auto deviceInfos = Yensured(api->GetCarsData())->FetchInfo({ id }, TInstant::Zero());
        auto deviceInfo = deviceInfos.GetResultPtr(id);

        auto firmwares = GetFirmwares(device, deviceInfo, firmwareInfo);
        for (auto&& firmware : firmwares) {
            auto& updatedInfo= UpdatedInfo[firmware];
            updatedInfo.Total++;
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": unexpected error in current cache constructed " << FormatExc(e) << Endl;
    }
}

NDrive::NVega::TFirmwareInfo TTelematicsFirmwareWatcher::TUpdater::GetCurrent(const TDBTag& tag) const {
    auto info = TTelematicsFirmwareTag::GetOnlyTaggedInfo(tag, Server);
    if (info && info->FullName) {
        return *info;
    }

    auto sensor = TTelematicsFirmwareTag::GetCurrentInfo(tag, Server);
    if (sensor) {
        if (!UpdateTagInfo(tag, *sensor, "copy")) {
            ERROR_LOG << GetId() << ": cannot update firmware info in tag" << Endl;
        }
        return *sensor;
    }

    return *info;
}


bool TTelematicsFirmwareWatcher::TUpdater::UpdateTagInfo(
    const TDBTag& tag,
    const NDrive::NVega::TFirmwareInfo& info,
    const TString& handler,
    const TInstant lastUpdate
) const {
    auto api = Server.GetDriveAPI();

    if (!api) {
        ERROR_LOG << GetId() << ": DriveAPI is missing" << Endl;
        return false;
    }

    const auto& deviceTagsManager = api->GetTagsManager().GetDeviceTags();

    auto copy = tag.Clone(api->GetTagsHistoryContext());
    if (!copy) {
        return false;
    }
    auto copiedFirmwareTag = copy.MutableTagAs<TTelematicsFirmwareTag>();
    if (!copiedFirmwareTag) {
        ERROR_LOG << GetId() << ": cannot cast cloned tag as TelematicsFirmwareTag for " << tag.GetObjectId() << Endl;
        return false;
    }
    copiedFirmwareTag->SetFirmware(info, handler);
    if (lastUpdate != TInstant::Zero()) {
        copiedFirmwareTag->SetLastUpdate(lastUpdate);
    }

    auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
    if (!deviceTagsManager.UpdateTagData(copy, RobotUserId, session)) {
        ERROR_LOG << GetId() << ": " << Id << " cannot update tag data " << session.GetStringReport() << Endl;
        return false;
    }
    if (!session.Commit()) {
        ERROR_LOG << GetId() << ": " << Id << " cannot commit transaction " << session.GetStringReport() << Endl;
        return false;
    }

    return true;
}

TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo> TTelematicsFirmwareWatcher::TUpdater::GetFirmwares(
    const TTaggedObject& object,
    const TDriveCarInfo* deviceInfo,
    const NDrive::NVega::TFirmwareInfo& firmwareInfo
) const {
    TVector<TTelematicsFirmwareTag::TDescription::TFirmwareInfo> result;

    auto firmwares = TTelematicsFirmwareTag::GetInfoFromDescription(Server);
    if (!firmwares) {
        return result;
    }

    auto comparator = [&object, &firmwareInfo, deviceInfo] (const TTelematicsFirmwareTag::TDescription::TFirmwareInfo& firmware) {
        auto info = firmwareInfo;
        if (deviceInfo && firmware.IsIgnoreFirmwareModel()) {
            const auto& currentInfo = firmware.GetInfo();

            info.Model = deviceInfo->GetModel();
            if (firmwareInfo.Type == NDrive::NVega::TFirmwareInfo::MT32K_MTX) {
                info.Type = currentInfo.SubType;
            }
        }

        return firmware.IsCompatibleWith(object, info);
    };
    CopyIf(firmwares->begin(), firmwares->end(), std::back_inserter(result), comparator);

    return result;
}

NDrive::NVega::TFirmwareInfo TTelematicsFirmwareWatcher::TUpdater::GetDesired(
    const TTaggedDevice& object,
    const TDriveCarInfo* deviceInfo,
    const NDrive::NVega::TFirmwareInfo& firmwareInfo
) const {
    NDrive::NVega::TFirmwareInfo result;

    auto firmwares = GetFirmwares(object, deviceInfo, firmwareInfo);

    if (firmwares.empty()) {
        ERROR_LOG << GetId() << ": try get desired firmware by empty firmware names" << Endl;
        return result;
    }

    if (firmwares.size() == 1) {
        const auto& firmware = firmwares.front();
        auto& updatedInfo = UpdatedInfo[firmware];

        result = firmware.GetInfo();
        updatedInfo.Updated++;
    } else {
        bool founded = false;

        for (auto&& firmware : firmwares) {
            auto& updatedInfo = UpdatedInfo[firmware];
            result = firmware.GetInfo();

            ui64 total = updatedInfo.Total;
            if (!total) {
                ERROR_LOG << GetId() << ": empty total for " << firmware.DebugString() << Endl;
                continue;
            }

            ui64 updated = updatedInfo.Updated;
            ui8 current = (updated * 100) / total;
            ui8 desired = result.Percent;

            INFO_LOG << GetId() << ": smart update info " << object.GetId() << " "
                     << result.FullName << " " << ToString(total) << " "
                     << ToString(updated) << " " << ToString(current) << " "
                     << ToString(desired) << Endl;

            if (current >= desired) {
                continue;
            }

            founded = true;
            updatedInfo.Updated++;

            NDrive::TEventLog::Log("SmartUpdateInfo", NJson::TMapBuilder
                ("car_id", object.GetId())
                ("model", result.Model)
                ("type", ToString(result.Type))
                ("total", total)
                ("updated", updated)
                ("current_percent", current)
                ("desired_percent", desired)
                ("filter", firmware.GetTagsFilter().ToString())
            );
            break;
        }

        if (!founded) {
            const auto& firmware = firmwares.back();
            result = firmware.GetInfo();
        }
    }

    return result;
}

TTelematicsFirmwareWatcher::TTelematicsFirmwareWatcher(const TTelematicsFirmwareWatcherConfig& config)
    : IBackgroundRegularProcessImpl<NDrive::IServer>(config)
    , Config(config)
{
}

bool TTelematicsFirmwareWatcher::DoExecuteImpl(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    if (!server) {
        return false;
    }

    const TDriveAPI* api = server->GetDriveAPI();
    if (!api) {
        ERROR_LOG << GetId() << ": DriveAPI is missing" << Endl;
        return false;
    }
    const NDrive::ISensorApi* sensors = server->GetSensorApi();
    if (!sensors) {
        ERROR_LOG << GetId() << ": SensorAPI is missing" << Endl;
        return false;
    }
    const TDeviceTagsManager& deviceTagsManager = api->GetTagsManager().GetDeviceTags();
    const TString& robotUserId = GetRobotUserId(server);

    TVector<TTaggedDevice> devices;
    if (!deviceTagsManager.GetObjectsFromCache({ TTelematicsFirmwareTag::Type() }, devices, TInstant::Zero())) {
        ERROR_LOG << GetId() << ": cannot Get " << TTelematicsFirmwareTag::Type() << " tagged ObjectsFromCache";
    }

    auto comparator = [](const TTaggedDevice& lhs, const TTaggedDevice& rhs) {
        return lhs.GetId() < rhs.GetId();
    };
    Sort(devices.begin(), devices.end(), comparator);

    TUpdater updater(GetId(), *server, robotUserId);
    updater.Init(devices);

    for (auto&& device : devices) try {
        const TString& id = device.GetId();
        const TString& imei = api->GetIMEI(id);

        auto tag = device.GetTag(TTelematicsFirmwareTag::Type());
        if (!tag) {
            ERROR_LOG << GetId() << ": " << TTelematicsFirmwareTag::Type() << " tag is missing for " << id << Endl;
            continue;
        }

        const auto deviceInfos = Yensured(api->GetCarsData())->FetchInfo({ id }, TInstant::Zero());
        auto deviceInfo = deviceInfos.GetResultPtr(id);

        auto current = updater.GetCurrent(*tag);
        if (!current.FullName) {
            ERROR_LOG << GetId() << ": cannot get current info from " << id << Endl;
            continue;
        }

        auto desired = updater.GetDesired(device, deviceInfo, current);

        if (!imei) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to missing IMEI" << Endl;
            continue;
        }

        auto firmwareUpdateInfo = TTelematicsFirmwareTag::GetUpdateInfo(*tag, *server);
        if (!firmwareUpdateInfo) {
            ERROR_LOG << GetId() << ": cannot find firmware info for " << id << ": " << firmwareUpdateInfo.GetError().what() << Endl;
            continue;
        }

        auto& [overloadName, ignoreCarStatus, maybeUpdateTimeRestrictions, lastUpdate] = *firmwareUpdateInfo;

        if (overloadName) {
            desired = NDrive::NVega::ParseFirmwareInfo(overloadName);
        }

        if (!desired.FullName) {
            ERROR_LOG << GetId() << ": firmware name is missing from " << id << " " << current.FullName << " " << current.Model << " " << ToString(current.Type) << Endl;
            continue;
        }

        INFO_LOG << GetId() << ": desired firmware for " << id << " " << desired.FullName << Endl;

        auto statuses = api->GetStateFiltersDB()->GetObjectStates();
        auto p = statuses.find(id);
        auto status = p != statuses.end() ? p->second : TString();
        if (!Config.GetRequiredStatuses().contains(status) && !ignoreCarStatus) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to status " << status << Endl;
            continue;
        }
        auto deadline = lastUpdate.Seconds() + Config.GetReupdateDelay().Seconds();
        if (Now().Seconds() < deadline) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to ReupdateDelay now - " <<  Now().Seconds() << " need - " << deadline << Endl;
            continue;
        }
        if (maybeUpdateTimeRestrictions && !maybeUpdateTimeRestrictions->IsActualNow(Now())) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to schedule restrictions" << Endl;
            continue;
        }
        const auto snapshot = server->GetSnapshotsManager().GetSnapshot(id);
        auto engineOn = snapshot.GetSensor(CAN_ENGINE_IS_ON);
        if (!engineOn) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to missing CAN_ENGINE_IS_ON" << Endl;
            continue;
        }
        if (engineOn && engineOn->ConvertTo<bool>()) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to sensor " << engineOn->GetName() << Endl;
            continue;
        }
        auto speed = snapshot.GetSensor(VEGA_SPEED, Config.GetRequiredSensorFreshness());
        if (!speed) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to missing or old VEGA_SPEED" << Endl;
            continue;
        }
        if (speed && speed->ConvertTo<double>() > 1) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to sensor " << speed->GetName() << Endl;
            continue;
        }

        auto firmware = sensors->GetSensor(imei, VEGA_MCU_FIRMWARE_VERSION).ExtractValueSync();
        if (!firmware) {
            NOTICE_LOG << GetId() << ": skip " << id << " due to missing " << firmware->GetName() << Endl;
            continue;
        }

        if (desired.FullName.Contains(current.FullName)) {
            INFO_LOG << GetId() << ": " << id << " firmware is up to date " << current.FullName << Endl;
            continue;
        }

        INFO_LOG << GetId() << ": try update firmware " << id << " " << desired.GetName() << " from " << current.GetName() << Endl;

        auto data = TTelematicsFirmwareTag::GetData(desired.GetName(), *server);
        if (!data) {
            ERROR_LOG << GetId() << ": cannot get new firmware data " << data.GetError().what() << Endl;
            continue;
        }

        auto handler = server->GetTelematicsClient().Upload(
            imei,
            "FIRMWARE",
            data.ExtractValue(),
            Config.GetUpdateTimeout()
        );

        NDrive::TEventLog::Log("UploadFirmware", NJson::TMapBuilder
            ("object_id", id)
            ("imei", imei)
            ("status", status)
            ("handler_id", handler.GetId())
        );

        if (!updater.UpdateTagInfo(*tag, desired, handler.GetId(), Now())) {
            ERROR_LOG << GetId() << ": cannot update firmware info in tag" << Endl;
            continue;
        }

        INFO_LOG << GetId() << ": firmware update for " << id << ": " << handler.GetId() << Endl;
    } catch (const std::exception& e) {
        ERROR_LOG << GetId() << ": an exception occurred for " << device.GetId() << ": " << FormatExc(e) << Endl;
    }

    return true;
}
