#include "processor.h"

#include "config.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

TTransformationWatcher::TTransformationWatcher(const TTransformationWatcherConfig& config)
    : IBackgroundRegularProcessImpl<NDrive::IServer>(config)
    , Config(config)
{
    Y_UNUSED(Config);
}

bool TTransformationWatcher::RestoreTransformations(TBackgroundProcessesManager* /*manager*/, IBackgroundProcess::TPtr /*self*/, const NDrive::IServer* server) const {
    const TDriveAPI* api = server->GetDriveAPI();
    const IDriveTagsManager& tagsManager = api->GetTagsManager();
    const TDeviceTagsManager& deviceTagManager = tagsManager.GetDeviceTags();
    const TString& robotUserId = GetRobotUserId(server);

    auto permissions = api->GetUserPermissions(robotUserId);
    if (!permissions) {
        ERROR_LOG << GetId() << ": cannot GetUserPermissions for " << robotUserId << Endl;
        return true;
    }

    TVector<TDBTag> tags;
    {
        auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
        if (!deviceTagManager.RestoreTags({}, {TTransformationTag::TypeName}, tags, tx)) {
            ERROR_LOG << "TransformationWatcher: cannot restore transformation tags " << tx.GetStringReport() << Endl;
            return true;
        }
    }

    TVector<TDBTag> tagsDrop;
    TVector<ITag::TPtr> predTag;
    auto now = Now();
    {
        bool fresh = false;
        for (ui32 att = 0; att < 10; ++att) {
            if (tagsManager.GetDeviceTags().GetHistoryManager().Update(Now())) {
                fresh = true;
                break;
            }
        }
        if (!fresh) {
            ERROR_LOG << "TransformationWatcher: cannot update history" << Endl;
            return false;
        }
    }

    ui32 fails = 0;

    auto tx = deviceTagManager.BuildTx<NSQL::Writable>();
    for (auto&& tag : tags) {
        auto transformation = dynamic_cast<const TTransformationTag*>(tag.GetData().Get());
        if (!transformation) {
            ERROR_LOG << "TransformationWatcher: cannot cast " << tag.GetTagId() << " to TransformationTag" << Endl;
            continue;
        }

        if (transformation->GetRollbackTimestamp() > now) {
            INFO_LOG << "TransformationWatcher: rollback timestamp not exceeded for " << tag.GetTagId() << Endl;
            continue;
        }
        auto history = tagsManager.GetDeviceTags().GetEventsByTag(tag.GetTagId(), tx);
        if (!history) {
            ERROR_LOG << "TransformationWatcher: cannot acquire history for " << tag.GetTagId() << ": " << tx.GetStringReport() << Endl;
            ++fails;
            continue;
        }
        if (history->size() < 2) {
            ERROR_LOG << "TransformationWatcher: insufficient history for " << tag.GetTagId() << Endl;
            ++fails;
            continue;
        }

        const TCarTagHistoryEvent& current = history->back();
        if (current->GetName() != tag->GetName()) {
            ERROR_LOG << "TransformationWatcher: inconsistent current tag name " << current->GetName() << Endl;
            ++fails;
            continue;
        }

        const TCarTagHistoryEvent* previous = nullptr;
        for (i32 i = history->size() - 2; i >= 0; --i) {
            if (history->at(i)->GetName() != TTransformationTag::TypeName) {
                previous = &history->at(i);
                break;
            }
        }
        if (!previous || (*previous)->GetName() != transformation->GetFrom()) {
            ERROR_LOG << "TransformationWatcher: inconsistent previous tag name " << (*previous)->GetName() << " " << transformation->GetFrom() << Endl;
            ++fails;
            continue;
        }

        TDBTag temporary = previous->Clone(server->GetDriveAPI()->GetTagsHistoryContext());
        if (!!temporary) {
            tagsDrop.emplace_back(tag);
            predTag.emplace_back(temporary.GetData());
        }
    }
    {
        for (ui32 i = 0; i < tagsDrop.size(); ++i) {
            auto transformed = TTransformationTag::Rollback(tagsDrop[i], predTag[i], *permissions, server, tx, nullptr);
            if (!transformed) {
                ERROR_LOG << GetId() << ": cannot Rollback " << tagsDrop[i].GetTagId() << ": " << tx.GetStringReport() << Endl;
                return true;
            }
            INFO_LOG << "TransformationWatcher: processed " << tagsDrop[i].GetTagId() << Endl;
        }
        if (!tx.Commit()) {
            ERROR_LOG << "TransformationWatcher: cannot evolve " << tx.GetStringReport() << Endl;
            return false;
        }
    }
    return (fails == 0);
}

bool TTransformationWatcher::DoExecuteImpl(TBackgroundProcessesManager* manager, IBackgroundProcess::TPtr self, const NDrive::IServer* server) const {
    INFO_LOG << "TransformationWatcher: wake up" << Endl;
    if (!RestoreTransformations(manager, self, server)) {
        ERROR_LOG << "TransformationWatcher: failed" << Endl;
        return true;
    }
    INFO_LOG << "TransformationWatcher: sleep" << Endl;
    return true;
}
