#include "config.h"

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <util/stream/file.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/env.h>

void TTvmConfig::Init(const TYandexConfig::Section* section) {
    CHECK_WITH_LOG(section);
    const auto& directives = section->GetDirectives();
    Cache = directives.Value("Cache", Cache);
    Secret = directives.Value("Secret", Secret);
    SecretFile = directives.Value("SecretFile", SecretFile);
    SelfClientId = directives.Value("SelfClientId", SelfClientId);
    StringSplitter(
        directives.Value("DestinationClientIds", JoinRange(",", DestinationClientIds.begin(), DestinationClientIds.end()))
    ).SplitBySet(" ,").SkipEmpty().ParseInto(&DestinationClientIds);
    StringSplitter(
        directives.Value("UserTicketEnvironments", JoinRange(",", UserTicketEnvironments.begin(), UserTicketEnvironments.end()))
    ).SplitBySet(" ,").SkipEmpty().ParseInto(&UserTicketEnvironments);

    if (!Secret && SecretFile) {
        Secret = TIFStream(SecretFile).ReadAll();
    }
}

void TTvmConfig::ToString(IOutputStream& os) const {
    os << "Cache: " << Cache << Endl;
    os << "SelfClientId: " << SelfClientId << Endl;
    os << "DestinationClientIds: " << JoinRange(",", DestinationClientIds.begin(), DestinationClientIds.end()) << Endl;
    os << "UserTicketEnvironments: " << JoinRange(",", UserTicketEnvironments.begin(), UserTicketEnvironments.end()) << Endl;
    if (SecretFile) {
        os << "SecretFile: " << SecretFile << Endl;
    } else {
        os << "Secret: " << Secret << Endl;
    }
}

void TMapsLinkerConfig::Init(const TYandexConfig::Section* section) {
    CHECK_WITH_LOG(section);
    const auto& directives = section->GetDirectives();
    Endpoint = directives.Value("Endpoint", Endpoint);
    RelinkerName = directives.Value("RelinkerName", RelinkerName);
}

void TMapsLinkerConfig::ToString(IOutputStream& os) const {
    os << "Endpoint: " << Endpoint << Endl;
    os << "RelinkerName: " << RelinkerName << Endl;
}

ui32 TMapsRouterConfig::GetSelfClientId() const {
    return SelfClientId;
}

ui32 TMapsRouterConfig::GetDestinationTvmId() const {
    return DestinationTvmId;
}

const TString& TMapsRouterConfig::GetHostUrl() const {
    return HostUrl;
}

const TDuration& TMapsRouterConfig::GetRequestTimeout() const {
    return RequestTimeout;
}

const TString& TMapsRouterConfig::GetVehicleType() const {
    return VehicleType;
}

void TMapsRouterConfig::Init(const TYandexConfig::Section* section) {
    CHECK_WITH_LOG(section);
    const auto& directives = section->GetDirectives();
    SelfClientId = directives.Value("SelfClientId", SelfClientId);
    DestinationTvmId = directives.Value("DestinationTvmId", DestinationTvmId);
    HostUrl = directives.Value("HostUrl", HostUrl);
    RequestTimeout = directives.Value("RequestTimeout", RequestTimeout);
    VehicleType = directives.Value("VehicleType", VehicleType);
}

void TMapsRouterConfig::ToString(IOutputStream& os) const {
    os << "SelfClientId: " << SelfClientId << Endl;
    os << "DestinationTvmId: " << DestinationTvmId << Endl;
    os << "HostUrl: " << HostUrl << Endl;
    os << "RequestTimeout: " << RequestTimeout << Endl;
    os << "VehicleType: " << VehicleType << Endl;
}

namespace {
    ui64 CompareSections(const TYandexConfig::Section& original, const TYandexConfig::Section& reparsed) {
        auto originalName = TStringBuf(original.Name);
        auto reparsedName = TStringBuf(reparsed.Name);
        Y_ENSURE(originalName == reparsedName, originalName << " == " << reparsedName);

        ui64 result = 0;
        for (auto&& [name, value] : original.GetDirectives()) {
            if (!reparsed.GetDirectives().contains(name)) {
                WARNING_LOG << originalName << ": directive " << name << " is missing" << Endl;
                result += 1;
            }
        }

        auto originalSections = original.GetAllChildren();
        auto reparsedSections = reparsed.GetAllChildren();
        for (auto&& [name, section] : originalSections) {
            auto reparsedSection = reparsedSections.find(name);
            if (reparsedSection == reparsedSections.end()) {
                WARNING_LOG << originalName << ": section " << name << " is missing" << Endl;
                result += 1;
                continue;
            }
            if (originalSections.count(name) > 1 || reparsedSections.count(name) > 1) {
                INFO_LOG << originalName << ": section " << name << " is skipped" << Endl;
                continue;
            }
            Y_ENSURE(section, originalName << ": original section " << name << " is null");
            Y_ENSURE(reparsedSection->second, originalName << ": reparsed section " << name << " is null");
            result += CompareSections(*section, *reparsedSection->second);
        }
        return result;
    }
}

void TServerBaseConfig::CheckConfig() const {
    TAnyYandexConfig reparsed;
    auto serialized = ToString();
    Y_ENSURE(reparsed.ParseMemory(serialized));

    auto originalRoot = RawConfig.GetRootSection();
    Y_ENSURE(originalRoot);
    auto reparsedRoot = reparsed.GetRootSection();
    Y_ENSURE(reparsedRoot);
    auto diff = CompareSections(*originalRoot, *reparsedRoot);
    auto limit = std::numeric_limits<ui64>::max();
    TryFromString(GetEnv("CONFIG_DIFFERENCES_LIMIT"), limit);
    Y_ENSURE(diff <= limit, diff << " differences exceeds limit " << limit);
}

void TServerBaseConfig::Init(const TYandexConfig::Section* section) {
    try {
        const TYandexConfig::Directives& directives = section->GetDirectives();
        const TYandexConfig::TSectionsMap serverSections = section->GetAllChildren();

        EventLog = directives.Value("EventLog", EventLog);
        UnifiedAgentEventLog = directives.Value("UnifiedAgentEventLog", UnifiedAgentEventLog);
        UnifiedAgentGrpcMaxMessageSize = directives.Value("UnifiedAgentGrpcMaxMessageSize", UnifiedAgentGrpcMaxMessageSize);
        UnifiedAgentGrpcReconnectDelay = directives.Value("UnifiedAgentGrpcReconnectDelay", UnifiedAgentGrpcReconnectDelay);
        UnifiedAgentGrpcSendDelay = directives.Value("UnifiedAgentGrpcSendDelay", UnifiedAgentGrpcSendDelay);
        UnifiedAgentMaxInflightBytes = directives.Value("UnifiedAgentMaxInflightBytes", UnifiedAgentMaxInflightBytes);
        ItsWatchPath = directives.Value("ItsWatchPath", ItsWatchPath);

        {
            auto itADConfig = serverSections.find("GlobalAsyncDelivery");
            if (itADConfig != serverSections.end()) {
                ADGlobalConfig.Reset(new TAsyncDeliveryResources::TConfig);
                ADGlobalConfig->Init(itADConfig->second);
                if (GetADGlobalConfig()) {
                    TAsyncDeliveryResources::InitGlobalResources(*GetADGlobalConfig());
                }
            }
        }

        {
            auto it = serverSections.find("RequestPolicy");
            if (it != serverSections.end()) {
                const TYandexConfig::TSectionsMap requestsPolicySection = it->second->GetAllChildren();
                for (auto&& i : requestsPolicySection) {
                    NSimpleMeta::TConfig config;
                    config.InitFromSection(i.second);
                    AssertCorrectConfig(!RequestPolicy.contains(i.first), "Policy duplication: %s", i.first.data());
                    RequestPolicy.emplace(i.first, std::move(config));
                }
            }
        }

        {
            auto it = serverSections.find("Notifications");
            if (it != serverSections.end()) {
                const TYandexConfig::TSectionsMap sections = it->second->GetAllChildren();
                for (auto&& i : sections) {
                    TString typeName;
                    if (!i.second->GetDirectives().GetValue("NotificationType", typeName)) {
                        typeName = i.first;
                    }

                    NDrive::INotifierConfig::TPtr generator = NDrive::INotifierConfig::TFactory::Construct(typeName);
                    AssertCorrectConfig(!!generator, "Incorrect typename for notification: " + typeName);

                    generator->SetTypeName(typeName);
                    generator->SetName(i.first);
                    generator->Init(i.second);

                    Notifications.emplace(i.first, generator);
                }
            }
        }

        {
            auto itRTLineAPIs = serverSections.find("RTLineAPIs");
            if (itRTLineAPIs != serverSections.end()) {
                const TYandexConfig::TSectionsMap children = itRTLineAPIs->second->GetAllChildren();
                for (auto&& i : children) {
                    TRTLineAPIConfig config;
                    config.Init(i.second, &RequestPolicy);
                    RTLineAPIConfigs.emplace(i.first, std::move(config));
                }
            }
        }

        {
            auto itHandlers = serverSections.find("Localization");
            if (itHandlers != serverSections.end()) {
                Localization.Reset(new TLocalizationConfig);
                Localization->Init(itHandlers->second);
            }
        }

        {
            auto itHandlers = serverSections.find("Settings");
            if (itHandlers != serverSections.end()) {
                Settings.Reset(new TSettingsConfig);
                Settings->Init(itHandlers->second);
            }
        }

        {
            auto itHandlers = serverSections.find("Databases");
            if (itHandlers != serverSections.end()) {
                const TYandexConfig::TSectionsMap dBases = itHandlers->second->GetAllChildren();
                {
                    for (auto&& i : dBases) {
                        NRTLine::TStorageOptions options;
                        AssertCorrectConfig(options.Init(i.second), "Incorrect section '%s' in Databases", i.first.data());
                        DatabaseConfigs.emplace(i.first, options);
                    }
                }
            }
        }

        {
            auto itHandlers = serverSections.find("ExternalDatabases");
            if (itHandlers != serverSections.end()) {
                const TYandexConfig::TSectionsMap dBases = itHandlers->second->GetAllChildren();
                for (auto&& i : dBases) {
                    NStorage::TDatabaseConfig dbConfig;
                    dbConfig.Init(i.second);
                    Databases.emplace(i.first, dbConfig);
                }
            }
        }

        {
            auto itProcessors = serverSections.find("Processors");
            AssertCorrectConfig(itProcessors != serverSections.end(), "No 'Processors' section in config");
            ProcessorDefaultTimeout = itProcessors->second->GetDirectives().Value("ProcessorDefaultTimeout", ProcessorDefaultTimeout);
            const TYandexConfig::TSectionsMap children = itProcessors->second->GetAllChildren();

            TMap<TString, const TYandexConfig::Section*> defaultSections;
            for (auto&& i : children) {
                if (i.first.StartsWith("default_config:")) {
                    defaultSections[i.first.substr(TString("default_config:").size())] = i.second;
                }
            }

            for (auto&& i : children) {
                if (i.first.StartsWith("default_config:")) {
                    continue;
                }
                const TYandexConfig::Section* defaultSection = nullptr;
                if (defaultSections.contains("default")) {
                    defaultSection = defaultSections["default"];
                }
                TString sectionName = i.first;
                size_t separatorPos = i.first.find(":");
                AssertCorrectConfig(separatorPos != 0, "Incorrect default config usage format");
                if (separatorPos != TString::npos) {
                    sectionName = sectionName.substr(separatorPos);
                    const TString defSectionName = sectionName.substr(0, separatorPos - 1);
                    auto defaultSectionIt = defaultSections.find(defSectionName);
                    defaultSection = defaultSectionIt->second;
                }
                TString typeName;
                TString aliasFor;
                const TYandexConfig::Section* section;
                if (i.second->GetDirectives().GetValue("AliasFor", aliasFor)) {
                    typeName = aliasFor;
                    auto&& itAlias = children.find(typeName);
                    AssertCorrectConfig(itAlias != children.end(), "Unknown processor alias '%s'", typeName.data());
                    section = itAlias->second;
                } else {
                    section = i.second;
                }
                TVersionedKey versionedKey(sectionName);

                section->GetDirectives().GetValue("ProcessorType", typeName);
                if (typeName.empty()) {
                    typeName = sectionName;
                }
                TVersionedKey processorKey(typeName);

                THolder<IRequestProcessorConfig> generator(IRequestProcessorConfig::TFactory::Construct(processorKey.GetName(), sectionName));
                AssertCorrectConfig(!!generator, "Incorrect processor type: '%s' in section '%s'", processorKey.GetName().data(), sectionName.data());
                if (defaultSection) {
                    generator->ReadDefaults(defaultSection);
                }
                generator->Init(section);
                if (!!aliasFor) {
                    generator->SetAliasFor(aliasFor);
                }
                ProcessorsInfo.emplace(versionedKey, std::move(generator));
            }
        }

        {
            auto itProcessors = serverSections.find("AuthModules");
            if (itProcessors != serverSections.end()) {
                const TYandexConfig::TSectionsMap children = itProcessors->second->GetAllChildren();
                for (auto&& i : children) {
                    TString authType;
                    AssertCorrectConfig(i.second->GetDirectives().GetValue("type", authType), "No auth type for %s", i.first.data());
                    THolder<IAuthModuleConfig> generator(IAuthModuleConfig::TFactory::Construct(authType, authType));
                    AssertCorrectConfig(!!generator, "Incorrect auth module type: '%s'", i.first.data());
                    generator->Init(i.second);
                    AuthModules.emplace(i.first, std::move(generator));
                }
            }
        }

        {
            auto it = serverSections.find("BackgroundProcesses");
            if (it != serverSections.end()) {
                BackgroundProcessesConfig.Reset(new TBackgroundProcessesManagerConfig);
                BackgroundProcessesConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("NotifiersManager");
            if (it != serverSections.end()) {
                NotifiersManagerConfig.Reset(new TNotifiersManagerConfig);
                NotifiersManagerConfig->Init(it->second);
            }
        }

        {
            auto it = serverSections.find("RTBackgroundManager");
            if (it != serverSections.end()) {
                RTBackgroundManagerConfig.Reset(new TRTBackgroundManagerConfig);
                RTBackgroundManagerConfig->Init(it->second);
            }
        }

        {
            auto itHttpServer = serverSections.find("HttpServer");
            AssertCorrectConfig(itHttpServer != serverSections.end(), "No 'HttpServer' section in config");
            HttpServerOptions.Init(itHttpServer->second->GetDirectives());
        }

        {
            auto itHttpStatuses = serverSections.find("HttpStatuses");
            if (itHttpStatuses != serverSections.end()) {
                HttpStatusManagerConfig.Init(itHttpStatuses->second);
            }
        }

        {
            auto itHandlers = serverSections.find("RequestHandlers");
            AssertCorrectConfig(itHandlers != serverSections.end(), "No 'RequestHandlers' section in config");

            const TYandexConfig::TSectionsMap handlers = itHandlers->second->GetAllChildren();
            {
                bool hasDefaultHandler = false;
                for (auto&& i : handlers) {
                    if (i.first == "default") {
                        hasDefaultHandler = true;
                    }
                    RequestHandlers[i.first].Init(i.second);
                }
                AssertCorrectConfig(hasDefaultHandler, "No 'default' section in 'RequestHandlers'");
            }
        }

        {
            auto itTaskExecutor = serverSections.find("GlobalTaskExecutor");
            if (itTaskExecutor != serverSections.end()) {
                TaskExecutorConfig.Reset(new TTaskExecutorConfig);
                TaskExecutorConfig->Init(itTaskExecutor->second);
            }
        }

        {
            auto itModulesConfig = serverSections.find("ModulesConfig");
            if (itModulesConfig != serverSections.end()) {
                ModulesConfig.Init(*this, serverSections);
                for (auto&& module : itModulesConfig->second->GetAllChildren()) {
                    UsingModules.insert(module.first);
                }
            }

            for (const auto& itAuthConfig : AuthModules) {
                const TSet<TString>& modules = itAuthConfig.second->GetRequiredModules();
                for (auto& module : modules) {
                    AssertCorrectConfig(UsingModules.find(module) != UsingModules.end(), "Daemon module '%s' not found, required for auth module '%s'", module.data(), itAuthConfig.first.data());
                }
            }
        }

        {
            for (auto&& [name, section] : MakeIteratorRange(serverSections.equal_range("Tvm"))) {
                TTvmConfig config;
                config.Init(section);
                const ui32 selfClientId = config.GetSelfClientId();
                AssertCorrectConfig(selfClientId != 0, "incorrect Tvm section: SelfClientId is zero or missing");
                AssertCorrectConfig(TvmConfigs.emplace(selfClientId, std::move(config)).second, "incorrect Tvm section: duplicate SelfClientId %d", selfClientId);
            }
        }

        {
            auto p = serverSections.find("Datasync");
            if (p != serverSections.end()) {
                DatasyncConfig = MakeHolder<TDatasyncClientConfig>();
                DatasyncConfig->Init(p->second);
            }
            ui32 selfClientId = DatasyncConfig ? DatasyncConfig->GetSelfClientId() : 0;
            ui32 destinationClientId = DatasyncConfig ? DatasyncConfig->GetDestinationClientId() : 0;
            if (selfClientId) {
                AssertCorrectConfig(TvmConfigs.contains(selfClientId), "incorrect Datasync section: Tvm for SelfClientId %d is not found", selfClientId);
                AssertCorrectConfig(
                    TvmConfigs.at(selfClientId).GetDestinationClientIds().contains(destinationClientId),
                    "incorrect Datasync section: Tvm for SelfClientId %d does not contain DestinationClientId %d", selfClientId, destinationClientId
                );
            }
        }

        {
            auto it = serverSections.find("Ciphers");
            if (it != serverSections.end()) {
                const TYandexConfig::TSectionsMap cipherSection = it->second->GetAllChildren();
                for (auto&& i : cipherSection) {
                    NOpenssl::TCipherConfig config;
                    config.Init(i.second);
                    AssertCorrectConfig(!Ciphers.contains(i.first), "Cipher duplication: %s", i.first.data());
                    Ciphers.emplace(i.first, std::move(config));
                }
            }
        }

        {
            auto it = serverSections.find("ExternalAccessTokensManager");
            if (it != serverSections.end()) {
                ExternalAccessTokensManagerConfig.Reset(new TExternalAccessTokensManagerConfig);
                ExternalAccessTokensManagerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("MapsLinker");
            if (it != serverSections.end()) {
                MapsLinkerConfig = MakeHolder<TMapsLinkerConfig>();
                MapsLinkerConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("MapsRouter");
            if (it != serverSections.end()) {
                MapsRouterConfig.Reset(new TMapsRouterConfig);
                MapsRouterConfig->Init(it->second);
            }
        }
        {
            auto it = serverSections.find("PedestrianRouter");
            if (it != serverSections.end()) {
                PedestrianRouterConfig.Reset(new TMapsRouterConfig);
                PedestrianRouterConfig->Init(it->second);
            }
        }
    } catch (const std::exception& e) {
        auto error = FormatExc(e);
        AbortFromCorruptedConfig("BaseServerConfig::Init: cannot parse config: %s", error.c_str());
    }
}

TString TServerBaseConfig::ToString() const {
    TStringStream ss;
    ToString(ss);
    return ss.Str();
}

void TServerBaseConfig::ToString(IOutputStream& os) const {
    os << DaemonConfig.ToString("DaemonConfig") << Endl;
    os << "<Server>" << Endl;

    os << "ItsWatchPath: " << ItsWatchPath << Endl;

    os << HttpServerOptions.ToString("HttpServer");

    os << "<RequestPolicy>" << Endl;
    for (auto&& i : RequestHandlers) {
        os << "<" << i.first << ">" << Endl;
        i.second.ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</RequestPolicy>" << Endl;

    os << "<Notifications>" << Endl;
    for (auto&& i : Notifications) {
        os << "<" << i.first << ">" << Endl;
        i.second->ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</Notifications>" << Endl;

    os << "<RequestHandlers>" << Endl;
    for (auto&& i : RequestHandlers) {
        os << "<" << i.first << ">" << Endl;
        i.second.ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</RequestHandlers>" << Endl;

    if (!!Localization) {
        os << "<Localization>" << Endl;
        Localization->ToString(os);
        os << "</Localization>" << Endl;
    }

    if (!!Settings) {
        os << "<Settings>" << Endl;
        Settings->ToString(os);
        os << "</Settings>" << Endl;
    }

    if (DatasyncConfig) {
        os << "<Datasync>" << Endl;
        DatasyncConfig->ToString(os);
        os << "</Datasync>" << Endl;
    }

    os << "<Databases>" << Endl;
    for (auto&& i : DatabaseConfigs) {
        os << "<" << i.first << ">" << Endl;
        i.second.ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</Databases>" << Endl;

    os << "<ExternalDatabases>" << Endl;
    for (auto&& db : Databases) {
        os << "<" << db.first << ">" << Endl;
        db.second.ToString(os);
        os << "</" << db.first << ">" << Endl;
    }
    os << "</ExternalDatabases>" << Endl;

    os << "<Processors>" << Endl;
    os << "ProcessorDefaultTimeout : " << ProcessorDefaultTimeout << Endl;
    for (auto&& i : ProcessorsInfo) {
        os << "<" << i.first.ToString() << ">" << Endl;
        os << "ProcessorType: " << i.second->GetSpecialType() << Endl;
        i.second->ToString(os);
        os << "</" << i.first.ToString() << ">" << Endl;
    }
    os << "</Processors>" << Endl;

    os << "<AuthModules>" << Endl;
    for (auto&& i : AuthModules) {
        os << "<" << i.first << ">" << Endl;
        os << "Type: " << i.second->GetName() << Endl;
        i.second->ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</AuthModules>" << Endl;

    os << "<RTLineAPIs>" << Endl;
    for (auto&& i : RTLineAPIConfigs) {
        os << "<" << i.first << ">" << Endl;
        i.second.ToString(os);
        os << "</" << i.first << ">" << Endl;
    }
    os << "</RTLineAPIs>" << Endl;

    if (ADGlobalConfig) {
        os << "<GlobalAsyncDelivery>" << Endl;
        ADGlobalConfig->ToString(os);
        os << "</GlobalAsyncDelivery>" << Endl;
    }

    if (!!TaskExecutorConfig) {
        os << "<GlobalTaskExecutor>" << Endl;
        TaskExecutorConfig->ToString(os);
        os << "</GlobalTaskExecutor>" << Endl;
    }

    if (RTBackgroundManagerConfig) {
        os << "<RTBackgroundManager>" << Endl;
        RTBackgroundManagerConfig->ToString(os);
        os << "</RTBackgroundManager>" << Endl;
    }

    if (NotifiersManagerConfig) {
        os << "<NotifiersManager>" << Endl;
        NotifiersManagerConfig->ToString(os);
        os << "</NotifiersManager>" << Endl;
    }

    if (BackgroundProcessesConfig) {
        os << "<BackgroundProcesses>" << Endl;
        BackgroundProcessesConfig->ToString(os);
        os << "</BackgroundProcesses>" << Endl;
    }

    os << "<Ciphers>" << Endl;
    for (auto&& cipherIt : Ciphers) {
        os << "<" << cipherIt.first << ">" << Endl;
        cipherIt.second.ToString(os);
        os << "</" << cipherIt.first << ">" << Endl;
    }
    os << "</Ciphers>" << Endl;

    if (MapsLinkerConfig) {
        os << "<MapsLinker>" << Endl;
        MapsLinkerConfig->ToString(os);
        os << "</MapsLinker>" << Endl;
    }
    if (MapsRouterConfig) {
        os << "<MapsRouter>" << Endl;
        MapsRouterConfig->ToString(os);
        os << "</MapsRouter>" << Endl;
    }
    if (PedestrianRouterConfig) {
        os << "<PedestrianRouter>" << Endl;
        PedestrianRouterConfig->ToString(os);
        os << "</PedestrianRouter>" << Endl;
    }

    os << "EventLog: " << EventLog << Endl;
    os << "UnifiedAgentEventLog: " << UnifiedAgentEventLog << Endl;
    os << "UnifiedAgentGrpcMaxMessageSize: " << UnifiedAgentGrpcMaxMessageSize << Endl;
    os << "UnifiedAgentGrpcReconnectDelay: " << UnifiedAgentGrpcReconnectDelay << Endl;
    os << "UnifiedAgentGrpcSendDelay: " << UnifiedAgentGrpcSendDelay << Endl;
    os << "UnifiedAgentMaxInflightBytes: " << UnifiedAgentMaxInflightBytes << Endl;
    DoToString(os);
    os << "</Server>" << Endl;
}

TServerBaseConfig::TServerBaseConfig(const TServerConfigConstructorParams& params)
    : DaemonConfig(*params.GetDaemonConfig())
    , ModulesConfig(*this, "ModulesConfig")
    , UnifiedAgentGrpcMaxMessageSize()
    , UnifiedAgentMaxInflightBytes()
{
    auto& config = RawConfig;
    CHECK_WITH_LOG(config.ParseMemory(params.GetText().data()));
    const TYandexConfig::Section* root = config.GetRootSection();
    const TYandexConfig::TSectionsMap rootSections = root->GetAllChildren();
    auto itServer = rootSections.find("Server");
    AssertCorrectConfig(itServer != rootSections.end(), "No 'Server' section in config");
    Init(itServer->second);
}

const TMap<TString, NStorage::TDatabaseConfig>& TServerBaseConfig::GetExternalDatabases() const {
    return Databases;
}

const TMapsLinkerConfig* TServerBaseConfig::GetMapsLinkerConfig() const {
    return MapsLinkerConfig.Get();
}

const TMapsRouterConfig* TServerBaseConfig::GetMapsRouterConfig() const {
    return MapsRouterConfig.Get();
}

const TMapsRouterConfig* TServerBaseConfig::GetPedestrianRouterConfig() const {
    return PedestrianRouterConfig.Get();
}

IRequestProcessorConfig::TPtr TServerBaseConfig::GetProcessorInfo(const TVersionedKey& processorKey) const {
    if (ProcessorsInfo.empty()) {
        return nullptr;
    }

    auto it = LowerKey(ProcessorsInfo, processorKey);
    if (it != ProcessorsInfo.end()) {
        if (it->first == processorKey) {
            return it->second;
        }
        if (it->first.Match(processorKey)) {
            return it->second;
        }
    } else {
        auto itr = ProcessorsInfo.rbegin();
        if (itr->first.Match(processorKey)) {
            return it->second;
        }
    }

    DEBUG_LOG << "GetProcessorInfo: " << processorKey.ToString() << ": NOT found" << Endl;
    return nullptr;
}
